<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PatrollerSchedule extends Model
{
    use HasFactory;

    protected $guarded = ['id'];

    protected $casts = [
        'scheduled_date' => 'date',
        'from_time' => 'datetime',
        'to_time' => 'datetime',
        'route_rate' => 'decimal:2',
        'gas' => 'decimal:2',
        'allowance' => 'decimal:2',
        'other_expense' => 'decimal:2',
        'job_started_at' => 'datetime',
        'job_ended_at' => 'datetime',
        'from_datetime' => 'datetime',
        'to_datetime' => 'datetime',
    ];

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function route()
    {
        return $this->belongsTo(Route::class);
    }

    public function employees()
    {
        return $this->belongsToMany(Employee::class, 'employee_patroller_schedule')
            ->withPivot('wage_types', 'wage_rate', 'gas_rate', 'allowance', 'other_expense', 'actual_start_at', 'actual_end_at', 'cancellation_status', 'cancellation_requested_at', 'cancellation_reason', 'is_addon', 'start_evidence_path', 'end_evidence_path')
            ->withTimestamps();
    }

    public function checkins()
    {
        return $this->hasMany(GuardCheckin::class);
    }

    // Scope for filtering by status
    public function scopeStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    // Scope for today's schedules
    public function scopeToday($query)
    {
        return $query->whereDate('scheduled_date', today());
    }

    // Scope for upcoming schedules
    public function scopeUpcoming($query)
    {
        return $query->where('scheduled_date', '>=', today())
            ->where('status', '!=', 'completed');
    }

    // Auto-generate duty number
    public static function generateDutyNumber($companyId)
    {
        $idSetting = \App\Models\IdSetting::where('company_id', $companyId)->first();

        if (!$idSetting) {
            throw new \Exception('ID settings not found for this company');
        }

        $prefix = $idSetting->patroller_prefix ?? 'P';
        $nextNumber = $idSetting->patroller_next_number ?? 1;

        $dutyNumber = $prefix . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);

        // Increment the next number
        $idSetting->increment('patroller_next_number');

        return $dutyNumber;
    }

    /**
     * Get the job sites for this schedule.
     */
    public function jobSites()
    {
        return $this->hasMany(PatrollerJobSite::class);
    }

    /**
     * Virtual attributes for consistency with Guard Schedule model in Add-on views
     */
    public function getFromDatetimeAttribute()
    {
        return $this->from_time;
    }

    public function getToDatetimeAttribute()
    {
        return $this->to_time;
    }

    /**
     * Virtual site relation for Patroller jobs (uses Route name)
     */
    public function getVirtualSiteAttribute()
    {
        return (object) [
            'name' => $this->route->name ?? 'N/A',
            'site_id' => 'RT-#' . ($this->route_id ?? '0')
        ];
    }

    public function issueTickets()
    {
        return $this->hasMany(PatrollerIssueTicket::class);
    }

    /**
     * Check if the job has been started.
     */
    public function isJobStarted(): bool
    {
        return $this->job_status === 'in_progress' || $this->job_status === 'completed' || $this->job_status === 'auto_ended';
    }

    /**
     * Check if the job has been completed.
     */
    public function isJobCompleted(): bool
    {
        return $this->job_status === 'completed' || $this->job_status === 'auto_ended';
    }

    /**
     * Check if the job is in progress.
     */
    public function isJobInProgress(): bool
    {
        return $this->job_status === 'in_progress';
    }

    /**
     * Check if the job can be started.
     */
    public function canStartJob(): bool
    {
        return $this->job_status === 'pending' && $this->from_time->isPast();
    }

    /**
     * Scopes for Job States
     */
    public function scopeActive($query)
    {
        return $query->where('patroller_schedules.job_status', 'in_progress');
    }

    public function scopeTodayUpcoming($query)
    {
        $now = now();
        return $query->whereDate('patroller_schedules.from_time', today())
            ->where('patroller_schedules.to_time', '>', $now)
            ->whereNotIn('patroller_schedules.job_status', ['in_progress', 'completed', 'auto_ended', 'cancelled']);
    }

    public function scopeUpcomingFuture($query)
    {
        return $query->whereDate('patroller_schedules.from_time', '>', today())
            ->where('patroller_schedules.status', '!=', 'pending')
            ->whereNotIn('patroller_schedules.job_status', ['completed', 'auto_ended', 'cancelled']);
    }

    public function scopeMissed($query)
    {
        $now = now();
        return $query->where(function ($q) use ($now) {
            $q->whereDate('patroller_schedules.from_time', '<', today())
                ->orWhere('patroller_schedules.to_time', '<', $now);
        })
            ->whereNotIn('patroller_schedules.job_status', ['in_progress', 'completed', 'auto_ended', 'cancelled'])
            ->whereDoesntHave('jobSites', function ($q) {
                $q->where('status', '!=', 'pending');
            });
    }

    public function scopeCompleted($query)
    {
        $now = now();
        return $query->where(function ($q) use ($now) {
            $q->whereIn('patroller_schedules.job_status', ['completed', 'auto_ended'])
                ->orWhere(function ($sub) use ($now) {
                    $sub->where('patroller_schedules.job_status', 'pending')
                        ->where('patroller_schedules.from_time', '<', $now)
                        ->whereHas('jobSites', function ($siteQ) {
                            $siteQ->where('status', '!=', 'pending');
                        });
                });
        });
    }

    public function scopePendingApproval($query)
    {
        return $query->where('patroller_schedules.status', 'pending')
            ->whereDate('patroller_schedules.from_time', '>', today());
    }

    public function scopeCancelled($query)
    {
        return $query->where('patroller_schedules.job_status', 'cancelled');
    }

    /**
     * Check if the job is overdue.
     */
    public function isOverdue(): bool
    {
        return $this->to_time->isPast() && $this->job_status === 'in_progress';
    }
    /**
     * Get the calculated status for display (matching guard job logic)
     */
    public function getCalculatedStatusAttribute()
    {
        $now = now();
        if ($this->job_status === 'cancelled') {
            return 'cancelled';
        }
        if ($this->job_status === 'in_progress')
            return 'active';
        if (in_array($this->job_status, ['completed', 'auto_ended']))
            return 'completed';

        if ($this->from_time->isToday() && $this->to_time->isFuture() && $this->job_status === 'pending') {
            return 'today';
        }

        if ($this->from_time->isPast() && $this->job_status === 'pending') {
            if ($this->jobSites()->where('status', '!=', 'pending')->exists()) {
                return 'completed';
            }
            return 'missed';
        }

        if ($this->job_status === 'pending' && $this->from_time->isFuture()) {
            return 'upcoming';
        }

        return $this->job_status;
    }

    /**
     * Get the status color based on calculated status
     */
    public function getStatusColorAttribute()
    {
        $status = $this->calculated_status;
        return match ($status) {
            'active' => '#2563EB',
            'completed' => '#16A34A',
            'missed' => '#DC2626',
            'today' => '#D97706',
            'upcoming' => '#D97706',
            'cancelled' => '#DC2626',
            default => '#9CA3AF',
        };
    }

    /**
     * Get the CSS classes for status badge in web view
     */
    public function getStatusClassesAttribute()
    {
        $status = $this->calculated_status;
        return match ($status) {
            'active' => 'bg-emerald-50 text-emerald-700 border border-emerald-100 animate-pulse',
            'completed' => 'bg-emerald-50 text-emerald-700 border border-emerald-100',
            'missed' => 'bg-red-50 text-red-700 border border-red-100',
            'today', 'upcoming' => 'bg-blue-50 text-blue-700 border border-blue-100',
            'cancelled' => 'bg-red-50 text-red-700 border border-red-100',
            default => 'bg-gray-100 text-gray-600 border border-gray-200',
        };
    }

    public function getStatusDotClassesAttribute()
    {
        $status = $this->calculated_status;
        return match ($status) {
            'active', 'completed' => 'bg-emerald-500',
            'missed', 'cancelled' => 'bg-red-500',
            'today', 'upcoming' => 'bg-blue-500',
            default => 'bg-gray-400',
        };
    }
}
