<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PatrollerJobSite extends Model
{
    use HasFactory;

    protected $fillable = [
        'patroller_schedule_id',
        'site_id',
        'employee_id',
        'checked_in_at',
        'checkin_latitude',
        'checkin_longitude',
        'checkin_images',
        'checked_out_at',
        'checkout_latitude',
        'checkout_longitude',
        'status',
    ];

    protected $casts = [
        'checkin_images' => 'array',
        'checked_in_at' => 'datetime',
        'checked_out_at' => 'datetime',
        'checkin_latitude' => 'decimal:8',
        'checkin_longitude' => 'decimal:8',
        'checkout_latitude' => 'decimal:8',
        'checkout_longitude' => 'decimal:8',
    ];

    /**
     * Get the patroller schedule that owns the job site.
     */
    public function patrollerSchedule()
    {
        return $this->belongsTo(PatrollerSchedule::class);
    }

    /**
     * Get the site for this job.
     */
    public function site()
    {
        return $this->belongsTo(Site::class);
    }

    /**
     * Get the employee assigned to this job site.
     */
    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    /**
     * Get the issue tickets for this job site.
     */
    public function issueTickets()
    {
        return $this->hasMany(PatrollerIssueTicket::class);
    }

    /**
     * Check if the site has been checked in.
     */
    public function isCheckedIn(): bool
    {
        return $this->status === 'checked_in' || $this->status === 'checked_out';
    }

    /**
     * Check if the site has been checked out.
     */
    public function isCheckedOut(): bool
    {
        return $this->status === 'checked_out';
    }

    /**
     * Get the duration of the site visit in minutes.
     */
    public function getVisitDuration(): ?int
    {
        if (!$this->checked_in_at || !$this->checked_out_at) {
            return null;
        }

        return $this->checked_in_at->diffInMinutes($this->checked_out_at);
    }
}
