<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PatrollerIssueTicket extends Model
{
    use HasFactory;

    protected $fillable = [
        'patroller_job_site_id',
        'patroller_schedule_id',
        'site_id',
        'employee_id',
        'ticket_number',
        'description',
        'status',
        'images',
        'latitude',
        'longitude',
    ];

    protected $casts = [
        'images' => 'array',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
    ];

    /**
     * Get the job site that owns the ticket.
     */
    public function jobSite()
    {
        return $this->belongsTo(PatrollerJobSite::class, 'patroller_job_site_id');
    }

    /**
     * Get the patroller schedule for this ticket.
     */
    public function patrollerSchedule()
    {
        return $this->belongsTo(PatrollerSchedule::class);
    }

    /**
     * Get the site for this ticket.
     */
    public function site()
    {
        return $this->belongsTo(Site::class);
    }

    /**
     * Get the employee who created the ticket.
     */
    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    /**
     * Generate a unique ticket number.
     */
    public static function generateTicketNumber(): string
    {
        $date = now()->format('Ymd');
        $prefix = "PT-{$date}-";

        // Get the last ticket number for today
        $lastTicket = self::where('ticket_number', 'like', "{$prefix}%")
            ->orderBy('ticket_number', 'desc')
            ->first();

        if ($lastTicket) {
            // Extract the sequence number and increment
            $lastSequence = (int) substr($lastTicket->ticket_number, -4);
            $newSequence = $lastSequence + 1;
        } else {
            $newSequence = 1;
        }

        return $prefix . str_pad($newSequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Check if the ticket is marked as bad.
     */
    public function isBad(): bool
    {
        return $this->status === 'bad';
    }

    /**
     * Check if the ticket is marked as good.
     */
    public function isGood(): bool
    {
        return $this->status === 'good';
    }

    /**
     * Scope a query to only include good tickets.
     */
    public function scopeGood($query)
    {
        return $query->where('status', 'good');
    }

    /**
     * Scope a query to only include bad tickets.
     */
    public function scopeBad($query)
    {
        return $query->where('status', 'bad');
    }

    /**
     * Scope a query to filter by site.
     */
    public function scopeForSite($query, $siteId)
    {
        return $query->where('site_id', $siteId);
    }
}
