<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Spatie\Permission\Traits\HasRoles;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class Employee extends Authenticatable
{
    use HasFactory, Notifiable, HasRoles, HasApiTokens, \App\Traits\LogsActivity;

    protected $guarded = ['id'];

    protected $casts = [
        'dob' => 'date',
        'is_active' => 'boolean',
        'social_links' => 'array',
        'unavailable_days' => 'array',
        'metadata' => 'array',
    ];

    public function issuedUniforms()
    {
        return $this->hasMany(EmployeeUniform::class);
    }

    public function fcmTokens()
    {
        return $this->hasMany(FcmToken::class);
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function experiences()
    {
        return $this->hasMany(EmployeeExperience::class);
    }

    public function identifications()
    {
        return $this->hasMany(EmployeeIdentification::class);
    }

    public function wageTypes()
    {
        return $this->belongsToMany(WageType::class, 'employee_wage_type')
            ->withPivot('rate');
    }

    public function zone()
    {
        return $this->belongsTo(Zone::class);
    }
    public function jobRole()
    {
        return $this->belongsTo(JobRole::class);
    }
    public function department()
    {
        return $this->belongsTo(Department::class);
    }

    public function permanentCountry()
    {
        return $this->belongsTo(Country::class, 'permanent_country_id');
    }

    public function permanentState()
    {
        return $this->belongsTo(State::class, 'permanent_state_id');
    }

    public function correspondingCountry()
    {
        return $this->belongsTo(Country::class, 'corresponding_country_id');
    }

    public function correspondingState()
    {
        return $this->belongsTo(State::class, 'corresponding_state_id');
    }

    public function licenceIssuingProvinceState()
    {
        return $this->belongsTo(State::class, 'licence_issuing_province_id');
    }

    public function skills()
    {
        return $this->belongsToMany(Skill::class);
    }

    public function schedules()
    {
        return $this->belongsToMany(Schedule::class, 'employee_schedule')
            ->withPivot('wage_rate', 'gas_rate', 'wage_types', 'actual_start_at', 'actual_end_at', 'start_lat', 'start_lng', 'end_lat', 'end_lng', 'checkin_images', 'checkout_images', 'cancellation_status')
            ->withTimestamps();
    }

    public function patrollerSchedules()
    {
        return $this->belongsToMany(PatrollerSchedule::class, 'employee_patroller_schedule')
            ->withPivot('wage_rate', 'gas_rate', 'allowance', 'other_expense', 'wage_types', 'actual_start_at', 'actual_end_at', 'cancellation_status', 'cancellation_requested_at', 'cancellation_reason')
            ->withTimestamps();
    }

    public function incidents()
    {
        return $this->hasMany(Incident::class);
    }

    public function checkpointScans()
    {
        return $this->hasMany(CheckpointScan::class);
    }

    public function bannedSites()
    {
        return $this->belongsToMany(Site::class, 'employee_site_bans')
            ->withPivot('reason', 'banned_at')
            ->withTimestamps();
    }

    public function getVisibilityMinutes()
    {
        $setting = CompanyJobRoleVisibility::where('company_id', $this->company_id)
            ->where('job_role_id', $this->job_role_id)
            ->first();

        return $setting ? $setting->visibility_minutes : 60;
    }

    public function getProfilePictureUrlAttribute()
    {
        if ($this->profile_picture) {
            return asset('storage/' . $this->profile_picture);
        }

        $gender = $this->gender ? strtolower($this->gender) : 'male';
        $key = 'default_employee_' . $gender . '_image';
        $path = \App\Models\Setting::where('key', $key)->value('value');

        return $path ? asset('storage/' . $path) : asset('images/default-avatar.png');
    }

    public function getFullNameAttribute()
    {
        return trim("{$this->first_name} {$this->middle_name} {$this->last_name}");
    }

    public function isProfileComplete()
    {
        $mandatoryFields = [
            'first_name',
            'email',
            'phone_number',
            'gender',
            'job_role_id',
            'license_number',
            'license_expiry',
            'file_license',
            'file_contract',
        ];

        foreach ($mandatoryFields as $field) {
            if (empty($this->$field)) {
                return false;
            }
        }

        return true;
    }
}
