<?php

namespace App\Jobs;

use App\Models\EmailCampaign;
use App\Models\Employee;
use App\Models\Lead;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class SendEmailCampaignJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $campaign;

    public function __construct(EmailCampaign $campaign)
    {
        $this->campaign = $campaign;
    }

    public function handle()
    {
        $emails = [];
        $companyId = $this->campaign->company_id;

        if ($this->campaign->recipient_type === 'employees') {
            $emails = Employee::where('company_id', $companyId)->whereNotNull('email')->pluck('email')->toArray();
        } elseif ($this->campaign->recipient_type === 'clients') {
            $emails = Lead::where('company_id', $companyId)->whereNotNull('email')->pluck('email')->toArray();
        } elseif ($this->campaign->recipient_type === 'custom') {
            $path = $this->campaign->custom_file_path;
            if ($path && Storage::disk('local')->exists($path)) {
                try {
                    // Using Maatwebsite Excel to handle .xlsx, .xls and .csv
                    $data = \Maatwebsite\Excel\Facades\Excel::toArray(new class {}, $path, 'local');

                    foreach ($data as $sheet) {
                        foreach ($sheet as $row) {
                            foreach ($row as $cell) {
                                if ($cell && filter_var(trim($cell), FILTER_VALIDATE_EMAIL)) {
                                    $emails[] = trim($cell);
                                }
                            }
                        }
                    }
                } catch (\Exception $e) {
                    \Illuminate\Support\Facades\Log::error("Excel Import Error for campaign {$this->campaign->name}: " . $e->getMessage());

                    // Fallback to simple CSV if Excel fails
                    $fullPath = Storage::disk('local')->path($path);
                    if (($handle = fopen($fullPath, "r")) !== FALSE) {
                        while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
                            foreach ($data as $cell) {
                                if (filter_var(trim($cell), FILTER_VALIDATE_EMAIL)) {
                                    $emails[] = trim($cell);
                                }
                            }
                        }
                        fclose($handle);
                    }
                }
            }
        } elseif ($this->campaign->recipient_type === 'particular' && $this->campaign->specific_email) {
            $emails = [$this->campaign->specific_email];
        }

        $emails = array_unique(array_filter($emails));
        $emailCount = count($emails);
        \Illuminate\Support\Facades\Log::info("Campaign: {$this->campaign->name} - Found {$emailCount} recipients.");

        $template = $this->campaign->template;

        if (!$template || empty($emails)) {
            \Illuminate\Support\Facades\Log::warning("Campaign: {$this->campaign->name} - No template or no recipients found. Aborting.");
            return;
        }

        foreach ($emails as $email) {
            // In production, we should queue individual emails to avoid timeout/blocking
            // For now, simple loop in job
            try {
                $response = Mail::mailer('mailgun')->send([], [], function ($message) use ($email, $template) {
                    $message->to($email)
                        ->subject($template->subject)
                        ->html($template->content);
                });

                // Get message ID for tracking (Laravel 9+ way)
                $messageId = null;
                if ($response instanceof \Illuminate\Mail\SentMessage) {
                    $messageId = $response->getMessageId();
                }

                \App\Models\EmailCampaignLog::create([
                    'email_campaign_id' => $this->campaign->id,
                    'email' => $email,
                    'message_id' => $messageId,
                    'status' => 'sent',
                ]);

                \Illuminate\Support\Facades\Log::info("Email sent successfully to: {$email} for campaign: {$this->campaign->name}");
            } catch (\Exception $e) {
            } catch (\Exception $e) {
                // ... (Log creation code) ...
                \App\Models\EmailCampaignLog::create([
                    'email_campaign_id' => $this->campaign->id,
                    'email' => $email,
                    'status' => 'failed',
                    'error' => $e->getMessage(),
                ]);

                \Illuminate\Support\Facades\Log::error("Failed to send email to: {$email} in campaign: {$this->campaign->name}. Error: " . $e->getMessage());
                \Illuminate\Support\Facades\Log::error("Mailgun Config Dump: Domain=" . config('services.mailgun.domain') . ", Endpoint=" . config('services.mailgun.endpoint'));
            }
        }
    }
}
