<?php

namespace App\Http\Controllers;

use App\Models\Schedule;
use App\Models\Site;
use App\Models\Employee;
use App\Models\IdSetting;
use App\Models\TimeZone;
use App\Models\WageType;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ScheduleController extends Controller
{
    public function index(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = Schedule::with(['site', 'employees'])
            ->where('schedules.company_id', $companyId);

        // Apply Filters
        if ($request->filled('site_id')) {
            $query->where('schedules.site_id', $request->site_id);
        }

        if ($request->filled('employee_id')) {
            $query->whereHas('employees', function ($q) use ($request) {
                $q->where('employees.id', $request->employee_id);
            });
        }

        if ($request->filled('is_addon') && $request->is_addon == '1') {
            $query->whereHas('employees', function ($q) {
                $q->where('employee_schedule.is_addon', true);
            });
        }

        if ($request->filled('from_date')) {
            $query->whereDate('schedules.from_datetime', '>=', $request->from_date);
        }
        if ($request->filled('to_date')) {
            $query->whereDate('schedules.from_datetime', '<=', $request->to_date);
        }

        $type = $request->query('type', 'all');
        $now = Carbon::now();
        $today = Carbon::today();

        // Refresh Settings
        $refreshEnabled = \App\Models\Setting::where('key', '=', 'dashboard_refresh_enabled')->value('value');
        $refreshInterval = \App\Models\Setting::where('key', '=', 'dashboard_refresh_interval')->value('value') ?? 5;

        // Late Tolerance
        $lateTolerance = (int) \App\Models\Setting::where('key', '=', 'late_checkin_tolerance')->value('value') ?: 10;

        switch ($type) {
            case 'active':
                $query->where('schedules.status', '=', 'active');
                break;
            case 'today':
                $query->whereDate('schedules.from_datetime', $today)
                    ->where('schedules.to_datetime', '>', $now)
                    ->whereNotIn('schedules.status', ['active', 'completed', 'cancelled']);
                break;
            case 'upcoming':
                $query->whereDate('schedules.from_datetime', '>', $today)
                    ->whereNotIn('schedules.status', ['active', 'completed', 'cancelled']);
                break;
            case 'completed':
                $query->where('schedules.status', 'completed');
                break;
            case 'cancelled':
                $query->where('schedules.status', 'cancelled');
                break;
            case 'missed':
                $query->where(function ($q) use ($now) {
                    $q->where('schedules.status', 'missed')
                        ->orWhere(function ($sq) use ($now) {
                            $sq->where('schedules.to_datetime', '<=', $now)
                                ->whereNotIn('schedules.status', ['active', 'completed', 'cancelled']);
                        });
                });
                break;
            case 'this_week':
                $query->whereBetween('schedules.from_datetime', [$now->startOfWeek(), $now->endOfWeek()]);
                break;
            case 'this_month':
                $query->whereMonth('schedules.from_datetime', $now->month)
                    ->whereYear('schedules.from_datetime', $now->year);
                break;
        }

        // Sorting
        $sort = $request->query('sort');
        $direction = $request->query('direction', 'asc');

        if ($sort) {
            $query->select('schedules.*'); // Ensure we select schedules columns if we join
            switch ($sort) {
                case 'site':
                    $query->join('sites', 'schedules.site_id', '=', 'sites.id')
                        ->orderBy('sites.name', $direction);
                    break;
                case 'employee':
                    // Sort by the first assigned employee's name
                    $query->orderBy(
                        \App\Models\Employee::select('first_name')
                            ->join('employee_schedule', 'employees.id', '=', 'employee_schedule.employee_id')
                            ->whereColumn('employee_schedule.schedule_id', 'schedules.id')
                            ->limit(1),
                        $direction
                    );
                    break;
                case 'data_time': // user said "Date & Time" so I'll use this key or just 'datetime'
                    $query->orderBy('from_datetime', $direction);
                    break;
                case 'status':
                    $query->orderBy('status', $direction);
                    break;
                default:
                    $query->latest();
                    break;
            }
        } else {
            $query->latest();
        }

        $schedules = $query->paginate(100);

        // Counts for tabs (filtering affects counts too ideally, but usually tabs are global categories. keeping global for now as per common pattern, or should they be filtered? Usually tabs act as primary filter. Let's keep counts global for the 'type' tabs, but maybe we should filter counts if a filter is active? For now, keep counts simple based on type only to avoid confusion)
        $baseQuery = Schedule::where('company_id', $companyId);

        // If specific constraints are needed on counts based on other filters, we can add them to baseQuery.
        // For now, let's keep tabs showing "All" available for that user context, and filters refine the list.

        $counts = [
            'all' => $baseQuery->count(),
            'active' => (clone $baseQuery)->where('status', '=', 'active')->count(),
            'today' => (clone $baseQuery)->whereDate('from_datetime', $today)->where('to_datetime', '>', $now)->whereNotIn('status', ['active', 'completed', 'cancelled'])->count(),
            'this_week' => (clone $baseQuery)->whereBetween('from_datetime', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()])->count(),
            'this_month' => (clone $baseQuery)->whereMonth('from_datetime', Carbon::now()->month)->whereYear('from_datetime', Carbon::now()->year)->count(),
            'upcoming' => (clone $baseQuery)->whereDate('from_datetime', '>', $today)->whereNotIn('status', ['active', 'completed', 'cancelled'])->count(),
            'completed' => (clone $baseQuery)->where('status', '=', 'completed')->count(),
            'missed' => (clone $baseQuery)->where(function ($q) use ($now) {
                $q->where('status', 'missed')
                    ->orWhere(function ($sq) use ($now) {
                        $sq->where('to_datetime', '<=', $now)
                            ->whereNotIn('status', ['active', 'completed', 'cancelled']);
                    });
            })->count(),
            'cancelled' => (clone $baseQuery)->where('status', '=', 'cancelled')->count(),
        ];

        // Data for Filters
        $sites = Site::where('company_id', $companyId)->orderBy('name')->get(['id', 'name']);
        $employees = Employee::where('company_id', $companyId)->orderBy('first_name')->get(['id', 'first_name', 'last_name']);

        if ($request->wantsJson()) {
            return response()->json([
                'schedules' => $schedules,
                'counts' => $counts
            ]);
        }

        return view('schedules.index', compact('schedules', 'type', 'counts', 'sites', 'employees', 'refreshEnabled', 'refreshInterval', 'lateTolerance'));
    }

    public function exportPdf(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = Schedule::with(['site', 'employees'])
            ->where('company_id', $companyId);

        // Apply Filters
        if ($request->filled('site_id')) {
            $query->where('site_id', $request->site_id);
        }

        if ($request->filled('employee_id')) {
            $query->whereHas('employees', function ($q) use ($request) {
                $q->where('employees.id', $request->employee_id);
            });
        }

        if ($request->filled('from_date')) {
            $query->whereDate('from_datetime', '>=', $request->from_date);
        }
        if ($request->filled('to_date')) {
            $query->whereDate('from_datetime', '<=', $request->to_date);
        }

        $type = $request->query('type', 'all');
        $today = Carbon::today();

        switch ($type) {
            case 'active':
                $query->where('status', 'active');
                break;
            case 'today':
                $query->whereDate('schedule_date', $today)
                    ->where('status', '!=', 'active')
                    ->where('status', '!=', 'completed');
                break;
            case 'upcoming':
                $query->whereDate('schedule_date', '>', $today);
                break;
            case 'completed':
                $query->where('status', 'completed');
                break;
            case 'missed':
                $query->whereDate('schedule_date', '<', $today)
                    ->where('status', '!=', 'active')
                    ->where('status', '!=', 'completed');
                break;
        }

        $schedules = $query->latest()->get(); // Get all for PDF

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('schedules.pdf', compact('schedules', 'type'));
        return $pdf->download('schedules_export_' . now()->format('Y-m-d_H-i') . '.pdf');
    }

    public function exportExcel(Request $request, \App\Services\ExcelExportService $excelService)
    {
        $companyId = session('selected_company_id');
        $query = Schedule::with(['site', 'employees'])
            ->where('company_id', $companyId);

        // Apply Filters (Same as exportPdf)
        if ($request->filled('site_id')) {
            $query->where('site_id', $request->site_id);
        }

        if ($request->filled('employee_id')) {
            $query->whereHas('employees', function ($q) use ($request) {
                $q->where('employees.id', $request->employee_id);
            });
        }

        if ($request->filled('from_date')) {
            $query->whereDate('from_datetime', '>=', $request->from_date);
        }
        if ($request->filled('to_date')) {
            $query->whereDate('from_datetime', '<=', $request->to_date);
        }

        $type = $request->query('type', 'all');
        $today = Carbon::today();

        switch ($type) {
            case 'active':
                $query->where('status', 'active');
                break;
            case 'today':
                $query->whereDate('schedule_date', $today)
                    ->where('status', '!=', 'active')
                    ->where('status', '!=', 'completed');
                break;
            case 'upcoming':
                $query->whereDate('schedule_date', '>', $today);
                break;
            case 'completed':
                $query->where('status', 'completed');
                break;
            case 'missed':
                $query->whereDate('schedule_date', '<', $today)
                    ->where('status', '!=', 'active')
                    ->where('status', '!=', 'completed');
                break;
        }

        $records = $query->latest()->get();

        $headers = ['Duty #', 'Site', 'Employee(s)', 'Date', 'From Time', 'To Time', 'Status'];
        $data = [];

        foreach ($records as $record) {
            $employees = $record->employees->map(fn($e) => "{$e->first_name} {$e->last_name} ({$e->employee_id})")->implode(', ');
            $data[] = [
                $record->duty_number,
                $record->site->name ?? 'N/A',
                $employees,
                $record->from_datetime->format('Y-m-d'),
                $record->from_datetime->format('H:i'),
                $record->to_datetime->format('H:i'),
                strtoupper($record->status)
            ];
        }

        return $excelService->generateReport(
            'Guard Schedules Report',
            $type,
            $headers,
            $data,
            [
                'filename_prefix' => 'Schedules_Export',
                'status_column_index' => 6,
                    'center_columns' => [0, 3, 4, 5, 6]
            ]
        );
    }

    public function create()
    {
        $companyId = session('selected_company_id');

        // Generate Duty Number
        $idSetting = IdSetting::where('company_id', $companyId)->first();
        $dutyNumber = 'SCH-0001';
        if ($idSetting) {
            $dutyNumber = ($idSetting->duty_prefix ?? 'DUTY-') . str_pad($idSetting->duty_next_number, 4, '0', STR_PAD_LEFT);
        }

        $sites = Site::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $employees = Employee::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('first_name')
            ->get();

        $timeZones = TimeZone::where('active', true)->get();
        $currentTz = config('app.timezone');

        $defaultDate = now()->format('Y-m-d');
        $defaultStart = now()->hour(8)->minute(0)->format('Y-m-d\TH:i');
        $defaultEnd = now()->hour(17)->minute(0)->format('Y-m-d\TH:i');

        $maxShiftHours = \App\Models\Setting::where('key', 'max_shift_hours')->value('value') ?? 12;

        return view('schedules.create', compact('dutyNumber', 'sites', 'employees', 'timeZones', 'currentTz', 'defaultDate', 'defaultStart', 'defaultEnd', 'maxShiftHours'));
    }

    public function store(Request $request)
    {
        $companyId = session('selected_company_id');

        $request->validate([
            'duty_number' => 'required|unique:schedules,duty_number',
            'schedule_date' => 'required|date',
            'site_id' => 'required|exists:sites,id',
            'site_rate' => 'required|numeric',
            'from_datetime' => 'required|date',
            'to_datetime' => 'required|date|after:from_datetime',
            'employees' => 'required|array|min:1',
            'employee_ids' => 'required|array|min:1',
            'repeat_until' => 'nullable|required_if:is_recurring,on|date|after:from_datetime',
            'exclude_days' => 'nullable|array',
            'exclude_days.*' => 'integer|between:0,6',
            'tour_route_ids' => 'nullable|array',
            'tour_route_ids.*' => 'exists:tour_routes,id',
        ]);

        // Max Shift Hours Validation
        if (auth()->user()->role === 'user') {
            $maxShiftHours = (float) (\App\Models\Setting::where('key', 'max_shift_hours')->value('value') ?? 12);
            $from = Carbon::parse($request->from_datetime);
            $to = Carbon::parse($request->to_datetime);
            $durationHours = $from->diffInMinutes($to) / 60;

            if ($durationHours > $maxShiftHours) {
                return redirect()->back()->withInput()->with('error', "Shift duration ({$durationHours} hrs) exceeds the maximum allowed shift hours ({$maxShiftHours} hrs) set by the administrator.");
            }
        }

        try {
            // Check for Banned Employees
            $bannedEmployees = DB::table('employee_site_bans')
                ->where('site_id', $request->site_id)
                ->whereIn('employee_id', $request->employee_ids)
                ->pluck('employee_id')
                ->toArray();

            if (!empty($bannedEmployees)) {
                $bannedNames = Employee::whereIn('id', $bannedEmployees)
                    ->get()
                    ->map(fn($e) => $e->first_name . ' ' . $e->last_name)
                    ->implode(', ');
                return redirect()->back()->withInput()->with('error', "Unable to schedule: The following employees are BANNED from this site: $bannedNames.");
            }

            DB::beginTransaction();
            $data = $request->except(['employees', 'employee_ids', '_token', '_method', 'is_recurring', 'repeat_until', 'exclude_days']);
            $data['company_id'] = $companyId;
            $data['active'] = true;

            // Sync Employees Helper Function
            $syncEmployees = function ($scheduleInstance) use ($request) {
                $employeeSyncData = [];
                $guardCount = count($request->employees);
                // The submitted site_rate is the TOTAL rate. We need to split it for per-employee bill_rate in pivot.
                $billRatePerGuard = ($guardCount > 0) ? ($request->site_rate / $guardCount) : $request->site_rate;

                foreach ($request->employees as $empId => $empData) {
                    $employeeSyncData[$empId] = [
                        'wage_rate' => $empData['wage_rate'] ?? 0,
                        'gas_rate' => $empData['gas_rate'] ?? null,
                        'bill_rate' => $billRatePerGuard, // Save per-guard portion of the rate
                        'wage_types' => json_encode(array_values($empData['wage_types'] ?? []))
                    ];
                }
                $scheduleInstance->employees()->sync($employeeSyncData);
            };

            $primaryFrom = Carbon::parse($request->from_datetime);
            $primaryTo = Carbon::parse($request->to_datetime);
            $durationSeconds = $primaryFrom->diffInSeconds($primaryTo);

            $endDate = ($request->has('is_recurring') && $request->filled('repeat_until'))
                ? Carbon::parse($request->repeat_until)->endOfDay()
                : $primaryFrom;

            $excludeDays = $request->input('exclude_days', []);
            $loopDate = $primaryFrom->copy();
            $createdCount = 0;
            $firstSchedule = null;
            $skippedDays = []; // Track skipped days for reporting

            $now = Carbon::now();
            $idSetting = IdSetting::firstOrCreate(['company_id' => $companyId]);

            // Fetch employees to check availability
            $schedulingEmployees = Employee::whereIn('id', $request->employee_ids)->get();

            while ($loopDate->lte($endDate)) {
                // Skip excluded days
                if (in_array($loopDate->dayOfWeek, $excludeDays)) {
                    $loopDate->addDay();
                    continue;
                }

                // Check for Employee Availability (Weekdays) - Skip instead of failing
                $currentDayName = $loopDate->format('l'); // Monday, Tuesday, etc.
                $hasUnavailableEmployee = false;
                foreach ($schedulingEmployees as $emp) {
                    $unavailableDays = $emp->unavailable_days ?? [];
                    if (in_array($currentDayName, $unavailableDays)) {
                        $hasUnavailableEmployee = true;
                        // Track which employee and day was skipped
                        $skippedDays[] = [
                            'date' => $loopDate->format('Y-m-d'),
                            'day' => $currentDayName,
                            'employee' => $emp->first_name . ' ' . $emp->last_name
                        ];
                        break; // No need to check other employees for this day
                    }
                }

                // Skip this day if any employee is unavailable
                if ($hasUnavailableEmployee) {
                    $loopDate->addDay();
                    continue;
                }

                $newData = $data;
                $newFrom = $loopDate->copy()->setTime($primaryFrom->hour, $primaryFrom->minute, $primaryFrom->second);
                $newTo = $newFrom->copy()->addSeconds($durationSeconds);

                if ($createdCount === 0) {
                    $newData['duty_number'] = $request->duty_number;
                } else {
                    $idSetting->refresh();
                    $newData['duty_number'] = ($idSetting->duty_prefix ?? 'DUTY-') . str_pad($idSetting->duty_next_number, 4, '0', STR_PAD_LEFT);
                }

                $newData['schedule_date'] = $loopDate->format('Y-m-d');
                $newData['from_datetime'] = $newFrom;
                $newData['to_datetime'] = $newTo;

                if ($newFrom->isSameDay($now)) {
                    $newData['status'] = 'today';
                } elseif ($newFrom->isFuture()) {
                    $newData['status'] = 'upcoming';
                } else {
                    $newData['status'] = 'missed';
                }

                $schedule = Schedule::create($newData);
                $syncEmployees($schedule);

                if (!empty($request->tour_route_ids)) {
                    $schedule->tourRoutes()->sync($request->tour_route_ids);
                }

                if ($createdCount === 0) {
                    $firstSchedule = $schedule;
                }

                $idSetting->increment('duty_next_number');
                $createdCount++;
                $loopDate->addDay();
            }

            if ($createdCount === 0) {
                DB::rollBack();
                return redirect()->back()->withInput()->with('error', "No schedules were created because all selected dates are in the exclude list or employees are unavailable on all days.");
            }

            DB::commit();

            // Build success message with skipped days info
            $successMessage = "Schedule(s) created successfully. {$createdCount} schedule(s) created.";

            if (count($skippedDays) > 0) {
                // Group skipped days by unique day names
                $skippedByDay = [];
                foreach ($skippedDays as $skip) {
                    $day = $skip['day'];
                    if (!isset($skippedByDay[$day])) {
                        $skippedByDay[$day] = [];
                    }
                    if (!in_array($skip['employee'], $skippedByDay[$day])) {
                        $skippedByDay[$day][] = $skip['employee'];
                    }
                }

                $skippedInfo = [];
                foreach ($skippedByDay as $day => $employees) {
                    $employeeList = implode(', ', array_unique($employees));
                    $skippedInfo[] = "{$day}s ({$employeeList} unavailable)";
                }

                $successMessage .= " Note: " . count($skippedDays) . " day(s) were automatically skipped: " . implode('; ', $skippedInfo) . ".";
            }

            return redirect()->route('schedules.index')->with('success', $successMessage);
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', "Failed to create schedule: " . $e->getMessage());
        }
    }

    public function show(Schedule $schedule)
    {
        return view('schedules.show', compact('schedule'));
    }

    public function getLocationLogs(\App\Models\Schedule $schedule)
    {
        $logs = $schedule->locationLogs()
            ->with('employee:id,first_name,last_name')
            ->orderBy('recorded_at')
            ->get()
            ->map(function ($log) {
                return [
                    'lat' => $log->latitude,
                    'lng' => $log->longitude,
                    'time' => $log->recorded_at->format('H:i'),
                    'name' => optional($log->employee)->first_name . ' ' . optional($log->employee)->last_name,
                ];
            });

        return response()->json($logs);
    }

    public function edit(Schedule $schedule)
    {
        $companyId = session('selected_company_id');
        $sites = Site::where('company_id', $companyId)->where('active', true)->get();
        $employees = Employee::where('company_id', $companyId)->where('active', true)->get();
        $timeZones = TimeZone::where('active', true)->get();
        $maxShiftHours = \App\Models\Setting::where('key', 'max_shift_hours')->value('value') ?? 12;

        return view('schedules.edit', compact('schedule', 'sites', 'employees', 'timeZones', 'maxShiftHours'));
    }

    public function update(Request $request, Schedule $schedule)
    {
        $request->validate([
            'schedule_date' => 'required|date',
            'site_id' => 'required|exists:sites,id',
            'site_rate' => 'required|numeric',
            'from_datetime' => 'required|date',
            'to_datetime' => 'required|date|after:from_datetime',
            'employees' => 'required|array|min:1',
            'employee_ids' => 'required|array|min:1',
        ]);

        // Max Shift Hours Validation
        if (auth()->user()->role === 'user') {
            $maxShiftHours = (float) (\App\Models\Setting::where('key', 'max_shift_hours')->value('value') ?? 12);
            $from = Carbon::parse($request->from_datetime);
            $to = Carbon::parse($request->to_datetime);
            $durationHours = $from->diffInMinutes($to) / 60;

            if ($durationHours > $maxShiftHours) {
                return redirect()->back()->withInput()->with('error', "Shift duration ({$durationHours} hrs) exceeds the maximum allowed shift hours ({$maxShiftHours} hrs) set by the administrator.");
            }
        }

        try {
            // Check for Banned Employees
            $bannedEmployees = DB::table('employee_site_bans')
                ->where('site_id', $request->site_id)
                ->whereIn('employee_id', $request->employee_ids)
                ->pluck('employee_id')
                ->toArray();

            if (!empty($bannedEmployees)) {
                $bannedNames = Employee::whereIn('id', $bannedEmployees)
                    ->get()
                    ->map(fn($e) => $e->first_name . ' ' . $e->last_name)
                    ->implode(', ');
                return redirect()->back()->withInput()->with('error', "Unable to update schedule: The following employees are BANNED from this site: $bannedNames.");
            }

            // Check for Employee Availability
            $checkDate = Carbon::parse($request->from_datetime);
            $checkDay = $checkDate->format('l');
            $schedulingEmployees = Employee::whereIn('id', $request->employee_ids)->get();

            foreach ($schedulingEmployees as $emp) {
                if (in_array($checkDay, $emp->unavailable_days ?? [])) {
                    return redirect()->back()->withInput()->with('error', "Unable to update: Employee {$emp->first_name} is unavailable on {$checkDay}s.");
                }
            }

            DB::beginTransaction();
            $data = $request->except(['employees', 'employee_ids', '_token', '_method']);
            $schedule->update($data);

            // Individual Sync for employees with their metadata
            $employeeSyncData = [];
            $guardCount = count($request->employees);
            // The submitted site_rate is the TOTAL rate. We need to split it for per-employee bill_rate in pivot.
            $billRatePerGuard = ($guardCount > 0) ? ($request->site_rate / $guardCount) : $request->site_rate;

            foreach ($request->employees as $empId => $empData) {
                $employeeSyncData[$empId] = [
                    'wage_rate' => $empData['wage_rate'] ?? 0,
                    'gas_rate' => $empData['gas_rate'] ?? null,
                    'bill_rate' => $billRatePerGuard, // Save per-guard portion of the rate
                    'wage_types' => json_encode(array_values($empData['wage_types'] ?? []))
                ];
            }
            $schedule->employees()->sync($employeeSyncData);

            // Reset status if it was missed but now end time is in future
            if ($schedule->status === 'missed' && $schedule->to_datetime > now()) {
                if ($schedule->from_datetime > now()) {
                    $schedule->update(['status' => 'upcoming']);
                } else {
                    $schedule->update(['status' => 'today']);
                }
            }

            DB::commit();
            return redirect()->route('schedules.index')->with('success', "Schedule for Duty #{$schedule->duty_number} updated successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withInput()->with('error', "Failed to update schedule: " . $e->getMessage());
        }
    }

    public function getEmployeeInfo(Request $request, $id)
    {
        try {
            $employee = Employee::with(['wageTypes', 'bannedSites'])->findOrFail($id);

            $start = $request->query('start');
            $end = $request->query('end');
            $conflicts = collect();

            if ($start && $end) {
                $conflicts = $employee->schedules()
                    ->where(function ($query) use ($start, $end) {
                        $query->whereBetween('from_datetime', [$start, $end])
                            ->orWhereBetween('to_datetime', [$start, $end])
                            ->orWhere(function ($q) use ($start, $end) {
                                $q->where('from_datetime', '<=', $start)
                                    ->where('to_datetime', '>=', $end);
                            });
                    })
                    ->with('site')
                    ->get();
            }

            // Count jobs (schedules) performed
            $jobsPerformed = $employee->schedules()
                ->where('status', 'completed')
                ->count();

            // Advanced Stat Holiday Check (Supports overnight split)
            $holidaySplit = [
                'has_holiday' => false,
                'regular_hours' => 0,
                'holiday_hours' => 0,
                'multiplier' => 1.0,
                'holiday_name' => null
            ];

            if ($start && $end) {
                $startTime = Carbon::parse($start);
                $endTime = Carbon::parse($end);
                $totalMinutes = $startTime->diffInMinutes($endTime);

                // Get all stat holidays for the company in this range
                $startDateStr = $startTime->toDateString();
                $endDateStr = $endTime->toDateString();

                $statHolidaysInRange = \App\Models\StatHoliday::where('company_id', $employee->company_id)
                    ->whereBetween('holiday_date', [$startDateStr, $endDateStr])
                    ->where('active', true)
                    ->get()
                    ->keyBy('holiday_date');

                if ($statHolidaysInRange->count() > 0) {
                    $holidaySplit['has_holiday'] = true;
                    // Analyze each hour segment
                    $tempTime = clone $startTime;
                    while ($tempTime->lessThan($endTime)) {
                        $currentDate = $tempTime->toDateString();
                        $nextTime = (clone $tempTime)->addHour();
                        if ($nextTime->greaterThan($endTime))
                            $nextTime = clone $endTime;

                        $minutes = $tempTime->diffInMinutes($nextTime);
                        $hours = $minutes / 60;

                        if (isset($statHolidaysInRange[$currentDate])) {
                            $holidaySplit['holiday_hours'] += $hours;
                            $holidaySplit['holiday_name'] = $statHolidaysInRange[$currentDate]->name;
                            $holidaySplit['multiplier'] = max($holidaySplit['multiplier'], $statHolidaysInRange[$currentDate]->multiplier);
                        } else {
                            $holidaySplit['regular_hours'] += $hours;
                        }
                        $tempTime = $nextTime;
                    }
                } else {
                    $holidaySplit['regular_hours'] = $totalMinutes / 60;
                }
            } else {
                // Fallback to schedule_date if start/end not provided
                $checkDate = $request->query('date') ?: ($start ? Carbon::parse($start)->toDateString() : null);
                if ($checkDate) {
                    $statHoliday = \App\Models\StatHoliday::where('company_id', $employee->company_id)
                        ->where('holiday_date', $checkDate)
                        ->where('active', true)
                        ->first();
                    if ($statHoliday) {
                        $holidaySplit['has_holiday'] = true;
                        $holidaySplit['multiplier'] = $statHoliday->multiplier;
                        $holidaySplit['holiday_name'] = $statHoliday->name;
                        // We don't know hours yet
                    }
                }
            }

            return response()->json([
                'id' => $employee->id,
                'employee_id' => $employee->employee_id,
                'name' => $employee->first_name . ' ' . $employee->last_name,
                'email' => $employee->email,
                'jobs_performed' => $jobsPerformed,
                'profile_picture' => $employee->profile_picture ? asset('storage/' . $employee->profile_picture) : asset('images/default-avatar.png'),
                'license_expiry' => $employee->license_expiry,
                'is_license_expired' => $employee->license_expiry ? Carbon::parse($employee->license_expiry)->isPast() : false,
                'unavailable_days' => $employee->unavailable_days ?? [],
                'holiday_split' => $holidaySplit,
                'conflicts' => $conflicts->map(function ($c) {
                    return [
                        'duty_number' => $c->duty_number,
                        'site_name' => $c->site ? $c->site->name : 'Unknown Site',
                        'from' => $c->from_datetime ? $c->from_datetime->format('M d, H:i') : 'N/A',
                        'to' => $c->to_datetime ? $c->to_datetime->format('M d, H:i') : 'N/A',
                    ];
                }),
                'wage_types' => $employee->wageTypes->map(function ($wt) use ($holidaySplit) {
                    return [
                        'id' => $wt->id,
                        'name' => $wt->name,
                        'base_rate' => $wt->pivot->rate,
                        'holiday_rate' => ($wt->pivot->rate * $holidaySplit['multiplier']), // Generic application
                        'rate' => $wt->pivot->rate // default
                    ];
                }),
                'banned_sites' => $employee->bannedSites->pluck('id')->toArray() // List of banned site IDs
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], 500);
        }
    }

    public function getSiteInfo($id)
    {
        $site = Site::findOrFail($id);

        // Total hours served by all employees
        $totalMinutes = Schedule::where('site_id', $id)
            ->where('status', 'completed')
            ->get()
            ->sum(function ($schedule) {
                return $schedule->from_datetime->diffInMinutes($schedule->to_datetime);
            });

        $totalHours = round($totalMinutes / 60, 2);

        return response()->json([
            'site_id' => $site->site_id,
            'name' => $site->name,
            'address' => $site->address_line_1 . ', ' . $site->city,
            'rate' => $site->rate,
            'total_hours' => $totalHours
        ]);
    }

    public function toggleStatus(Schedule $schedule)
    {
        $schedule->update(['active' => !$schedule->active]);

        if (request()->wantsJson()) {
            return response()->json(['success' => true, 'active' => (bool) $schedule->active]);
        }

        return back()->with('success', 'Schedule status updated.');
    }

    public function destroy(Schedule $schedule)
    {
        if ($schedule->company_id != session('selected_company_id')) {
            abort(403);
        }

        // Prevent deletion if inactive/completed or any employee has started
        // Check if any employee has started
        $hasStarted = $schedule->employees()->wherePivotNotNull('actual_start_at')->exists();

        if ($schedule->status === 'active' || $schedule->status === 'completed' || $hasStarted) {
            return back()->with('error', 'Cannot delete a schedule that has already started or is completed.');
        }

        $schedule->delete();

        return redirect()->route('schedules.index')->with('success', 'Schedule deleted successfully.');
    }

    public function downloadReport($id)
    {
        set_time_limit(120); // Increase to 2 minutes

        $schedule = Schedule::with(['site', 'employees', 'incidents'])->findOrFail($id);

        $isPdf = true;
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('schedules.reports.master', compact('schedule', 'isPdf'))
            ->setPaper('a4', 'portrait')
            ->setOption('isRemoteEnabled', true)
            ->setOption('isHtml5ParserEnabled', true);

        return $pdf->download("Duty_Report_{$schedule->duty_number}.pdf");
    }

    public function sendReportEmail($id)
    {
        set_time_limit(180);

        $schedule = Schedule::with(['site', 'company', 'employees', 'incidents'])->findOrFail($id);

        if (!$schedule->site->contact_email) {
            return back()->with('error', 'Site contact email is not set.');
        }

        try {
            $isPdf = true;
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('schedules.reports.master', compact('schedule', 'isPdf'))
                ->setPaper('a4', 'portrait')
                ->setOption('isRemoteEnabled', true)
                ->setOption('isHtml5ParserEnabled', true);

            $pdfOutput = $pdf->output();

            \Illuminate\Support\Facades\Mail::send('emails.schedule_report', ['schedule' => $schedule], function ($message) use ($schedule, $pdfOutput) {
                $message->to($schedule->site->contact_email)
                    ->subject("Duty Report: {$schedule->duty_number} - {$schedule->site->name}")
                    ->attachData($pdfOutput, "Duty_Report_{$schedule->duty_number}.pdf", [
                        'mime' => 'application/pdf',
                    ]);
            });

            return back()->with('success', "Report sent successfully to {$schedule->site->contact_email}");
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error("Failed to send schedule report email: " . $e->getMessage());
            return back()->with('error', 'Failed to send email: ' . $e->getMessage());
        }
    }

    public function viewReport($id)
    {
        $schedule = Schedule::with(['site', 'employees', 'incidents'])->findOrFail($id);

        $isPdf = false;
        return view('schedules.reports.master', compact('schedule', 'isPdf'));
    }

    public function downloadZip($id)
    {
        $schedule = Schedule::with(['site', 'employees', 'incidents'])->findOrFail($id);

        $dutyId = $schedule->duty_number ?? $schedule->id;
        $zipFileName = "Evidence_Duty_{$dutyId}.zip";
        $zipPath = storage_path("app/public/temp/{$zipFileName}");

        // Ensure temp directory exists
        if (!file_exists(dirname($zipPath))) {
            mkdir(dirname($zipPath), 0755, true);
        }

        $zip = new \ZipArchive;
        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === TRUE) {

            // We follow the requested structure:
            // images/
            //   checkin_images/
            //   checkout_images/
            //   report_incidents/
            //     incident1/
            //     incident2/

            $siteName = $schedule->site->name ?? 'Job';
            $cleanSiteName = str_replace([' ', '/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $siteName);
            $timestamp = $schedule->from_datetime->format('Y-m-d_H-i-s');
            $dutyFolder = "{$cleanSiteName}_{$timestamp}_{$dutyId}";

            foreach ($schedule->employees as $employee) {
                $pivot = $employee->pivot;
                $empId = $employee->id;

                // Checkin Images
                $checkinImages = json_decode($pivot->checkin_images ?? '[]');
                if (is_array($checkinImages)) {
                    foreach ($checkinImages as $index => $imagePath) {
                        $fullPath = storage_path('app/public/' . $imagePath);
                        if (file_exists($fullPath)) {
                            $extension = pathinfo($fullPath, PATHINFO_EXTENSION);
                            $zip->addFile($fullPath, "{$dutyFolder}/images/checkin_images/emp_{$empId}_img_{$index}.{$extension}");
                        }
                    }
                }

                // Checkout Images
                $checkoutImages = json_decode($pivot->checkout_images ?? '[]');
                if (is_array($checkoutImages)) {
                    foreach ($checkoutImages as $index => $imagePath) {
                        $fullPath = storage_path('app/public/' . $imagePath);
                        if (file_exists($fullPath)) {
                            $extension = pathinfo($fullPath, PATHINFO_EXTENSION);
                            $zip->addFile($fullPath, "{$dutyFolder}/images/checkout_images/emp_{$empId}_img_{$index}.{$extension}");
                        }
                    }
                }

                // Manual Evidence (Addon)
                if ($pivot->start_evidence_path) {
                    $fullPath = storage_path('app/public/' . $pivot->start_evidence_path);
                    if (file_exists($fullPath)) {
                        $extension = pathinfo($fullPath, PATHINFO_EXTENSION);
                        $zip->addFile($fullPath, "{$dutyFolder}/images/checkin_images/emp_{$empId}_manual.{$extension}");
                    }
                }
                if ($pivot->end_evidence_path) {
                    $fullPath = storage_path('app/public/' . $pivot->end_evidence_path);
                    if (file_exists($fullPath)) {
                        $extension = pathinfo($fullPath, PATHINFO_EXTENSION);
                        $zip->addFile($fullPath, "{$dutyFolder}/images/checkout_images/emp_{$empId}_manual.{$extension}");
                    }
                }
            }

            // Incident Images
            foreach ($schedule->incidents as $incident) {
                $incidentImages = $incident->images; // Casted as array in model
                if (is_array($incidentImages)) {
                    $incidentTime = $incident->created_at->format('H-i-s');
                    $incidentFolderName = "{$cleanSiteName}_{$incidentTime}";
                    foreach ($incidentImages as $index => $imagePath) {
                        $fullPath = storage_path('app/public/' . $imagePath);
                        if (file_exists($fullPath)) {
                            $extension = pathinfo($fullPath, PATHINFO_EXTENSION);
                            $zip->addFile($fullPath, "{$dutyFolder}/images/report_incident/{$incidentFolderName}/img_{$index}.{$extension}");
                        }
                    }
                }
            }

            $zip->close();
        }

        if (file_exists($zipPath)) {
            return response()->download($zipPath)->deleteFileAfterSend(true);
        }

        return back()->with('error', 'No evidence files found to download.');
    }

    public function approveCancellation(Schedule $schedule, Employee $employee)
    {
        // Update pivot status instead of detaching to allow the 'Cancelled' label visibility
        $schedule->employees()->updateExistingPivot($employee->id, [
            'cancellation_status' => 'approved'
        ]);

        // Check if ALL assigned guards are now cancelled
        $activeGuardsCount = $schedule->employees()
            ->wherePivot('cancellation_status', '!=', 'approved')
            ->count();

        $message = "Cancellation approved.";
        if ($activeGuardsCount === 0) {
            $schedule->update(['status' => 'cancelled']);
            $message .= " Only guard removed. All guards have cancelled, so Duty #" . $schedule->duty_number . " is now CANCELLED.";
        } else {
            $message .= " Guard has been removed from Duty #" . $schedule->duty_number . ".";
        }

        // Create notification for employee
        \App\Models\Notification::create([
            'company_id' => $schedule->company_id,
            'employee_id' => $employee->id,
            'title' => 'Cancellation Approved',
            'message' => 'Your request to cancel Duty #' . $schedule->duty_number . ' has been approved. You have been removed from this schedule.',
            'url' => route('employee.jobs.index', ['type' => 'all']),
        ]);

        return back()->with('success', $message);
    }

    public function rejectCancellation(Schedule $schedule, Employee $employee)
    {
        $schedule->employees()->updateExistingPivot($employee->id, [
            'cancellation_status' => 'rejected'
        ]);

        // Create notification for employee
        \App\Models\Notification::create([
            'company_id' => $schedule->company_id,
            'employee_id' => $employee->id,
            'title' => 'Cancellation Rejected',
            'message' => 'Your request to cancel Duty #' . $schedule->duty_number . ' has been rejected. You are still assigned to this schedule.',
            'url' => route('employee.jobs.index', ['type' => 'all']),
        ]);

        return back()->with('success', "Cancellation rejected. Employee remains on schedule.");
    }

    public function bulkUpdate(Request $request)
    {
        $request->validate([
            'schedule_ids' => 'required|string',
            'site_id' => 'nullable|exists:sites,id',
            'from_datetime' => 'nullable|date',
            'to_datetime' => 'nullable|date',
            'guard_action' => 'nullable|in:replace,add',
            'employee_ids' => 'nullable|array',
            'wage_rate' => 'nullable|numeric',
        ]);

        $ids = explode(',', $request->schedule_ids);
        $schedules = \App\Models\Schedule::whereIn('id', $ids)->get();

        if ($schedules->isEmpty()) {
            return back()->with('error', 'No schedules selected.');
        }

        DB::beginTransaction();
        try {
            foreach ($schedules as $schedule) {
                // Update Site
                if ($request->filled('site_id')) {
                    $schedule->site_id = $request->site_id;
                }

                // Update Dates
                if ($request->filled('from_datetime')) {
                    $schedule->from_datetime = $request->from_datetime;
                }
                if ($request->filled('to_datetime')) {
                    $schedule->to_datetime = $request->to_datetime;
                }

                $schedule->save();

                // Check Ban for Site Update (if site changed)
                if ($request->filled('site_id')) {
                    $currentGuards = $schedule->employees->pluck('id')->toArray();
                    $banned = DB::table('employee_site_bans')
                        ->where('site_id', $schedule->site_id)
                        ->whereIn('employee_id', $currentGuards)
                        ->exists();

                    if ($banned) {
                        throw new \Exception("Cannot move schedule #{$schedule->duty_number} to new site because assigned guards are banned there.");
                    }
                }

                // Handle Guards
                if ($request->guard_action && !empty($request->employee_ids)) {
                    // Check bans for new guards against (potentially new) site
                    $targetSiteId = $request->filled('site_id') ? $request->site_id : $schedule->site_id;
                    $bannedGuards = DB::table('employee_site_bans')
                        ->where('site_id', $targetSiteId)
                        ->whereIn('employee_id', $request->employee_ids)
                        ->exists();

                    if ($bannedGuards) {
                        throw new \Exception("Cannot assign selected guards to schedule #{$schedule->duty_number} because some are banned from the site.");
                    }

                    $syncData = [];
                    foreach ($request->employee_ids as $empId) {
                        // Determine Rate: Override OR generic logic (this is simplified as we can't query each guard's default wage type easily in bulk without complexity)
                        // If provided, use override. If not, use 0 or default handling logic relative to employee.
                        // For simplicity in bulk edit, if no rate is provided, we might default to 0 or need a better strategy. 
                        // Let's look up employee generic rate if possible or 0.
                        $rate = $request->wage_rate ?? 0; // Default to 0 or override

                        // We also need to construct the JSON wage_types structure if we want to be correct.
                        // Ideally, we fetch the employee's default wage types.
                        $employee = \App\Models\Employee::with('wageTypes')->find($empId);
                        $wageTypesJson = $employee->wageTypes->map(function ($wt) use ($rate) {
                            $r = $rate > 0 ? $rate : $wt->pivot->rate;
                            return [
                                'id' => $wt->id,
                                'name' => $wt->name,
                                'rate' => $r,
                                'allocated_hours' => 0 // In bulk edit we assume full shift or 0 initially? 
                                // Ideally allocated hours = shift duration.
                            ];
                        })->toJson();

                        $syncData[$empId] = [
                            'wage_types' => $wageTypesJson,
                            // 'rate' column is deprecated in favor of JSON but might still be there
                        ];
                    }

                    if ($request->guard_action === 'replace') {
                        $schedule->employees()->sync($syncData);
                    } elseif ($request->guard_action === 'add') {
                        $schedule->employees()->syncWithoutDetaching($syncData);
                    }
                }

                // Reset status if it was missed but now end time is in future
                if ($schedule->status === 'missed' && $schedule->to_datetime > now()) {
                    if ($schedule->from_datetime > now()) {
                        $schedule->update(['status' => 'upcoming']);
                    } else {
                        $schedule->update(['status' => 'today']);
                    }
                }
            }

            DB::commit();
            return back()->with('success', 'Selected schedules updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Update failed: ' . $e->getMessage());
        }
    }

    public function bulkDestroy(Request $request)
    {
        $request->validate([
            'schedule_ids' => 'required|string',
        ]);

        $ids = explode(',', $request->schedule_ids);
        $companyId = session('selected_company_id');

        // Only delete schedules belonging to this company that haven't started
        $count = \App\Models\Schedule::whereIn('id', $ids)
            ->where('company_id', $companyId)
            ->whereDoesntHave('employees', function ($q) {
                // Cannot bulk delete started schedules
                $q->whereNotNull('actual_start_at');
            })
            ->delete();

        if ($count == 0) {
            return back()->with('error', 'No schedules were deleted. They may have already started or do not exist.');
        }

        return back()->with('success', "$count schedules deleted successfully.");
    }
    public function emailFilteredSchedules(Request $request)
    {
        $companyId = session('selected_company_id');
        $siteId = $request->site_id;

        if (!$siteId) {
            return back()->with('error', 'Site ID is required.');
        }

        $site = Site::findOrFail($siteId);
        if (!$site->contact_email) {
            return back()->with('error', 'Site contact email is not set.');
        }

        $query = Schedule::with(['site', 'employees'])
            ->where('company_id', $companyId)
            ->where('site_id', $siteId);

        if ($request->filled('from_date')) {
            $query->whereDate('from_datetime', '>=', $request->from_date);
        }
        if ($request->filled('to_date')) {
            $query->whereDate('from_datetime', '<=', $request->to_date);
        }

        $schedules = $query->latest()->get();

        if ($schedules->isEmpty()) {
            return back()->with('error', 'No schedules found for the selected range.');
        }

        $type = 'filtered';
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('schedules.pdf', compact('schedules', 'type'));
        $pdfOutput = $pdf->output();

        $filename = "Schedules_Export_{$site->name}_" . now()->format('Ymd') . ".pdf";

        try {
            \Illuminate\Support\Facades\Mail::html("<p>Hello,</p><p>Please find attached the filtered schedule list for <strong>{$site->name}</strong>.</p><p>Regards,<br>Operations Team</p>", function ($message) use ($site, $pdfOutput, $filename) {
                $message->to($site->contact_email)
                    ->subject("Schedules List: {$site->name}")
                    ->attachData($pdfOutput, $filename, [
                        'mime' => 'application/pdf',
                    ]);
            });

            return back()->with('success', "Schedules list has been emailed to the client at {$site->contact_email}");
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error("Failed to send filtered schedules email: " . $e->getMessage());
            return back()->with('error', 'Failed to send email: ' . $e->getMessage());
        }
    }
}
