<?php

namespace App\Http\Controllers;

use App\Models\Route;
use App\Models\Site;
use App\Models\IdSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;

class RouteController extends Controller
{
    public function index(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = Route::where('company_id', $companyId)->with('sites');

        // Search
        if ($request->search) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        // Sort
        $sort = $request->get('sort', 'name');
        $direction = $request->get('direction', 'asc');
        $query->orderBy($sort, $direction);

        $routes = $query->paginate(100);

        return view('routes.index', compact('routes'));
    }

    public function create()
    {
        $companyId = session('selected_company_id');
        $sites = Site::where('company_id', $companyId)->where('active', true)->get();

        // Generate Route ID
        $setting = IdSetting::where('company_id', $companyId)->first();
        $prefix = $setting ? $setting->route_prefix : 'R';
        $nextNumber = $setting ? $setting->route_next_number : 1;
        $generatedId = $prefix . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);

        return view('routes.create', compact('sites', 'generatedId'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'route_code' => 'required|string|unique:routes,route_code',
            'sites' => 'nullable|array',
            'sites.*' => 'exists:sites,id',
            'rate' => 'nullable|numeric',
        ]);

        DB::transaction(function () use ($request) {
            $companyId = session('selected_company_id');

            $route = Route::create([
                'company_id' => $companyId,
                'route_code' => $request->route_code,
                'name' => $request->name,
                'rate' => $request->rate,
                'description' => $request->description,
            ]);

            if ($request->sites) {
                $route->sites()->attach($request->sites);
            }

            // Increment Route Number in IdSetting
            $setting = IdSetting::where('company_id', $companyId)->first();
            if ($setting) {
                $setting->increment('route_next_number');
            }
        });

        if ($request->has('create_another')) {
            return redirect()->route('routes.create')->with('success', 'Route created successfully!');
        }

        return redirect()->route('routes.index')->with('success', 'Route created successfully!');
    }

    public function show(Route $route)
    {
        if ($route->company_id != session('selected_company_id')) {
            abort(403);
        }

        return view('routes.show', compact('route'));
    }

    public function edit(Route $route)
    {
        if ($route->company_id != session('selected_company_id')) {
            abort(403);
        }

        $companyId = session('selected_company_id');
        $sites = Site::where('company_id', $companyId)->where('active', true)->get();
        $selectedSites = $route->sites->pluck('id')->toArray();

        return view('routes.edit', compact('route', 'sites', 'selectedSites'));
    }

    public function update(Request $request, Route $route)
    {
        if ($route->company_id != session('selected_company_id')) {
            abort(403);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'sites' => 'nullable|array',
            'sites.*' => 'exists:sites,id',
            'rate' => 'nullable|numeric',
        ]);

        DB::transaction(function () use ($request, $route) {
            $route->update([
                'name' => $request->name,
                'rate' => $request->rate,
                'description' => $request->description,
            ]);

            $route->sites()->sync($request->sites ?? []);
        });

        return redirect()->route('routes.index')->with('success', 'Route updated successfully!');
    }

    public function destroy(Route $route)
    {
        if ($route->company_id != session('selected_company_id')) {
            abort(403);
        }

        $route->delete();
        return redirect()->route('routes.index')->with('success', 'Route deleted successfully!');
    }

    public function toggleStatus(Route $route)
    {
        if ($route->company_id != session('selected_company_id')) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
        }

        $route->active = !$route->active;
        $route->save();

        return response()->json([
            'success' => true,
            'active' => $route->active,
            'message' => 'Route status updated successfully!'
        ]);
    }

    public function exportPdf(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = Route::where('company_id', $companyId)->with('sites');

        if ($request->search) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        $routes = $query->get();

        $headers = [
            'ID',
            'Route Code',
            'Name',
            'Sites',
            'Rate',
            'Description',
            'Active',
            'Created At',
            'Updated At'
        ];

        $data = $routes->map(function ($route) {
            return [
                $route->id,
                $route->route_code,
                $route->name,
                $route->sites->pluck('name')->implode(', ') ?: 'N/A', // Handle empty sites
                $route->rate,
                $route->description,
                $route->active ? 'Yes' : 'No',
                $route->created_at->format('Y-m-d H:i'),
                $route->updated_at->format('Y-m-d H:i'),
            ];
        });

        // Get company information
        $company = \App\Models\Company::find($companyId);

        $pdf = Pdf::loadView('pdf.sites_all', [ // Reusing the generic template
            'title' => 'Route Masters List',
            'headers' => $headers,
            'data' => $data,
            'company' => $company
        ])->setPaper('a4', 'landscape');

        return $pdf->download('routes-list-' . now()->format('Y-m-d') . '.pdf');
    }

    public function exportExcel(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = Route::where('company_id', $companyId)->with('sites');

        if ($request->search) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        $routes = $query->get();

        // Create new Spreadsheet
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set headers
        $headers = [
            'ID',
            'Route Code',
            'Name',
            'Sites',
            'Rate',
            'Description',
            'Active',
            'Created At',
            'Updated At'
        ];

        // Write headers
        $col = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($col . '1', $header);
            $col++;
        }

        // Style header row - Gray background and bold
        $headerStyle = [
            'font' => [
                'bold' => true,
            ],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => [
                    'rgb' => 'CCCCCC', // Gray color
                ],
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                    'color' => ['rgb' => '000000'],
                ],
            ],
        ];

        $lastColumn = chr(64 + count($headers)); // Convert number to letter
        $sheet->getStyle('A1:' . $lastColumn . '1')->applyFromArray($headerStyle);

        // Write data rows
        $row = 2;
        foreach ($routes as $route) {
            $data = [
                $route->id,
                $route->route_code,
                $route->name,
                $route->sites->pluck('name')->implode(', ') ?: 'N/A',
                $route->rate,
                $route->description,
                $route->active ? 'Yes' : 'No',
                $route->created_at->format('Y-m-d H:i'),
                $route->updated_at->format('Y-m-d H:i'),
            ];

            $col = 'A';
            foreach ($data as $value) {
                $sheet->setCellValue($col . $row, $value);
                $col++;
            }

            // Apply borders to data row
            $sheet->getStyle('A' . $row . ':' . $lastColumn . $row)->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        'color' => ['rgb' => '000000'],
                    ],
                ],
            ]);

            $row++;
        }

        // Auto-size columns
        foreach (range('A', $lastColumn) as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }

        // Create Excel file
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $filename = 'routes-list-' . now()->format('Y-m-d') . '.xlsx';

        // Set headers for download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }
}
