<?php

namespace App\Http\Controllers;

use App\Models\PatrollerSchedule;
use App\Models\Route;
use App\Models\Employee;
use App\Models\WageType;
use App\Models\Schedule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\StatHoliday;
use Illuminate\Support\Facades\Http;

class PatrollerScheduleController extends Controller
{
    public function index(Request $request)
    {
        $companyId = session('selected_company_id');

        // Get filter parameters
        $type = $request->get('type', 'all');
        $routeId = $request->get('route_id');
        $employeeId = $request->get('employee_id');
        $fromDate = $request->get('from_date');
        $toDate = $request->get('to_date');
        $sort = $request->get('sort', 'scheduled_date');
        $direction = $request->get('direction', 'desc');

        // Base query
        $query = PatrollerSchedule::with(['route', 'employees'])
            ->where('company_id', $companyId);

        // Apply status filter
        $now = now();
        switch ($type) {
            case 'active':
                $query->active();
                break;
            case 'today':
                $query->todayUpcoming();
                break;
            case 'upcoming':
                $query->upcomingFuture();
                break;
            case 'completed':
                $query->completed();
                break;
            case 'cancelled':
                $query->cancelled();
                break;
            case 'missed':
                $query->missed();
                break;
            case 'this_week':
                $query->whereBetween('from_time', [now()->startOfWeek(), now()->endOfWeek()]);
                break;
            case 'this_month':
                $query->whereBetween('from_time', [now()->startOfMonth(), now()->endOfMonth()]);
                break;
        }

        // Apply route filter
        if ($routeId) {
            $query->where('route_id', $routeId);
        }

        // Apply employee filter
        if ($employeeId) {
            $query->whereHas('employees', function ($q) use ($employeeId) {
                $q->where('employee_patroller_schedule.employee_id', $employeeId);
            });
        }

        // Apply date range filter
        if ($fromDate) {
            $query->whereDate('from_time', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('from_time', '<=', $toDate);
        }

        // Apply sorting
        switch ($sort) {
            case 'route':
                $query->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
                    ->orderBy('routes.name', $direction)
                    ->select('patroller_schedules.*');
                break;
            case 'status':
                $query->orderBy('status', $direction);
                break;
            default:
                $query->orderBy('scheduled_date', $direction)
                    ->orderBy('from_time', $direction);
        }

        $perPage = $request->input('per_page', 50);
        if ($perPage === 'all') {
            $perPage = $query->count();
        }
        $schedules = $query->paginate($perPage)->withQueryString();

        // Get counts for tabs using dynamic logic
        $counts = [
            'all' => PatrollerSchedule::where('company_id', $companyId)->count(),
            'active' => PatrollerSchedule::where('company_id', $companyId)->active()->count(),
            'today' => PatrollerSchedule::where('company_id', $companyId)->todayUpcoming()->count(),
            'upcoming' => PatrollerSchedule::where('company_id', $companyId)->upcomingFuture()->count(),
            'completed' => PatrollerSchedule::where('company_id', $companyId)->completed()->count(),
            'missed' => PatrollerSchedule::where('company_id', $companyId)->missed()->count(),
            'cancelled' => PatrollerSchedule::where('company_id', $companyId)->cancelled()->count(),
        ];

        // Get routes and employees for filters
        $routes = Route::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $employees = Employee::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('first_name')
            ->get();

        return view('patroller-schedules.index', compact('schedules', 'type', 'counts', 'routes', 'employees', 'fromDate', 'toDate'));
    }

    public function create()
    {
        $companyId = session('selected_company_id');

        $routes = Route::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $employees = Employee::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('first_name')
            ->get();

        $wageTypes = WageType::where('active', true)
            ->orderBy('name')
            ->get();

        $oldInput = old();

        // Get preview duty number
        // Get preview duty number
        $idSetting = \App\Models\IdSetting::where('company_id', $companyId)->first();
        $nextDutyNumber = 'Auto-generated';
        if ($idSetting) {
            $prefix = $idSetting->patroller_prefix ?? 'P';
            $nextNumber = $idSetting->patroller_next_number ?? 1;
            $nextDutyNumber = $prefix . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
        }

        $maxShiftHours = \App\Models\Setting::where('key', 'max_shift_hours')->value('value') ?? 12;

        return view('patroller-schedules.create', compact('routes', 'employees', 'wageTypes', 'oldInput', 'nextDutyNumber', 'maxShiftHours'));
    }

    public function store(Request $request)
    {
        $companyId = session('selected_company_id');

        $validated = $request->validate([
            'scheduled_date' => 'required|date',
            'route_id' => 'required|exists:routes,id',
            'route_rate' => 'required|numeric|min:0',
            'employee_ids' => 'required|array|min:1',
            'employee_ids.*' => 'exists:employees,id',
            'from_time' => 'required|date',
            'to_time' => 'required|date|after:from_time',
            'run_till_date' => 'nullable|date|after_or_equal:scheduled_date',
            'allowance' => 'nullable|numeric|min:0',
            'other_expense' => 'nullable|numeric|min:0',
            'customer_reference' => 'nullable|string|max:255',
            'comments' => 'nullable|string',
        ]);

        // Max Shift Hours Validation
        if (auth()->user()->role === 'user') {
            $maxShiftHours = (float) (\App\Models\Setting::where('key', 'max_shift_hours')->value('value') ?? 12);
            $from = Carbon::parse($request->from_time);
            $to = Carbon::parse($request->to_time);
            $durationHours = $from->diffInMinutes($to) / 60;

            if ($durationHours > $maxShiftHours) {
                return back()->withErrors(['error' => "Shift duration ({$durationHours} hrs) exceeds the maximum allowed shift hours ({$maxShiftHours} hrs) set by the administrator."])->withInput();
            }
        }

        // Validate employees
        $employeeIds = $request->employee_ids;
        $fromTime = Carbon::parse($request->from_time);
        $toTime = Carbon::parse($request->to_time);

        foreach ($employeeIds as $employeeId) {
            $employee = Employee::find($employeeId);

            // Check license expiry
            if ($employee->license_expiry && Carbon::parse($employee->license_expiry)->isPast()) {
                return back()->withErrors([
                    'employee_ids' => "Employee {$employee->first_name} {$employee->last_name}'s license has expired."
                ])->withInput();
            }

            // Check if already scheduled (guard schedules)
            $guardConflict = Schedule::whereHas('employees', function ($q) use ($employeeId) {
                $q->where('employee_schedule.employee_id', $employeeId);
            })
                ->where(function ($q) use ($fromTime, $toTime) {
                    $q->whereBetween('from_datetime', [$fromTime, $toTime])
                        ->orWhereBetween('to_datetime', [$fromTime, $toTime])
                        ->orWhere(function ($q2) use ($fromTime, $toTime) {
                            $q2->where('from_datetime', '<=', $fromTime)
                                ->where('to_datetime', '>=', $toTime);
                        });
                })
                ->exists();

            if ($guardConflict) {
                return back()->withErrors([
                    'employee_ids' => "Employee {$employee->first_name} {$employee->last_name} is already scheduled for a guard duty during this time."
                ])->withInput();
            }

            // Check if already scheduled (patroller schedules)
            $patrollerConflict = PatrollerSchedule::whereHas('employees', function ($q) use ($employeeId) {
                $q->where('employee_patroller_schedule.employee_id', $employeeId);
            })
                ->where(function ($q) use ($fromTime, $toTime) {
                    $q->whereBetween('from_time', [$fromTime, $toTime])
                        ->orWhereBetween('to_time', [$fromTime, $toTime])
                        ->orWhere(function ($q2) use ($fromTime, $toTime) {
                            $q2->where('from_time', '<=', $fromTime)
                                ->where('to_time', '>=', $toTime);
                        });
                })
                ->exists();

            if ($patrollerConflict) {
                return back()->withErrors([
                    'employee_ids' => "Employee {$employee->first_name} {$employee->last_name} is already scheduled for a patroller duty during this time."
                ])->withInput();
            }

            // Check if banned from any site in the route
            $route = Route::with('sites')->find($request->route_id);
            $bannedSites = $employee->bannedSites->pluck('id')->toArray();
            $routeSites = $route->sites->pluck('id')->toArray();
            $bannedInRoute = array_intersect($bannedSites, $routeSites);

            if (!empty($bannedInRoute)) {
                $bannedSiteNames = $route->sites->whereIn('id', $bannedInRoute)->pluck('name')->implode(', ');
                return back()->withErrors([
                    'employee_ids' => "Employee {$employee->first_name} {$employee->last_name} is banned from site(s): {$bannedSiteNames} in this route."
                ])->withInput();
            }
        }

        DB::beginTransaction();
        try {
            $fromTime = Carbon::parse($request->from_time);
            $toTime = Carbon::parse($request->to_time);

            // Determine date range - start date comes from from_time as per user logic
            $startDate = $fromTime->copy()->startOfDay();
            $endDate = $request->run_till_date ? Carbon::parse($request->run_till_date)->startOfDay() : $startDate;

            // Get excluded days (0=Sunday, 1=Monday, ..., 6=Saturday)
            $excludeDays = $request->exclude_days ?? [];

            $createdSchedules = [];
            $currentDate = $startDate->copy();

            while ($currentDate->lte($endDate)) {
                // Check if this day should be excluded
                $dayOfWeek = $currentDate->dayOfWeek; // 0=Sunday, 1=Monday, ..., 6=Saturday

                if (!in_array($dayOfWeek, $excludeDays)) {
                    // Generate duty number for each schedule
                    $dutyNumber = PatrollerSchedule::generateDutyNumber($companyId);

                    // Determine status
                    $status = 'pending';
                    if ($currentDate->isToday()) {
                        $status = 'pending';
                    } elseif ($currentDate->isFuture()) {
                        $status = 'upcoming';
                    }

                    // Adjust from_time and to_time for current date
                    $scheduleFromTime = $currentDate->copy()->setTimeFrom($fromTime);
                    $scheduleToTime = $currentDate->copy()->setTimeFrom($toTime);

                    // If to_time is earlier than from_time, it means it goes to next day
                    if ($scheduleToTime->lt($scheduleFromTime)) {
                        $scheduleToTime->addDay();
                    }

                    // Create patroller schedule
                    $schedule = PatrollerSchedule::create([
                        'company_id' => $companyId,
                        'duty_number' => $dutyNumber,
                        'scheduled_date' => $request->scheduled_date,
                        'route_id' => $request->route_id,
                        'route_rate' => $request->route_rate,
                        'from_time' => $scheduleFromTime,
                        'to_time' => $scheduleToTime,
                        'allowance' => $request->allowance,
                        'other_expense' => $request->other_expense,
                        'customer_reference' => $request->customer_reference,
                        'comments' => $request->comments,
                        'status' => $status,
                        'job_status' => 'pending',
                    ]);

                    // Attach employees with wage details
                    foreach ($employeeIds as $employeeId) {
                        $wageTypes = $request->input("wage_types_{$employeeId}", []);
                        $wageTypesWithHours = [];

                        // Build wage types array with allocated hours and names for easier display
                        foreach ($wageTypes as $wageTypeId) {
                            $rate = $request->input("wage_rate_{$employeeId}_{$wageTypeId}", 0);
                            $allocatedHours = $request->input("allocated_hours_{$employeeId}_{$wageTypeId}", 0);

                            $wtModel = \App\Models\WageType::find($wageTypeId);

                            $wageTypesWithHours[] = [
                                'id' => $wageTypeId,
                                'name' => $wtModel ? $wtModel->name : 'Unknown',
                                'rate' => floatval($rate),
                                'allocated_hours' => floatval($allocatedHours),
                            ];
                        }

                        $gasRate = $request->input("gas_rate_{$employeeId}", 0);
                        $allowance = $request->input("allowance_{$employeeId}", 0);
                        $otherExpense = $request->input("other_expense_{$employeeId}", 0);

                        $schedule->employees()->attach($employeeId, [
                            'wage_types' => json_encode($wageTypesWithHours),
                            'wage_rate' => 0, // Not used when wage_types are specified
                            'gas_rate' => $gasRate,
                            'allowance' => $allowance,
                            'other_expense' => $otherExpense,
                        ]);
                    }

                    $createdSchedules[] = $dutyNumber;
                }

                $currentDate->addDay();
            }

            DB::commit();

            $count = count($createdSchedules);
            $message = $count === 1
                ? "Patroller schedule {$createdSchedules[0]} created successfully."
                : "{$count} patroller schedules created successfully from {$startDate->format('M d')} to {$endDate->format('M d, Y')}.";

            return redirect()->route('patroller-schedules.index')
                ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Failed to create patroller schedule: ' . $e->getMessage()])
                ->withInput();
        }
    }

    public function show(PatrollerSchedule $patrollerSchedule)
    {
        if ($patrollerSchedule->company_id != session('selected_company_id')) {
            abort(403);
        }

        $patrollerSchedule->load(['route.sites', 'employees', 'company']);

        return view('patroller-schedules.show', compact('patrollerSchedule'));
    }

    public function edit(PatrollerSchedule $patrollerSchedule)
    {
        if ($patrollerSchedule->company_id != session('selected_company_id')) {
            abort(403);
        }

        $companyId = session('selected_company_id');

        $routes = Route::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('name')
            ->get();

        $employees = Employee::where('company_id', $companyId)
            ->where('active', true)
            ->orderBy('first_name')
            ->get();

        $wageTypes = WageType::where('active', true)
            ->orderBy('name')
            ->get();

        $patrollerSchedule->load(['route', 'employees']);
        $maxShiftHours = \App\Models\Setting::where('key', 'max_shift_hours')->value('value') ?? 12;

        return view('patroller-schedules.edit', compact('patrollerSchedule', 'routes', 'employees', 'wageTypes', 'maxShiftHours'));
    }

    public function update(Request $request, PatrollerSchedule $patrollerSchedule)
    {
        if ($patrollerSchedule->company_id != session('selected_company_id')) {
            abort(403);
        }

        $validated = $request->validate([
            'scheduled_date' => 'required|date',
            'route_id' => 'required|exists:routes,id',
            'route_rate' => 'required|numeric|min:0',
            'employee_ids' => 'required|array|min:1',
            'employee_ids.*' => 'exists:employees,id',
            'from_time' => 'required|date',
            'to_time' => 'required|date|after:from_time',
            'allowance' => 'nullable|numeric|min:0',
            'other_expense' => 'nullable|numeric|min:0',
            'customer_reference' => 'nullable|string|max:255',
            'comments' => 'nullable|string',
        ]);

        // Max Shift Hours Validation
        if (auth()->user()->role === 'user') {
            $maxShiftHours = (float) (\App\Models\Setting::where('key', 'max_shift_hours')->value('value') ?? 12);
            $from = Carbon::parse($request->from_time);
            $to = Carbon::parse($request->to_time);
            $durationHours = $from->diffInMinutes($to) / 60;

            if ($durationHours > $maxShiftHours) {
                return back()->withErrors(['error' => "Shift duration ({$durationHours} hrs) exceeds the maximum allowed shift hours ({$maxShiftHours} hrs) set by the administrator."])->withInput();
            }
        }

        DB::beginTransaction();
        try {
            $fromTime = Carbon::parse($request->from_time);
            $toTime = Carbon::parse($request->to_time);

            // Update schedule
            $patrollerSchedule->update([
                'scheduled_date' => $request->scheduled_date,
                'route_id' => $request->route_id,
                'route_rate' => $request->route_rate,
                'from_time' => $fromTime,
                'to_time' => $toTime,
                'customer_reference' => $request->customer_reference,
                'comments' => $request->comments,
            ]);

            // Sync employees with wage details
            $syncData = [];
            foreach ($request->employee_ids as $employeeId) {
                $wageTypes = $request->input("wage_types_{$employeeId}", []);
                $wageTypesWithHours = [];

                // Build wage types array with allocated hours and names for easier display
                foreach ($wageTypes as $wageTypeId) {
                    $rate = $request->input("wage_rate_{$employeeId}_{$wageTypeId}", 0);
                    $allocatedHours = $request->input("allocated_hours_{$employeeId}_{$wageTypeId}", 0);

                    $wtModel = \App\Models\WageType::find($wageTypeId);

                    $wageTypesWithHours[] = [
                        'id' => $wageTypeId,
                        'name' => $wtModel ? $wtModel->name : 'Unknown',
                        'rate' => floatval($rate),
                        'allocated_hours' => floatval($allocatedHours),
                    ];
                }

                $gasRate = $request->input("gas_rate_{$employeeId}", 0);
                $allowance = $request->input("allowance_{$employeeId}", 0);
                $otherExpense = $request->input("other_expense_{$employeeId}", 0);

                $syncData[$employeeId] = [
                    'wage_types' => json_encode($wageTypesWithHours),
                    'wage_rate' => 0, // Not used when wage_types are specified
                    'gas_rate' => $gasRate,
                    'allowance' => $allowance,
                    'other_expense' => $otherExpense,
                ];
            }

            $patrollerSchedule->employees()->sync($syncData);

            DB::commit();

            return redirect()->route('patroller-schedules.index')
                ->with('success', "Patroller schedule {$patrollerSchedule->duty_number} updated successfully.");

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Failed to update patroller schedule: ' . $e->getMessage()])
                ->withInput();
        }
    }

    public function destroy(PatrollerSchedule $patrollerSchedule)
    {
        if ($patrollerSchedule->company_id != session('selected_company_id')) {
            abort(403);
        }

        $dutyNumber = $patrollerSchedule->duty_number;
        $patrollerSchedule->delete();

        return redirect()->route('patroller-schedules.index')
            ->with('success', "Patroller schedule {$dutyNumber} deleted successfully.");
    }

    public function cancel(PatrollerSchedule $patrollerSchedule)
    {
        if ($patrollerSchedule->company_id != session('selected_company_id')) {
            abort(403);
        }

        if (in_array($patrollerSchedule->job_status, ['completed', 'auto_ended'])) {
            return back()->with('error', 'Cannot cancel a completed job.');
        }

        $patrollerSchedule->update([
            'status' => 'cancelled',
            'job_status' => 'cancelled'
        ]);

        return back()->with('success', "Patroller schedule {$patrollerSchedule->duty_number} has been cancelled.");
    }

    public function approveCancellation(PatrollerSchedule $patrollerSchedule, Employee $employee)
    {
        $patrollerSchedule->employees()->updateExistingPivot($employee->id, [
            'cancellation_status' => 'approved'
        ]);

        // Check if ALL assigned patrollers are now cancelled
        $activePatrollersCount = $patrollerSchedule->employees()
            ->wherePivot('cancellation_status', '!=', 'approved')
            ->count();

        $message = "Cancellation approved.";
        if ($activePatrollersCount === 0) {
            $patrollerSchedule->update(['status' => 'cancelled', 'job_status' => 'cancelled']);
            $message .= " All patrollers have cancelled, so Duty #" . $patrollerSchedule->duty_number . " is now CANCELLED.";
        } else {
            $message .= " Patroller has been removed from Duty #" . $patrollerSchedule->duty_number . ".";
        }

        // Create notification for employee
        \App\Models\Notification::create([
            'company_id' => $patrollerSchedule->company_id,
            'employee_id' => $employee->id,
            'title' => 'Cancellation Approved',
            'message' => 'Your request to cancel Patroller Duty #' . $patrollerSchedule->duty_number . ' has been approved. You have been removed from this schedule.',
            'url' => route('employee.patroller-jobs.index', ['type' => 'all']),
        ]);

        return back()->with('success', $message);
    }

    public function rejectCancellation(PatrollerSchedule $patrollerSchedule, Employee $employee)
    {
        $patrollerSchedule->employees()->updateExistingPivot($employee->id, [
            'cancellation_status' => 'rejected'
        ]);

        // Create notification for employee
        \App\Models\Notification::create([
            'company_id' => $patrollerSchedule->company_id,
            'employee_id' => $employee->id,
            'title' => 'Cancellation Rejected',
            'message' => 'Your request to cancel Patroller Duty #' . $patrollerSchedule->duty_number . ' has been rejected. You are still assigned to this schedule.',
            'url' => route('employee.patroller-jobs.index', ['type' => 'all']),
        ]);

        return back()->with('success', "Cancellation rejected. Patroller remains on schedule.");
    }

    // API Methods for AJAX calls

    public function getEmployeeWageTypes(Request $request, Employee $employee)
    {
        $companyId = session('selected_company_id');

        if ($employee->company_id != $companyId) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        \Log::info('Debugging SIN for Employee: ' . $employee->id . ' - Raw SIN: "' . $employee->sin_number . '"');

        // Calculate Holiday Split
        $holidaySplit = [
            'has_holiday' => false,
            'regular_hours' => 0,
            'holiday_hours' => 0,
            'multiplier' => 1.0,
            'holiday_name' => null
        ];

        $start = $request->query('from_time');
        $end = $request->query('to_time');

        if ($start && $end) {
            $startTime = Carbon::parse($start);
            $endTime = Carbon::parse($end);
            $totalMinutes = $startTime->diffInMinutes($endTime);

            // Get all stat holidays for the company in this range
            $startDateStr = $startTime->toDateString();
            $endDateStr = $endTime->toDateString();

            $statHolidaysInRange = StatHoliday::where('company_id', $employee->company_id)
                ->whereBetween('holiday_date', [$startDateStr, $endDateStr])
                ->where('active', true)
                ->get()
                ->keyBy('holiday_date');

            if ($statHolidaysInRange->count() > 0) {
                $holidaySplit['has_holiday'] = true;
                // Analyze each hour segment
                $tempTime = clone $startTime;
                while ($tempTime->lessThan($endTime)) {
                    $currentDate = $tempTime->toDateString();
                    $nextTime = (clone $tempTime)->addHour();
                    if ($nextTime->greaterThan($endTime))
                        $nextTime = clone $endTime;

                    $minutes = $tempTime->diffInMinutes($nextTime);
                    $hours = $minutes / 60;

                    if (isset($statHolidaysInRange[$currentDate])) {
                        $holidaySplit['holiday_hours'] += $hours;
                        $holidaySplit['holiday_name'] = $statHolidaysInRange[$currentDate]->name;
                        $holidaySplit['multiplier'] = max($holidaySplit['multiplier'], $statHolidaysInRange[$currentDate]->multiplier);
                    } else {
                        $holidaySplit['regular_hours'] += $hours;
                    }
                    $tempTime = $nextTime;
                }
            } else {
                $holidaySplit['regular_hours'] = $totalMinutes / 60;
            }
        }

        // Get employee's wage types from employee_wage_type table
        $employeeWageTypes = DB::table('employee_wage_type')
            ->join('wage_types', 'employee_wage_type.wage_type_id', '=', 'wage_types.id')
            ->where('employee_wage_type.employee_id', $employee->id)
            ->select(
                'wage_types.id',
                'wage_types.name',
                'employee_wage_type.rate'
            )
            ->get()
            ->map(function ($wageType) use ($employee) {
                $rate = $wageType->rate !== null ? (float) $wageType->rate : 0;

                // If rate is 0 and it's a SIN wage type, try to use employee's SIN number
                if ($rate <= 0 && stripos($wageType->name, 'SIN') !== false) {
                    $sinValue = preg_replace('/[^0-9.]/', '', $employee->sin_number);
                    if (is_numeric($sinValue) && (float) $sinValue > 0) {
                        $rate = (float) $sinValue;
                    }
                }

                return [
                    'id' => $wageType->id,
                    'name' => $wageType->name,
                    'rate' => $rate,
                    'hours' => 0,
                ];
            });

        // Check if SIN wage type is present
        $hasSin = $employeeWageTypes->contains(function ($wt) {
            return stripos($wt['name'], 'SIN') !== false;
        });

        if (!$hasSin) {
            $sinWageType = \App\Models\WageType::where('name', 'LIKE', '%SIN%')->first();
            if ($sinWageType) {
                $rate = 0;
                // Attempt to use sin_number as rate if numeric
                $sinValue = preg_replace('/[^0-9.]/', '', $employee->sin_number);
                if (is_numeric($sinValue) && (float) $sinValue > 0) {
                    $rate = (float) $sinValue;
                }

                $employeeWageTypes->push([
                    'id' => $sinWageType->id,
                    'name' => $sinWageType->name,
                    'rate' => $rate,
                    'hours' => 0,
                ]);
            }
        }

        \Log::info('Fetched wage types for employee', [
            'employee_id' => $employee->id,
            'wage_types' => $employeeWageTypes
        ]);

        return response()->json([
            'employee' => [
                'id' => $employee->id,
                'name' => $employee->first_name . ' ' . $employee->last_name,
                'license_expiry' => $employee->license_expiry,
                'license_expired' => $employee->license_expiry && Carbon::parse($employee->license_expiry)->isPast(),
                'holiday_split' => $holidaySplit,
                'unavailable_days' => array_values($employee->unavailable_days ?? []),
            ],
            'wage_types' => $employeeWageTypes,
        ]);
    }

    public function getRouteDetails(Route $route)
    {
        $companyId = session('selected_company_id');

        if ($route->company_id != $companyId) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $route->load('sites');

        return response()->json([
            'route' => [
                'id' => $route->id,
                'name' => $route->name,
                'description' => $route->description,
                'rate' => $route->rate ?? 0,
            ],
            'sites' => $route->sites->map(function ($site) {
                return [
                    'id' => $site->id,
                    'name' => $site->name,
                    'address' => $site->address,
                    'city' => $site->city,
                    'state' => $site->state,
                    'zip_code' => $site->zip_code,
                    'contact_person' => $site->contact_person,
                    'contact_phone' => $site->contact_phone,
                ];
            }),
        ]);
    }

    public function checkEmployeeAvailability(Request $request)
    {
        try {
            $companyId = session('selected_company_id');

            $validated = $request->validate([
                'employee_id' => 'required|exists:employees,id',
                'route_id' => 'required|exists:routes,id',
                'from_time' => 'nullable|date',
                'to_time' => 'nullable|date|after:from_time',
            ]);

            $employee = Employee::find($request->employee_id);
            $route = Route::with('sites')->find($request->route_id);

            if (!$employee || !$route) {
                return response()->json(['error' => 'Employee or route not found'], 404);
            }

            // Parse times only if provided
            $fromTime = $request->from_time ? Carbon::parse($request->from_time) : null;
            $toTime = $request->to_time ? Carbon::parse($request->to_time) : null;

            $warnings = [];
            $errors = [];

            \Log::info('Checking employee availability', [
                'employee_id' => $employee->id,
                'employee_name' => $employee->first_name . ' ' . $employee->last_name,
                'route_id' => $route->id,
                'route_name' => $route->name,
                'from_time' => $fromTime,
                'to_time' => $toTime,
                'license_expiry' => $employee->license_expiry,
                'unavailable_days' => $employee->unavailable_days
            ]);

            // Check General Availability (Unavailable Days)
            $unavailableDays = $employee->unavailable_days ?? [];

            // If a specific date is being checked, verify against unavailable days
            if ($fromTime) {
                $dayName = $fromTime->format('l'); // Monday, Tuesday, etc.
                if (in_array($dayName, $unavailableDays)) {
                    $errors[] = [
                        'type' => 'unavailable_day',
                        'message' => "Employee is marked as unavailable on {$dayName} s.",
                    ];
                }
            }

            // Check license expiry
            if ($employee->license_expiry && Carbon::parse($employee->license_expiry)->isPast()) {
                $errors[] = [
                    'type' => 'license_expired',
                    'message' => "License expired on " . Carbon::parse($employee->license_expiry)->format('M d, Y'),
                ];
                \Log::warning('License expired', ['employee_id' => $employee->id, 'expiry' => $employee->license_expiry]);
            }

            // Check banned sites in route
            try {
                $bannedSites = $employee->bannedSites()->pluck('sites.id')->toArray();
                $routeSites = $route->sites ? $route->sites->pluck('id')->toArray() : [];
                $bannedInRoute = array_intersect($bannedSites, $routeSites);

                \Log::info('Banned sites check', [
                    'employee_banned_sites' => $bannedSites,
                    'route_sites' => $routeSites,
                    'banned_in_route' => $bannedInRoute,
                ]);

                if (!empty($bannedInRoute)) {
                    $bannedSiteNames = $route->sites->whereIn('id', $bannedInRoute)->pluck('name')->toArray();
                    $errors[] = [
                        'type' => 'banned_sites',
                        'message' => "Banned from: " . implode(', ', $bannedSiteNames),
                        'sites' => $bannedSiteNames,
                    ];
                    \Log::warning('Employee banned from sites', ['employee_id' => $employee->id, 'sites' => $bannedSiteNames]);
                }
            } catch (\Exception $e) {
                \Log::error('Error checking banned sites', ['error' => $e->getMessage()]);
                // Continue without banned site check
            }

            // Check guard schedule conflicts (only if times are provided)
            $guardConflicts = [];
            if ($fromTime && $toTime) {
                $guardConflicts = Schedule::whereHas('employees', function ($q) use ($employee) {
                    $q->where('employee_id', $employee->id);
                })
                    ->where(function ($q) use ($fromTime, $toTime) {
                        $q->whereBetween('from_datetime', [$fromTime, $toTime])
                            ->orWhereBetween('to_datetime', [$fromTime, $toTime])
                            ->orWhere(function ($q2) use ($fromTime, $toTime) {
                                $q2->where('from_datetime', '<=', $fromTime)
                                    ->where('to_datetime', '>=', $toTime);
                            });
                    })
                    ->with('site')
                    ->get();
            }

            foreach ($guardConflicts as $conflict) {
                $warnings[] = [
                    'type' => 'guard_conflict',
                    'message' => "Guard duty at {$conflict->site->name} from " .
                        Carbon::parse($conflict->from_datetime)->format('M d, H:i') . " to " .
                        Carbon::parse($conflict->to_datetime)->format('H:i'),
                    'duty_number' => $conflict->duty_number,
                ];
            }

            // Check patroller schedule conflicts (only if times are provided)
            $patrollerConflicts = [];
            if ($fromTime && $toTime) {
                $patrollerConflicts = PatrollerSchedule::whereHas('employees', function ($q) use ($employee) {
                    $q->where('employee_id', $employee->id);
                })
                    ->where(function ($q) use ($fromTime, $toTime) {
                        $q->whereBetween('from_time', [$fromTime, $toTime])
                            ->orWhereBetween('to_time', [$fromTime, $toTime])
                            ->orWhere(function ($q2) use ($fromTime, $toTime) {
                                $q2->where('from_time', '<=', $fromTime)
                                    ->where('to_time', '>=', $toTime);
                            });
                    })
                    ->with('route')
                    ->get();
            }

            foreach ($patrollerConflicts as $conflict) {
                $warnings[] = [
                    'type' => 'patroller_conflict',
                    'message' => "Patroller duty on route {$conflict->route->name} from " .
                        Carbon::parse($conflict->from_time)->format('M d, H:i') . " to " .
                        Carbon::parse($conflict->to_time)->format('H:i'),
                    'duty_number' => $conflict->duty_number,
                ];
            }

            $response = [
                'employee' => [
                    'id' => $employee->id,
                    'name' => $employee->first_name . ' ' . $employee->last_name,
                    'unavailable_days' => array_values($unavailableDays),
                ],
                'has_errors' => !empty($errors),
                'has_warnings' => !empty($warnings),
                'errors' => $errors,
                'warnings' => $warnings,
            ];

            \Log::info('Validation response', $response);

            return response()->json($response);
        } catch (\Exception $e) {
            \Log::error('Error in checkEmployeeAvailability', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'error' => 'An error occurred while checking employee availability',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search for available employees based on date and time
     */
    public function searchAvailableEmployees(Request $request)
    {
        try {
            $companyId = session('selected_company_id');
            $scheduledDateRaw = $request->scheduled_date;
            $fromTimeRaw = $request->from_time;
            $toTimeRaw = $request->to_time;

            \Log::info('Quick Check started', [
                'company_id' => $companyId,
                'date' => $scheduledDateRaw,
                'from' => $fromTimeRaw,
                'to' => $toTimeRaw
            ]);

            if (!$scheduledDateRaw || !$fromTimeRaw || !$toTimeRaw) {
                return response()->json(['error' => 'Date, From and To times are required'], 422);
            }

            // Improved parsing: avoid double date specification by checking if input already has date
            try {
                if (strpos($fromTimeRaw, 'T') !== false || strpos($fromTimeRaw, '-') !== false) {
                    $fromTime = Carbon::parse($fromTimeRaw);
                } else {
                    $fromTime = Carbon::parse($scheduledDateRaw . ' ' . $fromTimeRaw);
                }

                if (strpos($toTimeRaw, 'T') !== false || strpos($toTimeRaw, '-') !== false) {
                    $toTime = Carbon::parse($toTimeRaw);
                } else {
                    $toTime = Carbon::parse($scheduledDateRaw . ' ' . $toTimeRaw);
                }
            } catch (\Exception $e) {
                \Log::error('Date parsing failed in Quick Check', [
                    'from' => $fromTimeRaw,
                    'to' => $toTimeRaw,
                    'date' => $scheduledDateRaw,
                    'error' => $e->getMessage()
                ]);
                return response()->json(['error' => 'Invalid date or time format: ' . $e->getMessage()], 422);
            }

            // Handle overnight shifts (e.g., 22:00 to 06:00)
            if ($toTime->lessThanOrEqualTo($fromTime)) {
                $toTime->addDay();
            }

            $dayName = $fromTime->format('l');

            // 1. Get IDs of employees with GUARD JOB conflicts
            $guardConflictEmpIds = \DB::table('employee_schedule')
                ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
                ->where('schedules.company_id', $companyId)
                ->where('schedules.status', '!=', 'cancelled')
                ->where(function ($q) use ($fromTime, $toTime) {
                    $q->whereBetween('from_datetime', [$fromTime, $toTime])
                        ->orWhereBetween('to_datetime', [$fromTime, $toTime])
                        ->orWhere(function ($q2) use ($fromTime, $toTime) {
                            $q2->where('from_datetime', '<=', $fromTime)
                                ->where('to_datetime', '>=', $toTime);
                        });
                })
                ->pluck('employee_id')
                ->unique()
                ->toArray();

            // 2. Get IDs of employees with PATROLLER JOB conflicts
            $patrollerConflictEmpIds = \DB::table('employee_patroller_schedule')
                ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
                ->where('patroller_schedules.company_id', $companyId)
                ->whereNotIn('patroller_schedules.job_status', ['cancelled'])
                ->where(function ($q) use ($fromTime, $toTime) {
                    $q->whereBetween('from_time', [$fromTime, $toTime])
                        ->orWhereBetween('to_time', [$fromTime, $toTime])
                        ->orWhere(function ($q2) use ($fromTime, $toTime) {
                            $q2->where('from_time', '<=', $fromTime)
                                ->where('to_time', '>=', $toTime);
                        });
                })
                ->pluck('employee_id')
                ->unique()
                ->toArray();

            $excludedEmpIds = array_unique(array_merge($guardConflictEmpIds, $patrollerConflictEmpIds));

            // 3. Get all active employees and filter manually for profile availability & license
            $employees = Employee::where('company_id', $companyId)
                ->where('active', true)
                ->whereNotIn('id', $excludedEmpIds)
                ->get();

            $availableEmployees = [];
            foreach ($employees as $employee) {
                // Profile Availability
                $unavailableDays = $employee->unavailable_days ?? [];
                if (is_array($unavailableDays) && in_array($dayName, $unavailableDays)) {
                    continue;
                }

                // License Expiry
                if ($employee->license_expiry && Carbon::parse($employee->license_expiry)->isPast()) {
                    continue;
                }

                $availableEmployees[] = [
                    'id' => $employee->id,
                    'name' => $employee->first_name . ' ' . $employee->last_name,
                    'emp_id' => $employee->employee_id ?? 'ID:' . $employee->id,
                    'photo_url' => $employee->profile_picture
                        ? asset('storage/' . $employee->profile_picture)
                        : 'https://ui-avatars.com/api/?name=' . urlencode($employee->first_name . ' ' . $employee->last_name) . '&color=7F9CF5&background=EBF4FF',
                ];
            }

            \Log::info('Quick Check completed', ['count' => count($availableEmployees)]);
            return response()->json($availableEmployees);
        } catch (\Exception $e) {
            \Log::error('Error in searchAvailableEmployees', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json(['error' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Export selected schedules to PDF
     */
    public function exportPdf(Request $request)
    {
        $ids = $request->input('ids');
        $companyId = session('selected_company_id');

        if (empty($ids)) {
            return back()->with('error', 'Please select at least one record to export.');
        }

        $schedules = PatrollerSchedule::with(['route', 'employees', 'jobSites', 'issueTickets'])
            ->where('company_id', $companyId)
            ->whereIn('id', $ids)
            ->orderBy('scheduled_date', 'desc')
            ->orderBy('from_time', 'desc')
            ->get();

        // Get company information
        $company = \App\Models\Company::find($companyId);

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('patroller-schedules.pdf', compact('schedules', 'company'))
            ->setPaper('a4', 'landscape')
            ->setOptions([
                'tempDir' => public_path(),
                'chroot' => public_path(),
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
                'defaultFont' => 'DejaVu Sans',
                'dpi' => 72, // Reduced DPI to decrease image file size
                'defaultCompression' => true // Enable internal PDF compression
            ]);

        return $pdf->download('patroller-schedules-' . now()->format('Y-m-d') . '.pdf');
    }

    /**
     * Export selected schedules to Excel
     */
    public function exportExcel(Request $request, \App\Services\ExcelExportService $excelService)
    {
        $ids = $request->input('ids');
        $companyId = session('selected_company_id');

        if (empty($ids)) {
            return back()->with('error', 'Please select at least one record to export.');
        }

        $schedules = PatrollerSchedule::with(['route', 'employees', 'jobSites', 'issueTickets'])
            ->where('company_id', $companyId)
            ->whereIn('id', $ids)
            ->orderBy('scheduled_date', 'desc')
            ->orderBy('from_time', 'desc')
            ->get();

        $headers = ['Date', 'Route', 'Employees', 'Job Status', 'Sched Time', 'Actual Time', 'Sites Visited', 'Tickets'];
        $data = [];

        foreach ($schedules as $schedule) {
            $employees = $schedule->employees->map(fn($e) => $e->user->name ?? ($e->first_name . ' ' . $e->last_name))->implode(', ');
            
            $schedTime = "In: " . $schedule->from_time->format('M d, Y h:i A') . "\nOut: " . $schedule->to_time->format('M d, Y h:i A');
            
            $actualTime = "-";
            if ($schedule->job_started_at || $schedule->job_ended_at) {
                $actualTime = "";
                if ($schedule->job_started_at) {
                    $actualTime .= "In: " . Carbon::parse($schedule->job_started_at)->format('M d, Y h:i A');
                }
                if ($schedule->job_ended_at) {
                    $actualTime .= ($actualTime ? "\n" : "") . "Out: " . Carbon::parse($schedule->job_ended_at)->format('M d, Y h:i A');
                }
            }

            $sitesVisited = $schedule->jobSites->whereNotNull('checked_in_at')->count() . ' / ' . $schedule->jobSites->count();

            $data[] = [
                $schedule->scheduled_date->format('M d, Y'),
                ($schedule->route->name ?? 'N/A') . ($schedule->route->run_no ? ' (' . $schedule->route->run_no . ')' : ''),
                $employees ?: 'Unassigned',
                ucfirst(str_replace('_', ' ', $schedule->job_status)),
                $schedTime,
                $actualTime,
                $sitesVisited,
                $schedule->issueTickets->count()
            ];
        }

        return $excelService->generateReport(
            'Patroller Schedules Report',
            'Summary',
            $headers,
            $data,
            [
                'filename_prefix' => 'Patroller_Schedules',
                'status_column_index' => 3,
                'center_columns' => [0, 3, 6, 7]
            ]
        );
    }

    /**
     * Export single schedule detail to PDF
     */
    public function exportSinglePdf(PatrollerSchedule $patroller_schedule)
    {
        $schedule = $patroller_schedule->load([
            'route.sites',
            'employees',
            'jobSites' => function ($query) {
                $query->orderBy('checked_in_at', 'asc');
            },
            'issueTickets.site',
            'issueTickets.employee'
        ]);

        // Get company information
        $company = \App\Models\Company::find($schedule->company_id);

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('patroller-schedules.single-pdf', compact('schedule', 'company'))
            ->setPaper('a4', 'portrait')
            ->setOptions([
                'tempDir' => public_path(),
                'chroot' => public_path(),
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
                'defaultFont' => 'DejaVu Sans',
                'dpi' => 72, // Reduced DPI to decrease image file size
                'defaultCompression' => true // Enable internal PDF compression
            ]);

        return $pdf->download('patroller-schedule-' . $schedule->duty_number . '-' . now()->format('Y-m-d') . '.pdf');
    }

    /**
     * Export ticket images as ZIP file organized by site
     */
    public function exportTicketImages(PatrollerSchedule $patroller_schedule)
    {
        $schedule = $patroller_schedule->load([
            'route',
            'issueTickets.site'
        ]);

        // Check if there are any tickets with images
        $ticketsWithImages = $schedule->issueTickets->filter(function ($ticket) {
            return !empty($ticket->images) && is_array($ticket->images) && count($ticket->images) > 0;
        });

        if ($ticketsWithImages->count() === 0) {
            return back()->with('error', 'No ticket images found for this duty.');
        }

        // Create folder name: DutyNumber_RouteName
        $folderName = $schedule->duty_number . '_' . ($schedule->route->name ?? 'Route');
        $folderName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $folderName); // Sanitize

        // Group tickets by site
        $ticketsBySite = $ticketsWithImages->groupBy('site_id');

        $zipperMode = env('ZIPPER_MODE', 'Internal');

        if (strtolower($zipperMode) === 'external') {
            $fileEntries = [];
            foreach ($ticketsBySite as $siteId => $tickets) {
                $site = $tickets->first()->site;
                $siteName = $site ? preg_replace('/[^A-Za-z0-9_\-]/', '_', $site->name) : 'Unknown_Site';

                foreach ($tickets as $ticket) {
                    if (!empty($ticket->images) && is_array($ticket->images)) {
                        foreach ($ticket->images as $index => $imagePath) {
                            $sourceImage = storage_path('app/public/' . $imagePath);
                            if (file_exists($sourceImage)) {
                                $extension = pathinfo($imagePath, PATHINFO_EXTENSION);
                                $filename = $ticket->ticket_number . '_' . ($index + 1) . '.' . $extension;
                                
                                $fileEntries[] = [
                                    'source_path' => $sourceImage,
                                    'target_path' => $folderName . '/' . $siteName . '/' . $filename
                                ];
                            }
                        }
                    }
                }
            }

            if (!empty($fileEntries)) {
                try {
                    $response = Http::timeout(300)->withOptions(['stream' => true])
                        ->post(env('ZIPPER_URL', 'http://localhost:8080/zip'), [
                            'zip_name' => $folderName . '_' . now()->format('Ymd_His'),
                            'files' => $fileEntries
                        ]);

                    if ($response->successful()) {
                        return response()->streamDownload(function () use ($response) {
                            $stream = $response->toPsrResponse()->getBody();
                            while (!$stream->eof()) {
                                echo $stream->read(8192);
                                if (ob_get_level() > 0) ob_flush();
                                flush();
                            }
                        }, $folderName . '_' . now()->format('Ymd_His') . '.zip', [
                            'Content-Type' => 'application/zip',
                        ]);
                    }
                } catch (\Exception $e) {
                    \Log::error('Zipper Microservice Error: ' . $e->getMessage());
                    // Fallback to internal mode happens automatically by continuing
                }
            }
        }

        // [Internal Mode] Create temporary directory
        $tempPath = storage_path('app/temp/' . uniqid('tickets_'));
        if (!file_exists($tempPath)) {
            mkdir($tempPath, 0755, true);
        }

        $mainFolder = $tempPath . '/' . $folderName;
        mkdir($mainFolder, 0755, true);

        foreach ($ticketsBySite as $siteId => $tickets) {
            $site = $tickets->first()->site;
            $siteName = $site ? preg_replace('/[^A-Za-z0-9_\-]/', '_', $site->name) : 'Unknown_Site';

            $siteFolder = $mainFolder . '/' . $siteName;
            mkdir($siteFolder, 0755, true);

            foreach ($tickets as $ticket) {
                if (!empty($ticket->images) && is_array($ticket->images)) {
                    foreach ($ticket->images as $index => $imagePath) {
                        // Get the full path to the image
                        $sourceImage = storage_path('app/public/' . $imagePath);

                        if (file_exists($sourceImage)) {
                            // Create filename: TicketNumber_ImageIndex.extension
                            $extension = pathinfo($imagePath, PATHINFO_EXTENSION);
                            $filename = $ticket->ticket_number . '_' . ($index + 1) . '.' . $extension;

                            // Copy image to site folder
                            copy($sourceImage, $siteFolder . '/' . $filename);
                        }
                    }
                }
            }
        }

        // Create ZIP file
        $zipFileName = $folderName . '_' . now()->format('Ymd_His') . '.zip';
        $zipPath = storage_path('app/temp/' . $zipFileName);

        $zip = new \ZipArchive();
        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === true) {
            // Add files to ZIP
            $files = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($mainFolder),
                \RecursiveIteratorIterator::LEAVES_ONLY
            );

            foreach ($files as $file) {
                if (!$file->isDir()) {
                    $filePath = $file->getRealPath();
                    $relativePath = substr($filePath, strlen($tempPath) + 1);
                    $zip->addFile($filePath, $relativePath);
                }
            }

            $zip->close();

            // Clean up temp folder
            $this->deleteDirectory($mainFolder);

            // Download and then delete ZIP
            return response()->download($zipPath)->deleteFileAfterSend(true);
        }

        return back()->with('error', 'Failed to create ZIP file.');
    }

    /**
     * Recursively delete a directory
     */
    private function deleteDirectory($dir)
    {
        if (!file_exists($dir)) {
            return true;
        }

        if (!is_dir($dir)) {
            return unlink($dir);
        }

        foreach (scandir($dir) as $item) {
            if ($item == '.' || $item == '..') {
                continue;
            }

            if (!$this->deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {
                return false;
            }
        }

        return rmdir($dir);
    }

    public function getLocations(PatrollerSchedule $patroller_schedule)
    {
        $locations = $patroller_schedule->jobSites()
            ->where(function ($query) {
                $query->where(function ($q) {
                    $q->whereNotNull('checkin_latitude')->whereNotNull('checkin_longitude');
                })->orWhere(function ($q) {
                    $q->whereNotNull('checkout_latitude')->whereNotNull('checkout_longitude');
                });
            })
            ->with(['site', 'employee'])
            ->get()
            ->flatMap(function ($jobSite) {
                $points = [];
                $employeeName = $jobSite->employee ? ($jobSite->employee->first_name . ' ' . $jobSite->employee->last_name) : 'Unknown';

                // Check-in point
                if ($jobSite->checkin_latitude && $jobSite->checkin_longitude) {
                    $points[] = [
                        'lat' => $jobSite->checkin_latitude,
                        'lng' => $jobSite->checkin_longitude,
                        'time' => $jobSite->checked_in_at ? $jobSite->checked_in_at->format('h:i A') : 'N/A',
                        'name' => "{$employeeName} (Check In @ {$jobSite->site->name})",
                        'site_name' => $jobSite->site->name,
                        'type' => 'checkin'
                    ];
                }
                // Check-out point
                if ($jobSite->checkout_latitude && $jobSite->checkout_longitude) {
                    $points[] = [
                        'lat' => $jobSite->checkout_latitude,
                        'lng' => $jobSite->checkout_longitude,
                        'time' => $jobSite->checked_out_at ? $jobSite->checked_out_at->format('h:i A') : 'N/A',
                        'name' => "{$employeeName} (Check Out @ {$jobSite->site->name})",
                        'site_name' => $jobSite->site->name,
                        'type' => 'checkout'
                    ];
                }
                return $points;
            })
            ->values();

        return response()->json($locations);
    }
}
