<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Site;
use App\Models\Employee;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use App\Models\Schedule;
use Illuminate\Support\Facades\Mail;

class OperationalReportController extends Controller
{
    public function index(Request $request)
    {
        $company_id = session('selected_company_id');
        $reportType = trim($request->input('type', 'checkin'));

        $query = $this->buildQuery($company_id, $request);
        $reports = $query->paginate(20);

        // Filters Data
        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();
        $employees = Employee::where('company_id', $company_id)->orderBy('first_name')->get();

        // Return the USER view
        $company = \App\Models\Company::find($company_id);
        return view('reports.operational.index', compact('reports', 'reportType', 'sites', 'employees', 'company'));
    }

    public function exportPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $reportType = trim($request->input('type', 'checkin'));

        $query = $this->buildQuery($company_id, $request);
        $reports = $query->get();

        // Reusing the PDF view, might need to duplicate if layout differs heavily, 
        // but typically PDF reports are standalone.
        // Let's assume admin PDF is fine, or we can make a generic one.
        // For now, I'll use the existing admin PDF view as it likely doesn't have nav/sidebar.
        $company = \App\Models\Company::find($company_id);
        $pdf = Pdf::loadView('admin.reports.operational.pdf', compact('reports', 'reportType', 'company'));
        return $pdf->download("Operational_Report_{$reportType}_" . now()->format('Ymd_Hi') . '.pdf');
    }

    public function exportExcel(Request $request)
    {
        $company_id = session('selected_company_id');
        $reportType = trim($request->input('type', 'checkin'));

        $query = $this->buildQuery($company_id, $request);
        $reports = $query->get();

        $filename = "Operational_Report_{$reportType}_" . now()->format('Ymd_Hi') . ".csv";

        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];

        $columns = ['Date', 'Duty Number', 'Employee', 'Site Name', 'Scheduled Start', 'Scheduled End', 'Actual Start', 'Actual End', 'Status'];

        $callback = function () use ($reports, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($reports as $schedule) {
                $empName = $schedule->employee_name ?? 'N/A';

                fputcsv($file, [
                    Carbon::parse($schedule->schedule_date)->format('Y-m-d'),
                    $schedule->duty_number,
                    $empName,
                    $schedule->site_name ?? 'Unknown',
                    Carbon::parse($schedule->from_datetime)->format('Y-m-d H:i'),
                    Carbon::parse($schedule->to_datetime)->format('Y-m-d H:i'),
                    $schedule->actual_start_at ? Carbon::parse($schedule->actual_start_at)->format('Y-m-d H:i') : 'N/A',
                    $schedule->actual_end_at ? Carbon::parse($schedule->actual_end_at)->format('Y-m-d H:i') : 'N/A',
                    ucfirst($schedule->status)
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function buildQuery($company_id, Request $request)
    {
        $reportType = trim($request->input('type', 'checkin'));

        if ($reportType === 'scans_history') {
            $query = DB::table('employee_schedule')
                ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
                ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
                ->join('sites', 'schedules.site_id', '=', 'sites.id')
                ->where('schedules.company_id', $company_id)
                ->whereIn('schedules.status', ['active', 'completed'])
                ->select(
                    'schedules.id as schedule_id',
                    'schedules.duty_number',
                    'sites.name as site_name',
                    'schedules.from_datetime',
                    'schedules.to_datetime',
                    'employee_schedule.actual_start_at',
                    'employee_schedule.actual_end_at',
                    'schedules.status',
                    'employees.id as employee_pk',
                    'employee_schedule.id as employee_schedule_id',
                    DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                    DB::raw("(SELECT COUNT(*) FROM checkpoint_scans WHERE checkpoint_scans.schedule_id = schedules.id AND checkpoint_scans.employee_id = employees.id) as total_scans")
                )
                // ->having('total_scans', '>', 0) // Optional filter
                ->orderBy('schedules.schedule_date', 'desc');

            if ($request->filled('start_date')) {
                $query->whereDate('schedules.schedule_date', '>=', $request->start_date);
            }
            if ($request->filled('end_date')) {
                $query->whereDate('schedules.schedule_date', '<=', $request->end_date);
            }
            if ($request->filled('site_id')) {
                $query->where('schedules.site_id', $request->site_id);
            }
            if ($request->filled('employee_id')) {
                $query->where('employee_schedule.employee_id', $request->employee_id);
            }
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where('schedules.duty_number', 'like', "%{$search}%");
            }

            return $query;
        }

        $query = DB::table('employee_schedule')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->where('schedules.company_id', $company_id)
            ->select(
                'schedules.id as schedule_id',
                'schedules.schedule_date',
                'schedules.duty_number',
                'schedules.status',
                'schedules.from_datetime',
                'schedules.to_datetime',
                'sites.name as site_name',
                'employees.first_name',
                'employees.last_name',
                'employees.id as employee_id',
                (DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name")),
                'employee_schedule.actual_start_at',
                'employee_schedule.actual_end_at',
                'employee_schedule.id as report_id'
            )
            ->whereNotNull('employee_schedule.actual_start_at')
            ->orderBy('schedules.schedule_date', 'desc');

        if ($reportType === 'checkout') {
            $query->whereNotNull('employee_schedule.actual_end_at');
        }

        // Filters
        if ($request->filled('start_date')) {
            $query->whereDate('schedules.schedule_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('schedules.schedule_date', '<=', $request->end_date);
        }
        if ($request->filled('site_id')) {
            $query->where('schedules.site_id', $request->site_id);
        }
        if ($request->filled('employee_id')) {
            $query->where('employee_schedule.employee_id', $request->employee_id);
        }
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('schedules.duty_number', 'like', "%{$search}%");
        }

        return $query;
    }

    public function incidentAnalysis(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = \App\Models\Incident::whereHas('employee', function ($q) use ($companyId) {
            $q->where('company_id', $companyId);
        });

        // Filters
        if ($request->filled('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
        if ($request->filled('site_id')) {
            $query->whereHas('schedule', function ($q) use ($request) {
                $q->where('site_id', $request->site_id);
            });
        }
        if ($request->filled('employee_id')) {
            $query->where('employee_id', $request->employee_id);
        }

        // Parent Category Stats
        $parentStats = (clone $query)
            ->with('incidentParentCategory')
            ->select('incident_parent_category_id', \Illuminate\Support\Facades\DB::raw('count(*) as count'))
            ->groupBy('incident_parent_category_id')
            ->get()
            ->map(function ($item) {
                return [
                    'label' => $item->incidentParentCategory->description ?? 'Uncategorized',
                    'count' => $item->count
                ];
            });

        // Top 5 Categories
        $topCategories = (clone $query)
            ->with('incidentCategory')
            ->select('incident_category_id', \Illuminate\Support\Facades\DB::raw('count(*) as count'))
            ->groupBy('incident_category_id')
            ->orderByDesc('count')
            ->limit(5)
            ->get()
            ->map(function ($item) {
                return [
                    'label' => $item->incidentCategory->description ?? 'Other',
                    'count' => $item->count
                ];
            });

        // Incidents Over Time
        $incidentsOverTime = (clone $query)
            ->selectRaw('DATE(created_at) as date, count(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Site distribution (Top 5)
        $siteStats = (clone $query)
            ->join('schedules', 'incidents.schedule_id', '=', 'schedules.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->select('sites.name', \Illuminate\Support\Facades\DB::raw('count(*) as count'))
            ->groupBy('sites.name')
            ->orderByDesc('count')
            ->limit(5)
            ->get();

        // Populate filters
        $sites = \App\Models\Site::where('company_id', $companyId)->where('active', true)->orderBy('name')->get();
        $employees = \App\Models\Employee::where('company_id', $companyId)->where('active', true)->orderBy('first_name')->get();

        return view('reports.operational.incident_analysis', compact('parentStats', 'topCategories', 'incidentsOverTime', 'siteStats', 'sites', 'employees'));
    }

    public function incidentCounts(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getIncidentCountsData($company_id, $request);
        $reportData = $data['reportData'];
        $days = $data['days'];
        $month = $data['month'];
        $year = $data['year'];
        // Pass $data['sites'] as $sites to view if needed, but in original code 
        // $sites was the result of query BEFORE reloading all sites.
        // The getIncidentCountsData returns filtered sites in 'sites'.
        // We also need all sites for the dropdown.
        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();

        // Reload all sites for filter dropdown if site_id was filtered
        $allSites = Site::where('company_id', $company_id)->orderBy('name')->get();

        return view('reports.operational.incident_counts', compact('reportData', 'days', 'month', 'year', 'sites'));
    }

    public function exportIncidentCountsPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getIncidentCountsData($company_id, $request);

        $reportData = $data['reportData'];
        $days = $data['days'];
        $month = $data['month'];
        $year = $data['year'];
        $sites = $data['sites'];

        $monthName = Carbon::create(null, $month, 1)->format('F');

        $company = \App\Models\Company::find($company_id);
        $pdf = Pdf::loadView('admin.reports.operational.incident_counts_pdf', compact('reportData', 'days', 'month', 'year', 'sites', 'monthName', 'company'))
            ->setPaper('a4', 'landscape');

        return $pdf->download("Incident_Counts_{$monthName}_{$year}.pdf");
    }

    public function guardPayouts(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getGuardPayoutData($company_id, $request);
        $payouts = $data['payouts'];
        $groupedPayouts = $data['groupedPayouts'];

        $employees = Employee::where('company_id', $company_id)->orderBy('first_name')->get();

        return view('reports.operational.guard_payouts', compact('payouts', 'groupedPayouts', 'employees'));
    }

    public function exportGuardPayoutsPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getGuardPayoutData($company_id, $request);
        $groupedPayouts = $data['groupedPayouts'];
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $reportType = $request->get('report_type', 'actual');
        $company = \App\Models\Company::find($company_id);
        $pdf = Pdf::loadView('admin.reports.operational.guard_payouts_pdf', compact('groupedPayouts', 'startDate', 'endDate', 'reportType', 'company'))
            ->setPaper('a4', 'landscape');

        return $pdf->download("Guard_Payouts_{$reportType}_{$startDate}_{$endDate}.pdf");
    }

    public function exportGuardPayoutsExcel(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getGuardPayoutData($company_id, $request);
        $groupedPayouts = $data['groupedPayouts'];
        
        $reportType = $request->get('report_type', 'actual');
        $headers = ['Employee / Duty #', 'Site', 'Site ID', 'From', 'To', 'Hrs', 'Wage Components', 'Total Amount'];
        
        $excelData = [];
        foreach ($groupedPayouts as $employeeName => $group) {
            // Employee Header Row
            $excelData[] = [$employeeName, '', '', '', '', '', '', ''];

            foreach ($group['items'] as $row) {
                // Build wage components string
                $wageTypes = json_decode($row->wage_types ?? '[]', true);
                $wageComponentsStr = '';
                if (!empty($wageTypes)) {
                    $components = [];
                    foreach ($wageTypes as $wageType) {
                        $componentHours = $wageType['allocated_hours'] ?? 0;
                        $componentRate = $wageType['rate'] ?? 0;
                        $componentPay = $componentHours * $componentRate;
                        $components[] = "{$wageType['name']}: " . number_format($componentHours, 2) . "hrs × $" . number_format($componentRate, 2) . " = $" . number_format($componentPay, 2);
                    }
                    $wageComponentsStr = implode("\n", $components);
                }

                $gasRate = $row->gas_rate ?? 0;
                if ($gasRate > 0) {
                    $wageComponentsStr .= ($wageComponentsStr ? "\n" : "") . "GAS/CASH: $" . number_format($gasRate, 2);
                }

                $excelData[] = [
                    '  #' . $row->duty_number . (isset($row->stat_multiplier) && $row->stat_multiplier > 1 ? ' (STAT)' : ''),
                    $row->site_name,
                    $row->site_custom_id,
                    Carbon::parse($row->view_start)->format('M d, Y H:i'),
                    Carbon::parse($row->view_end)->format('M d, Y H:i'),
                    number_format($row->view_hours, 2),
                    $wageComponentsStr ?: 'No wage data',
                    '$' . number_format($row->total_amount, 2)
                ];
            }

            // Subtotal Row
            $excelData[] = [
                '  Total',
                '',
                '',
                '',
                '',
                number_format($group['total_hours'], 2),
                '',
                '$' . number_format($group['total_amount'], 2)
            ];
            
            // Empty row for spacing
            $excelData[] = ['', '', '', '', '', '', '', ''];
        }

        $excelService = new \App\Services\ExcelExportService();
        return $excelService->generateReport(
            'Guard Payout Summary',
            ucfirst($reportType) . ' Hours',
            $headers,
            $excelData,
            [
                'filename_prefix' => 'Guard_Payouts',
                'center_columns' => [2, 5],
            ]
        );
    }

    private function getGuardPayoutData($company_id, Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));
        $reportType = $request->get('report_type', 'actual');

        $query = DB::table('employee_schedule')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->leftJoin('stat_holidays', function ($join) use ($company_id) {
                $join->on(DB::raw('DATE(schedules.schedule_date)'), '=', 'stat_holidays.holiday_date')
                    ->where('stat_holidays.company_id', '=', $company_id)
                    ->where('stat_holidays.active', '=', true);
            })
            ->where('schedules.company_id', $company_id)
            ->where('schedules.status', 'completed')
            ->whereNotNull('employee_schedule.actual_start_at')
            ->whereNotNull('employee_schedule.actual_end_at')
            ->whereDate('schedules.schedule_date', '>=', $startDate)
            ->whereDate('schedules.schedule_date', '<=', $endDate);

        if ($request->filled('employee_id')) {
            $query->where('employee_schedule.employee_id', $request->employee_id);
        }

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_schedule.id', $ids);
        }

        $payouts = $query->select(
            'employee_schedule.id',
            'schedules.schedule_date',
            'schedules.duty_number',
            'schedules.from_datetime',
            'schedules.to_datetime',
            'sites.name as site_name',
            'sites.site_id as site_custom_id',
            'employees.id as employee_id',
            'employees.first_name',
            'employees.last_name',
            'employees.email',
            'employee_schedule.actual_start_at',
            'employee_schedule.actual_end_at',
            'employee_schedule.wage_types',
            DB::raw("employee_schedule.wage_rate * COALESCE(stat_holidays.multiplier, 1) as wage_rate"),
            DB::raw("employee_schedule.gas_rate * COALESCE(stat_holidays.multiplier, 1) as gas_rate"),
            DB::raw("(SELECT rate FROM employee_wage_type WHERE employee_wage_type.employee_id = employees.id ORDER BY rate DESC LIMIT 1) * COALESCE(stat_holidays.multiplier, 1) as fallback_rate"),
            DB::raw("COALESCE(stat_holidays.multiplier, 1) as stat_multiplier"),
            'stat_holidays.name as stat_holiday_name',
            DB::raw("(SELECT COUNT(*) FROM incidents WHERE incidents.schedule_id = schedules.id AND incidents.employee_id = employees.id) as tickets_count")
        )
            ->orderBy('employees.first_name')
            ->orderBy('schedules.schedule_date')
            ->get();

        // Fetch Employee Wage Rates for lookup
        $employeeIds = $payouts->pluck('employee_id')->unique()->toArray();
        $employeeRates = DB::table('employee_wage_type')
            ->join('wage_types', 'employee_wage_type.wage_type_id', '=', 'wage_types.id')
            ->whereIn('employee_id', $employeeIds)
            ->select('employee_id', 'wage_type_id', 'rate', 'wage_types.name as type_name')
            ->get()
            ->groupBy('employee_id')
            ->map(function ($items) {
                return $items->keyBy('wage_type_id');
            });

        // Process grouping and calculation
        $groupedPayouts = [];

        foreach ($payouts as $payout) {
            $empName = $payout->first_name . ' ' . $payout->last_name . ' - ' . $payout->duty_number;
            $empKey = "EMP{$payout->employee_id} - {$payout->first_name} {$payout->last_name}";

            $scheduledStart = Carbon::parse($payout->from_datetime);
            $scheduledEnd = Carbon::parse($payout->to_datetime);
            $actualStart = Carbon::parse($payout->actual_start_at);
            $actualEnd = Carbon::parse($payout->actual_end_at);

            // Determine effective start/end for SPLITTING purposes
            // If report_type is actual, we split based on actual times.
            // If report_type is scheduled, we split based on scheduled times (?)
            // For simplicity, let's keep the split logic centered on the 'Active' timeframe.
            // If mode is 'actual', active is actualStart/End.
            // If mode is 'scheduled', active is scheduledStart/End.

            if ($reportType == 'scheduled') {
                $effStart = $scheduledStart;
                $effEnd = $scheduledEnd;
            } else {
                // Raw Actuals (no early login clamping, as per user request for "Actual Working usage")
                $effStart = $actualStart;
                $effEnd = $actualEnd;
            }

            $itemsToProcess = [];
            $splitMode = $request->filled('split_shifts');

            if ($splitMode && $effStart->toDateString() !== $effEnd->toDateString()) {
                // Split at Midnight (Start of Next Day)
                $midnight = $effStart->copy()->addDay()->startOfDay();

                if ($midnight->lessThan($effEnd)) {
                    $item1 = clone $payout;
                    // We temporarily override the fields to calculate duration correctly for the part
                    // But we also want to preserve the original "other" times for display? 
                    // Let's store "split_start" and "split_end" for calculation.
                    $item1->calc_start = $effStart;
                    $item1->calc_end = $midnight;
                    $item1->is_split_part = true;
                    $itemsToProcess[] = $item1;

                    $item2 = clone $payout;
                    $item2->calc_start = $midnight;
                    $item2->calc_end = $effEnd;
                    $item2->schedule_date = $midnight->format('Y-m-d'); // Update date bucket
                    $item2->is_split_part = true;
                    $itemsToProcess[] = $item2;
                } else {
                    $payout->calc_start = $effStart;
                    $payout->calc_end = $effEnd;
                    $itemsToProcess[] = $payout;
                }
            } else {
                $payout->calc_start = $effStart;
                $payout->calc_end = $effEnd;
                $itemsToProcess[] = $payout;
            }

            foreach ($itemsToProcess as $item) {
                // Calculate Target Hours (Payable Hours)
                $calcStart = $item->calc_start; // Carbon object
                $calcEnd = $item->calc_end;
                $targetHours = 0;
                if ($calcEnd->greaterThan($calcStart)) {
                    $targetHours = $calcStart->diffInMinutes($calcEnd) / 60;
                }

                // Prepare Display Data
                // We always want to show BOTH Scheduled and Actual columns.
                // If it's a split part, this is tricky. We'll show the "Part" duration for the active mode.
                // For the inactive mode, we'll just show the Full duration (or --) because we can't easily split the "other" timeline 
                // perfectly aligned unless they are identical.
                // Simplification for display:
                $item->display_sched_in = $scheduledStart->format('Y-m-d H:i');
                $item->display_sched_out = $scheduledEnd->format('Y-m-d H:i');
                $item->display_sched_hours = $scheduledStart->diffInMinutes($scheduledEnd) / 60;

                $item->display_act_in = $actualStart->format('Y-m-d H:i');
                $item->display_act_out = $actualEnd->format('Y-m-d H:i');
                $item->display_act_hours = 0;
                if ($actualEnd->greaterThan($actualStart)) {
                    $item->display_act_hours = $actualStart->diffInMinutes($actualEnd) / 60;
                }

                // IF this is a split part, we update the "Active" mode's display columns to reflect the PART.
                // The "Inactive" mode's columns remain as the full original shift (context).
                if (!empty($item->is_split_part)) {
                    if ($reportType == 'scheduled') {
                        $item->display_sched_in = $calcStart->format('Y-m-d H:i');
                        $item->display_sched_out = $calcEnd->format('Y-m-d H:i');
                        $item->display_sched_hours = $targetHours;
                    } else {
                        $item->display_act_in = $calcStart->format('Y-m-d H:i');
                        $item->display_act_out = $calcEnd->format('Y-m-d H:i');
                        $item->display_act_hours = $targetHours;
                    }
                }

                // Attach View Properties for PDF Consistency
                if ($reportType == 'scheduled') {
                    $item->view_start = $item->display_sched_in;
                    $item->view_end = $item->display_sched_out;
                    $item->view_hours = $item->display_sched_hours;
                } else {
                    $item->view_start = $item->display_act_in;
                    $item->view_end = $item->display_act_out;
                    $item->view_hours = $item->display_act_hours;
                }

                // Wage Calculation Logic
                $wageTypesData = json_decode($item->wage_types ?? '[]', true);
                $hasWageTypes = !empty($wageTypesData) && is_array($wageTypesData);

                // Calculate Wage Rate and Amount
                $rate = ($item->wage_rate > 0) ? $item->wage_rate : ($item->fallback_rate ?? 0);
                $calculatedRate = 0;
                $sinRate = 0;
                $otherRate = 0;
                $displayTypes = [];
                $processedWageTypes = [];
                $amount = 0;

                if ($hasWageTypes) {
                    $empRates = $employeeRates[$item->employee_id] ?? collect([]);

                    // Calculate total allocated in DB for scaling
                    $dbAllocatedTotal = 0;
                    foreach ($wageTypesData as $wtInfo) {
                        if (is_array($wtInfo))
                            $dbAllocatedTotal += ($wtInfo['allocated_hours'] ?? 0);
                    }

                    // Sorting Logic: Priority for "sin".
                    usort($wageTypesData, function ($a, $b) {
                        $nameA = is_array($a) ? strtolower($a['name'] ?? '') : '';
                        $nameB = is_array($b) ? strtolower($b['name'] ?? '') : '';
                        $isSinA = strpos($nameA, 'sin') !== false;
                        $isSinB = strpos($nameB, 'sin') !== false;
                        if ($isSinA && !$isSinB)
                            return -1;
                        if (!$isSinA && $isSinB)
                            return 1;
                        return 0;
                    });

                    $remainingHours = $targetHours;

                    foreach ($wageTypesData as $wtInfo) {
                        $idToLookup = null;
                        $allocatedHours = 0;

                        if (is_array($wtInfo)) {
                            $idToLookup = $wtInfo['id'] ?? null;
                            $dbAllocated = $wtInfo['allocated_hours'] ?? 0;

                            if ($reportType == 'scheduled') {
                                // Universal Scaling for Scheduled
                                if ($dbAllocatedTotal > 0) {
                                    $allocatedHours = ($dbAllocated / $dbAllocatedTotal) * $targetHours;
                                } else {
                                    $allocatedHours = $dbAllocated;
                                }
                            } else {
                                // Waterfall for Actuals
                                $allocatedHours = min($remainingHours, $dbAllocated);
                                $remainingHours -= $allocatedHours;
                                if ($remainingHours < 0)
                                    $remainingHours = 0;
                            }
                        } elseif (is_scalar($wtInfo)) {
                            $idToLookup = $wtInfo;
                            $allocatedHours = $targetHours;
                            $remainingHours = 0;
                        }

                        if ($idToLookup) {
                            $record = $empRates->get($idToLookup);

                            // Get rate and name from pivot if available
                            $pivotRate = is_array($wtInfo) ? ($wtInfo['rate'] ?? null) : null;
                            $pivotName = is_array($wtInfo) ? ($wtInfo['name'] ?? null) : null;
                            $profileRate = $record ? $record->rate : null;
                            $profileName = $record ? $record->type_name : null;

                            // Always prioritize pivot rate for Actuals as per user request
                            // For Scheduled, maybe keep profile fallback if pivot is empty
                            if ($reportType == 'scheduled') {
                                $effectiveRate = $profileRate ?? $pivotRate ?? 0;
                            } else {
                                $effectiveRate = $pivotRate ?? $profileRate ?? 0;
                            }

                            $effectiveName = $pivotName ?? $profileName ?? 'Unknown';

                            if ($effectiveRate > 0 || $record) {
                                $calculatedRate += $effectiveRate;
                                $displayTypes[] = $effectiveName . ' ($' . number_format($effectiveRate, 2) . ')';
                                if (stripos($effectiveName, 'sin') !== false)
                                    $sinRate += $effectiveRate;
                                else
                                    $otherRate += $effectiveRate;

                                $amount += ($allocatedHours * $effectiveRate);

                                $processedWageTypes[] = [
                                    'id' => $idToLookup,
                                    'name' => $effectiveName,
                                    'rate' => $effectiveRate,
                                    'allocated_hours' => $allocatedHours
                                ];
                            }
                        }
                    }
                } else {
                    $amount = $targetHours * $rate;
                }

                // Add GAS/CASH to total amount (only once per shift if split)
                if ($item->gas_rate > 0 && ($item->calc_start == $effStart)) {
                    $amount += $item->gas_rate;
                }

                $item->wage_type_display = !empty($displayTypes) ? implode(' + ', $displayTypes) : 'sin';
                $multiplier = $item->stat_multiplier ?? 1;

                if ($calculatedRate > 0) {
                    // $rate = $calculatedRate * $multiplier; // Rate is just for display if mixed
                }

                // Store processed data
                $item->wage_types = json_encode($processedWageTypes);
                $item->hours = $targetHours; // Paid Hours
                $item->total_amount = $amount;

                if (!isset($groupedPayouts[$empKey])) {
                    $groupedPayouts[$empKey] = [
                        'items' => [],
                        'total_hours' => 0,
                        'total_amount' => 0
                    ];
                }

                $groupedPayouts[$empKey]['items'][] = $item;
                $groupedPayouts[$empKey]['total_hours'] += $targetHours;
                $groupedPayouts[$empKey]['total_amount'] += $amount;
            }
        }

        return ['payouts' => $payouts, 'groupedPayouts' => $groupedPayouts];
    }

    public function consolidatedReport(Request $request)
    {
        $data = $this->getConsolidatedReportData($request);
        $groupedConsolidated = $data['groupedConsolidated'];
        $startDate = $data['startDate'];
        $endDate = $data['endDate'];

        $company_id = session('selected_company_id');
        $employees = Employee::where('company_id', $company_id)->orderBy('first_name')->get();

        return view('reports.operational.consolidated', compact('groupedConsolidated', 'employees', 'startDate', 'endDate'));
    }

    public function exportConsolidatedPdf(Request $request)
    {
        $data = $this->getConsolidatedReportData($request);
        $groupedConsolidated = $data['groupedConsolidated'];
        $startDate = $data['startDate'];
        $endDate = $data['endDate'];

        $company_id = session('selected_company_id');
        $company = \App\Models\Company::find($company_id);

        $pdf = Pdf::loadView('reports.operational.consolidated_pdf', compact('groupedConsolidated', 'startDate', 'endDate', 'company'))
            ->setPaper('a4', 'landscape');

        return $pdf->download('Consolidated_Report_' . now()->format('Ymd_His') . '.pdf');
    }

    public function exportConsolidatedExcel(Request $request)
    {
        $data = $this->getConsolidatedReportData($request);
        $groupedConsolidated = $data['groupedConsolidated'];

        $headers = ['Site/Route', 'Role', 'Duty #', 'Start', 'End', 'Type', 'Hrs', 'SIN Rate', 'Other Rate', 'Gas', 'Allow', 'Tkts', 'Total'];
        $excelData = [];

        foreach ($groupedConsolidated as $empHeader => $items) {
            // Employee Header Row
            $excelData[] = [$empHeader, '', '', '', '', '', '', '', '', '', '', '', ''];

            $totalHrs = 0;
            $totalGas = 0;
            $totalAllowance = 0;
            $totalTickets = 0;
            $totalAmount = 0;

            foreach ($items as $row) {
                $totalHrs += $row->actual_hrs;
                $totalGas += $row->gas;
                $totalAllowance += $row->allowance;
                $totalTickets += $row->tickets_count;
                $totalAmount += $row->total_amount;

                $excelData[] = [
                    $row->site_route,
                    ucfirst($row->role),
                    $row->duty_number,
                    $row->start_time ? $row->start_time->format('m-d H:i') : '-',
                    $row->end_time ? $row->end_time->format('m-d H:i') : '-',
                    strtoupper($row->pay_type),
                    number_format($row->actual_hrs, 2),
                    number_format($row->sin_rate, 2),
                    number_format($row->other_rate, 2),
                    number_format($row->gas, 0),
                    number_format($row->allowance, 0),
                    $row->tickets_count,
                    '$' . number_format($row->total_amount, 2)
                ];
            }

            // Subtotal Row
            $excelData[] = [
                '  Grand Total',
                '',
                '',
                '',
                '',
                '',
                number_format($totalHrs, 2),
                '-',
                '-',
                number_format($totalGas, 0),
                number_format($totalAllowance, 0),
                $totalTickets,
                '$' . number_format($totalAmount, 2)
            ];

            // Spacer
            $excelData[] = ['', '', '', '', '', '', '', '', '', '', '', '', ''];
        }

        $excelService = new \App\Services\ExcelExportService();
        return $excelService->generateReport(
            'Consolidated Operational Report',
            'Summary Payouts',
            $headers,
            $excelData,
            [
                'filename_prefix' => 'Consolidated_Report',
                'center_columns' => [2, 3, 4, 11],
            ]
        );
    }

    private function getConsolidatedReportData(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        // Fetch Guards Data
        $guardData = $this->getGuardPayoutData($company_id, $request);
        $guardsGrouped = $guardData['groupedPayouts'];

        // Fetch Patrollers Data
        $patrollerData = $this->getPatrollerPayoutData($company_id, $request);
        $patrollersGrouped = $patrollerData['groupedPayouts'];

        // Helper to process items
        $processItems = function ($group, $role) {
            $processed = [];
            foreach ($group['items'] ?? [] as $item) {
                $wageTypes = json_decode($item->wage_types ?? '[]', true);

                $sinRate = 0;
                $otherRate = 0;

                if (!empty($wageTypes)) {
                    foreach ($wageTypes as $wt) {
                        $name = strtolower($wt['name'] ?? '');
                        $rate = $wt['rate'] ?? 0;
                        if (str_contains($name, 'sin')) {
                            $sinRate += $rate;
                        } else {
                            $otherRate += $rate;
                        }
                    }
                } else {
                    if ($item->wage_type_display === 'sin') {
                        $sinRate = $item->wage_rate;
                    } else {
                        $sinRate = $item->wage_rate;
                    }
                }

                if ($sinRate == 0 && $item->wage_rate > 0 && $otherRate == 0) {
                    $sinRate = $item->wage_rate;
                }

                $processed[] = (object) [
                    'site_route' => $item->site_name ?? ($item->route_name ?? 'N/A'),
                    'role' => $role,
                    'duty_number' => $item->duty_number,
                    'start_time' => $item->actual_start_at ? Carbon::parse($item->actual_start_at) : null,
                    'end_time' => $item->actual_end_at ? Carbon::parse($item->actual_end_at) : null,
                    'pay_type' => $item->wage_type_display ?? 'sin',
                    'actual_hrs' => $item->hours,
                    'sin_rate' => $sinRate,
                    'other_rate' => $otherRate,
                    'gas' => $item->gas_rate,
                    'allowance' => $item->allowance ?? 0,
                    'tickets_count' => $item->tickets_count ?? 0,
                    'total_amount' => $item->total_amount,
                    'employee_name' => $item->first_name . ' ' . $item->last_name,
                    'employee_email' => $item->email ?? '',
                    'employee_id' => $item->employee_id
                ];
            }
            return $processed;
        };

        $mergedItems = [];

        foreach ($guardsGrouped as $key => $group) {
            $mergedItems = array_merge($mergedItems, $processItems($group, 'guard'));
        }
        foreach ($patrollersGrouped as $key => $group) {
            $mergedItems = array_merge($mergedItems, $processItems($group, 'patroller'));
        }

        usort($mergedItems, function ($a, $b) {
            if (!$a->start_time)
                return 1;
            if (!$b->start_time)
                return -1;
            return $a->start_time <=> $b->start_time;
        });

        $groupedConsolidated = collect($mergedItems)->groupBy(function ($item) {
            return "{$item->employee_name} (FE{$item->employee_id})";
        });

        return [
            'groupedConsolidated' => $groupedConsolidated,
            'startDate' => $startDate,
            'endDate' => $endDate
        ];
    }

    private function getPatrollerPayoutData($company_id, Request $request)
    {
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->where('patroller_schedules.status', '!=', 'cancelled')
            ->whereNotNull('employee_patroller_schedule.actual_start_at')
            ->whereNotNull('employee_patroller_schedule.actual_end_at')
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate);

        if ($request->filled('employee_id')) {
            $query->where('employee_patroller_schedule.employee_id', $request->employee_id);
        }

        $payouts = $query->select(
            'patroller_schedules.scheduled_date',
            'patroller_schedules.duty_number',
            'patroller_schedules.from_time as from_datetime',
            'patroller_schedules.to_time as to_datetime',
            'routes.name as route_name',
            'employees.id as employee_id',
            'employees.first_name',
            'employees.last_name',
            'employees.email', // Added email for grouping
            'employee_patroller_schedule.actual_start_at',
            'employee_patroller_schedule.actual_end_at',
            'employee_patroller_schedule.wage_types',
            'employee_patroller_schedule.wage_rate',
            'employee_patroller_schedule.gas_rate',
            'employee_patroller_schedule.allowance',
            'employee_patroller_schedule.other_expense',
            // Ticket count subquery
            DB::raw("(SELECT COUNT(*) FROM patroller_issue_tickets WHERE patroller_issue_tickets.patroller_schedule_id = patroller_schedules.id) as tickets_count")
        )
            ->orderBy('employees.first_name')
            ->orderBy('patroller_schedules.scheduled_date')
            ->get();

        // fetch wage rates lookup if needed (omitted for brevity, assuming generic logic similar to guards)

        $groupedPayouts = [];

        // Similar processing loop as getGuardPayoutData 
        // We'll simplify and reuse the logic structure

        foreach ($payouts as $payout) {
            $empKey = "EMP{$payout->employee_id}"; // Simple key for internal grouping

            $start = Carbon::parse($payout->actual_start_at);
            $end = Carbon::parse($payout->actual_end_at);
            $hours = 0;
            if ($end->greaterThan($start)) {
                $hours = $start->diffInMinutes($end) / 60;
            }

            // Wage calculation
            $wageTypesData = json_decode($payout->wage_types ?? '[]', true);
            $rate = $payout->wage_rate;
            $amount = 0;
            $displayTypes = [];

            // Basic logic: if wage types exist, sum amounts, else rate * hours
            $processedWageTypes = [];

            if (!empty($wageTypesData) && is_array($wageTypesData)) {
                foreach ($wageTypesData as $wt) {
                    // Assuming simple structure for patroller pivot wage types if it matches guard pivot
                    // OR it might be just IDs.
                    // The migration created generic JSON field.
                    // Let's assume it matches Guard structure for now if populated.
                    // If just ID list, we'd need IDs lookup.
                    // If array of objects {id, name, rate, hours}:
                    if (is_array($wt)) {
                        $wHours = $wt['allocated_hours'] ?? $hours;
                        $wRate = $wt['rate'] ?? 0;
                        $amount += ($wHours * $wRate);
                        $displayTypes[] = ($wt['name'] ?? 'Type') . " ($$wRate)";

                        $processedWageTypes[] = $wt;
                    }
                }
            } else {
                $amount = $hours * $rate;
                $processedWageTypes[] = ['name' => 'Standard', 'rate' => $rate, 'allocated_hours' => $hours];
            }

            // Add expenses
            $amount += ($payout->gas_rate ?? 0);
            $amount += ($payout->allowance ?? 0);
            $amount += ($payout->other_expense ?? 0);

            $payout->hours = $hours;
            $payout->total_amount = $amount;
            $payout->wage_type_display = !empty($displayTypes) ? implode(',', $displayTypes) : 'sin';
            $payout->wage_types = json_encode($processedWageTypes); // Normalized

            if (!isset($groupedPayouts[$empKey])) {
                $groupedPayouts[$empKey] = ['items' => []];
            }
            $groupedPayouts[$empKey]['items'][] = $payout;
        }

        return ['groupedPayouts' => $groupedPayouts];
        $data = $this->getIncidentCountsData(session('selected_company_id'), $request);
        $reportData = $data['reportData'];
        $days = $data['days'];
        $month = $data['month'];
        $year = $data['year'];

        $monthName = \Carbon\Carbon::create()->month($month)->format('F');

        $pdf = Pdf::loadView('admin.reports.operational.incident_counts_pdf', compact('reportData', 'days', 'monthName', 'year'))
            ->setPaper('a4', 'landscape');

        return $pdf->download("Incident_Counts_{$monthName}_{$year}.pdf");
    }

    public function exportIncidentCountsExcel(Request $request)
    {
        $data = $this->getIncidentCountsData(session('selected_company_id'), $request);
        $reportData = $data['reportData'];
        $days = $data['days'];
        $month = $data['month'];
        $year = $data['year'];

        $monthName = \Carbon\Carbon::create()->month($month)->format('F');
        $filename = "Incident_Counts_{$monthName}_{$year}.csv";

        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];

        // Build Columns: Site, 1, 2, ..., Days, Total
        $columns = ['Site'];
        for ($d = 1; $d <= $days; $d++)
            $columns[] = $d;
        $columns[] = 'Total';

        $callback = function () use ($reportData, $columns, $days) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            // Totals Row Accumulator
            $colTotals = array_fill(1, $days, 0);
            $grandTotal = 0;

            foreach ($reportData as $row) {
                $csvRow = [$row['site']];
                foreach ($row['days'] as $d => $count) {
                    $csvRow[] = $count;
                    $colTotals[$d] += $count;
                }
                $csvRow[] = $row['total'];
                $grandTotal += $row['total'];
                fputcsv($file, $csvRow);
            }

            // Totals Row
            $totalsRow = ['Totals'];
            for ($d = 1; $d <= $days; $d++)
                $totalsRow[] = $colTotals[$d];
            $totalsRow[] = $grandTotal;
            fputcsv($file, $totalsRow);

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function getIncidentCountsData($company_id, Request $request)
    {
        $now = Carbon::now();

        $month = $request->get('month', $now->month);
        $year = $request->get('year', $now->year);

        $startDate = Carbon::createFromDate($year, $month, 1)->startOfDay();

        if ($month == $now->month && $year == $now->year) {
            $endDate = $now->copy()->endOfDay();
        } else {
            $endDate = $startDate->copy()->endOfMonth()->endOfDay();
        }

        $days = $endDate->day;

        $query = Site::where('company_id', $company_id)->orderBy('name');
        if ($request->filled('site_id')) {
            $query->where('id', $request->site_id);
        }
        $sites = $query->get();

        $stats = DB::table('incidents')
            ->join('schedules', 'incidents.schedule_id', '=', 'schedules.id')
            ->where('schedules.company_id', $company_id)
            ->whereBetween('incidents.created_at', [$startDate, $endDate])
            ->select(
                'schedules.site_id',
                DB::raw('DAY(incidents.created_at) as day'),
                DB::raw('count(*) as count')
            )
            ->groupBy('schedules.site_id', 'day')
            ->get();

        $reportData = [];
        $statsMap = [];
        foreach ($stats as $stat) {
            $statsMap[$stat->site_id][$stat->day] = $stat->count;
        }

        foreach ($sites as $site) {
            $row = ['site' => $site->name, 'site_id' => $site->id, 'days' => [], 'total' => 0];
            for ($d = 1; $d <= $days; $d++) {
                $count = $statsMap[$site->id][$d] ?? 0;
                $row['days'][$d] = $count;
                $row['total'] += $count;
            }
            $reportData[] = $row;
        }

        return [
            'reportData' => $reportData,
            'days' => $days,
            'month' => $month,
            'year' => $year,
            'sites' => $sites
        ];
    }

    private function getPayoutsQuery($company_id, Request $request)
    {
        $query = DB::table('employee_schedule')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->leftJoin('stat_holidays', function ($join) use ($company_id) {
                $join->on(DB::raw('DATE(schedules.schedule_date)'), '=', 'stat_holidays.holiday_date')
                    ->where('stat_holidays.company_id', '=', $company_id)
                    ->where('stat_holidays.active', '=', true);
            })
            ->where('schedules.company_id', $company_id)
            ->where('schedules.status', 'completed')
            ->select(
                'employee_schedule.id',
                'schedules.schedule_date',
                'schedules.duty_number',
                'schedules.from_datetime',
                'schedules.to_datetime',
                'sites.name as site_name',
                'sites.site_id as site_custom_id',
                'employees.first_name',
                'employees.last_name',
                'employee_schedule.actual_start_at',
                'employee_schedule.actual_end_at',
                'employee_schedule.wage_types',
                'employee_schedule.employee_id',
                'schedules.site_rate as schedule_site_rate',
                DB::raw("employee_schedule.wage_rate * COALESCE(stat_holidays.multiplier, 1) as wage_rate"),
                DB::raw("employee_schedule.gas_rate * COALESCE(stat_holidays.multiplier, 1) as gas_rate"),
                DB::raw("COALESCE(NULLIF(employee_schedule.bill_rate, 0), sites.rate) as bill_rate"),
                DB::raw("(SELECT rate FROM employee_wage_type WHERE employee_wage_type.employee_id = employees.id ORDER BY rate DESC LIMIT 1) * COALESCE(stat_holidays.multiplier, 1) as fallback_rate"),
                DB::raw("COALESCE(stat_holidays.multiplier, 1) as stat_multiplier"),
                'stat_holidays.name as stat_holiday_name'
            );

        if ($request->filled('start_date')) {
            $query->whereDate('schedules.schedule_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('schedules.schedule_date', '<=', $request->end_date);
        }
        if ($request->filled('site_id')) {
            $query->where('schedules.site_id', $request->site_id);
        }
        if ($request->filled('employee_id')) {
            $query->where('employee_schedule.employee_id', $request->employee_id);
        }
        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_schedule.id', $ids);
        }

        return $query;
    }

    public function payouts(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->getPayoutsQuery($company_id, $request);
        $payouts = $query->orderBy('schedules.schedule_date', 'desc')->paginate(20);

        $calculationMode = $request->query('calculation_mode', 'actual');
        $this->calculateWageRates($payouts->items(), $calculationMode);

        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();
        $employees = Employee::where('company_id', $company_id)->orderBy('first_name')->get();

        return view('reports.operational.payouts', compact('payouts', 'sites', 'employees', 'calculationMode'));
    }

    public function exportPayoutsPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->getPayoutsQuery($company_id, $request);
        $payouts = $query->orderBy('schedules.schedule_date', 'desc')->get();

        $calculationMode = $request->query('calculation_mode', 'actual');
        $this->calculateWageRates($payouts, $calculationMode);

        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $company = \App\Models\Company::find($company_id);

        $pdf = Pdf::loadView('admin.reports.operational.payouts_pdf', compact('payouts', 'startDate', 'endDate', 'calculationMode', 'company'))
            ->setPaper('a4', 'landscape');

        return $pdf->download('Payout_Summary_' . now()->format('Ymd_Hi') . '.pdf');
    }


    public function exportPayoutsExcel(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->getPayoutsQuery($company_id, $request);
        $payouts = $query->orderBy('schedules.schedule_date', 'desc')->get();

        $calculationMode = $request->query('calculation_mode', 'actual');
        $this->calculateWageRates($payouts, $calculationMode);

        $headers = ['Date', 'Duty #', 'Employee', 'Site', 'Site ID', 'In Time', 'Out Time', 'Hours', 'Wage Details', 'Total'];
        $data = [];

        foreach ($payouts as $row) {
            $viewStart = $row->view_start ? Carbon::parse($row->view_start) : null;
            $viewEnd = $row->view_end ? Carbon::parse($row->view_end) : null;
            $viewHours = $row->view_hours ?? 0;

            // Build wage components string
            $wageTypes = json_decode($row->wage_types ?? '[]', true);
            $wageComponentsStr = '';

            if (!empty($wageTypes)) {
                $components = [];
                foreach ($wageTypes as $wageType) {
                    $componentHours = $wageType['allocated_hours'] ?? 0;
                    $componentRate = $wageType['rate'] ?? 0;
                    $componentPay = $componentHours * $componentRate;
                    $components[] = "{$wageType['name']}: " . number_format($componentHours, 2) . "hrs × \${$componentRate} = \${$componentPay}";
                }
                $wageComponentsStr = implode(' | ', $components);
            }

            $gasRate = $row->gas_rate ?? 0;
            if ($gasRate > 0) {
                $wageComponentsStr .= ($wageComponentsStr ? ' | ' : '') . "GAS/CASH: \${$gasRate}";
            }

            $data[] = [
                Carbon::parse($row->schedule_date)->format('Y-m-d'),
                $row->duty_number,
                $row->first_name . ' ' . $row->last_name . (isset($row->stat_multiplier) && $row->stat_multiplier > 1 ? ' (STAT)' : ''),
                $row->site_name,
                $row->site_custom_id,
                $viewStart ? $viewStart->format('Y-m-d H:i') : '--',
                $viewEnd ? $viewEnd->format('Y-m-d H:i') : '--',
                number_format($viewHours, 2),
                $wageComponentsStr ?: 'No wage data',
                number_format($row->total_payout, 2)
            ];
        }

        $excelService = new \App\Services\ExcelExportService();
        return $excelService->generateReport(
            'Payout Summary Report',
            'Summary',
            $headers,
            $data,
            [
                'filename_prefix' => 'Payout_Summary',
                'center_columns' => [0, 1, 4, 5, 6, 7],
            ]
        );
    }

    public function incidents(Request $request)
    {
        $company_id = session('selected_company_id');

        $query = $this->buildIncidentsQuery($company_id, $request);
        $incidents = $query->orderBy('incidents.created_at', 'desc')->paginate(20);

        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();
        $employees = Employee::where('company_id', $company_id)->orderBy('first_name')->get();

        return view('reports.operational.incidents', compact('incidents', 'sites', 'employees'));
    }

    public function sitePayouts(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getSitePayoutData($company_id, $request);
        $groupedPayouts = $data['groupedPayouts'];

        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();

        return view('reports.operational.site_payouts', compact('groupedPayouts', 'sites'));
    }

    public function exportSitePayoutsPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getSitePayoutData($company_id, $request);
        $groupedPayouts = $data['groupedPayouts'];
        $reportType = $request->get('report_type', 'actual');
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $company = \App\Models\Company::find($company_id);

        $pdf = Pdf::loadView('admin.reports.operational.site_payouts_pdf', compact('groupedPayouts', 'startDate', 'endDate', 'reportType', 'company'))
            ->setPaper('a4', 'landscape');

        return $pdf->download("Site_Payouts_{$reportType}_{$startDate}_{$endDate}.pdf");
    }

    public function exportSitePayoutsExcel(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getSitePayoutData($company_id, $request);
        $groupedPayouts = $data['groupedPayouts'];
        $reportType = $request->get('report_type', 'actual');
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $filename = "Site_Payouts_{$reportType}_{$startDate}_{$endDate}.xlsx";
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Headers
        $headers = ['Site', 'Employee', 'Duty Number', 'From Date', 'From Time', 'To Date', 'To Time', 'Total Hrs', 'Site Rate', 'Total Amount'];
        $sheet->fromArray($headers, NULL, 'A1');

        // Style Headers
        $headerStyle = [
            'font' => [
                'bold' => true,
                'color' => ['argb' => 'FFFFFFFF'],
            ],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => ['argb' => 'FF1E293B'], // Slate-800
            ],
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                ],
            ],
        ];
        $sheet->getStyle('A1:J1')->applyFromArray($headerStyle);

        $rowNumber = 2;
        foreach ($groupedPayouts as $siteName => $group) {
            foreach ($group['items'] as $row) {
                $sheet->setCellValue('A' . $rowNumber, $siteName);
                $sheet->setCellValue('B' . $rowNumber, $row->first_name . ' ' . $row->last_name);
                $sheet->setCellValue('C' . $rowNumber, $row->duty_number);
                $sheet->setCellValue('D' . $rowNumber, Carbon::parse($row->from_datetime)->format('Y-m-d'));
                $sheet->setCellValue('E' . $rowNumber, Carbon::parse($row->from_datetime)->format('H:i'));
                $sheet->setCellValue('F' . $rowNumber, Carbon::parse($row->to_datetime)->format('Y-m-d'));
                $sheet->setCellValue('G' . $rowNumber, Carbon::parse($row->to_datetime)->format('H:i'));
                $sheet->setCellValue('H' . $rowNumber, number_format($row->hours, 2));
                $sheet->setCellValue('I' . $rowNumber, number_format($row->site_rate, 2));
                $sheet->setCellValue('J' . $rowNumber, number_format($row->total_amount, 2));
                $rowNumber++;
            }

            // Subtotal Row
            $sheet->setCellValue('A' . $rowNumber, $siteName . ' Total');
            $sheet->mergeCells("A{$rowNumber}:G{$rowNumber}");

            $sheet->setCellValue('H' . $rowNumber, number_format($group['total_hours'], 2));
            $sheet->setCellValue('J' . $rowNumber, number_format($group['total_amount'], 2));

            // Style Subtotal
            $sheet->getStyle("A{$rowNumber}:J{$rowNumber}")->applyFromArray([
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['argb' => 'FFF3F4F6'],
                ],
                'font' => [
                    'bold' => true,
                ],
            ]);
            $sheet->getStyle("A{$rowNumber}")->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_RIGHT);

            $rowNumber++;
            $rowNumber++; // Extra space
        }

        // Apply Borders to all data
        if ($rowNumber > 2) {
            $lastRow = $rowNumber - 2;
            $sheet->getStyle('A2:J' . $lastRow)->getBorders()->getAllBorders()->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN);
        }

        // Auto Size Columns
        foreach (range('A', 'J') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);

        return response()->streamDownload(function () use ($writer) {
            $writer->save('php://output');
        }, $filename);
    }

    private function getSitePayoutData($company_id, Request $request)
    {
        $query = $this->getPayoutsQuery($company_id, $request);

        $payouts = $query->orderBy('sites.name')
            ->orderBy('schedules.schedule_date')
            ->get();

        $reportType = $request->get('report_type', 'actual'); // 'actual' or 'scheduled'
        $groupedPayouts = [];

        foreach ($payouts as $payout) {
            $siteKey = $payout->site_name;

            if ($reportType == 'scheduled') {
                // Use scheduled hours (from_datetime to to_datetime)
                $scheduledStart = Carbon::parse($payout->from_datetime);
                $scheduledEnd = Carbon::parse($payout->to_datetime);
                $hours = $scheduledStart->diffInMinutes($scheduledEnd) / 60;

                $payout->view_start = $scheduledStart;
                $payout->view_end = $scheduledEnd;
            } else {
                // Use actual hours (Raw, No clamping)
                $scheduledStart = Carbon::parse($payout->from_datetime);
                $actualStart = Carbon::parse($payout->actual_start_at);
                $effectiveStart = $actualStart;

                $end = Carbon::parse($payout->actual_end_at);
                if ($effectiveStart->greaterThan($end))
                    $effectiveStart = $end;

                $hours = $effectiveStart->diffInMinutes($end) / 60;

                $payout->view_start = $effectiveStart;
                $payout->view_end = $end;
            }

            // Use the split bill rate from the pivot table (per guard portion of site rate)
            $siteRate = $payout->bill_rate ?? 0;
            $amount = $hours * $siteRate;

            $payout->hours = $hours;
            $payout->site_rate = $siteRate;
            $payout->total_amount = $amount;

            if (!isset($groupedPayouts[$siteKey])) {
                $groupedPayouts[$siteKey] = [
                    'items' => [],
                    'total_hours' => 0,
                    'total_amount' => 0
                ];
            }

            $groupedPayouts[$siteKey]['items'][] = $payout;
            $groupedPayouts[$siteKey]['total_hours'] += $hours;
            $groupedPayouts[$siteKey]['total_amount'] += $amount;
        }

        return ['payouts' => $payouts, 'groupedPayouts' => $groupedPayouts];
    }

    public function siteMargin(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getSiteMarginData($company_id, $request);
        $groupedMargins = $data['groupedMargins'];

        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();
        $employees = Employee::where('company_id', $company_id)
            ->orderBy('first_name')
            ->orderBy('last_name')
            ->get();

        return view('reports.operational.site_margin', compact('groupedMargins', 'sites', 'employees'));
    }

    public function exportSiteMarginPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getSiteMarginData($company_id, $request);
        $groupedMargins = $data['groupedMargins'];

        $reportType = $request->get('report_type', 'actual');
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $company = \App\Models\Company::find($company_id);

        $pdf = Pdf::loadView('admin.reports.operational.site_margin_pdf', compact('groupedMargins', 'startDate', 'endDate', 'reportType', 'company'))
            ->setPaper('a4', 'landscape');

        return $pdf->download("Site_Margin_{$reportType}_{$startDate}_{$endDate}.pdf");
    }

    public function exportSiteMarginExcel(Request $request)
    {
        $company_id = session('selected_company_id');
        $data = $this->getSiteMarginData($company_id, $request);
        $groupedMargins = $data['groupedMargins'];

        $headers = ['Duty ID', 'Guard Name', 'In Date', 'In Time', 'Out Date', 'Out Time', 'Total Hours', 'Site Rate', 'Income', 'Wage Components', 'Expense', 'Margin'];
        $excelData = [];

        foreach ($groupedMargins as $siteName => $group) {
            // Site Header Row
            $excelData[] = [$siteName, '', '', '', '', '', '', '', '', '', '', ''];

            foreach ($group['items'] as $row) {
                // Build wage components string
                $wageTypes = json_decode($row->wage_types ?? '[]', true);
                $wageComponentsStr = '';
                $totalExpense = 0;

                if (!empty($wageTypes)) {
                    $components = [];
                    foreach ($wageTypes as $wageType) {
                        $hours = $wageType['allocated_hours'] ?? 0;
                        $rate = $wageType['rate'] ?? 0;
                        $amount = $hours * $rate;
                        $totalExpense += $amount;
                        $components[] = "{$wageType['name']}: " . number_format($hours, 2) . "hrs × $" . number_format($rate, 2) . " = $" . number_format($amount, 2);
                    }
                    $wageComponentsStr = implode("\n", $components);
                }

                $gasRate = $row->gas_rate ?? 0;
                if ($gasRate > 0) {
                    $wageComponentsStr .= ($wageComponentsStr ? "\n" : "") . "GAS/CASH: $" . number_format($gasRate, 2);
                    $totalExpense += $gasRate;
                }

                $margin = $row->total_income - $totalExpense;

                $excelData[] = [
                    $row->duty_number . (isset($row->stat_multiplier) && $row->stat_multiplier > 1 ? ' (STAT)' : ''),
                    $row->first_name . ' ' . $row->last_name,
                    Carbon::parse($row->view_start)->format('Y-m-d'),
                    Carbon::parse($row->view_start)->format('h:i A'),
                    Carbon::parse($row->view_end)->format('Y-m-d'),
                    Carbon::parse($row->view_end)->format('h:i A'),
                    number_format($row->hours, 2),
                    '$' . number_format($row->bill_rate, 2),
                    '$' . number_format($row->total_income, 2),
                    $wageComponentsStr ?: 'No wage data',
                    '$' . number_format($totalExpense, 2),
                    '$' . number_format($margin, 2)
                ];
            }

            // Subtotal Row
            $totalMargin = $group['total_income'] - $group['total_expense'];
            $excelData[] = [
                '  Total',
                '',
                '',
                '',
                '',
                '',
                number_format($group['total_hours'], 2),
                '',
                '$' . number_format($group['total_income'], 2),
                '',
                '$' . number_format($group['total_expense'], 2),
                '$' . number_format($totalMargin, 2)
            ];

            // Spacer
            $excelData[] = ['', '', '', '', '', '', '', '', '', '', '', ''];
        }

        $excelService = new \App\Services\ExcelExportService();
        return $excelService->generateReport(
            'Site Margin Report',
            ucfirst($request->get('report_type', 'actual')) . ' Hours',
            $headers,
            $excelData,
            [
                'filename_prefix' => 'Site_Margin',
                'center_columns' => [2, 3, 4, 5],
            ]
        );
    }

    private function getSiteMarginData($company_id, Request $request)
    {
        $query = $this->getPayoutsQuery($company_id, $request);
        $payouts = $query->orderBy('sites.name')
            ->orderBy('schedules.schedule_date')
            ->get();

        // Process wage rates for all payouts
        $reportType = $request->get('report_type', 'actual');
        $this->calculateWageRates($payouts, $reportType);

        $groupedMargins = [];

        foreach ($payouts as $payout) {
            $siteKey = $payout->site_name;

            if ($reportType == 'scheduled') {
                $start = Carbon::parse($payout->from_datetime);
                $end = Carbon::parse($payout->to_datetime);
                $hours = $start->diffInMinutes($end) / 60;

                $payout->view_start = $start;
                $payout->view_end = $end;
            } else {
                // Raw Actuals
                $start = Carbon::parse($payout->actual_start_at);
                $end = Carbon::parse($payout->actual_end_at);
                if ($start->greaterThan($end))
                    $start = $end; // Sanity
                $hours = $start->diffInMinutes($end) / 60;

                $payout->view_start = $start;
                $payout->view_end = $end;
            }
            $payout->view_hours = $hours;

            // Calculate Expense using wage components
            // The wage_types have been processed by calculateWageRates
            $wageTypes = json_decode($payout->wage_types ?? '[]', true);
            $wageExpense = 0;

            if (!empty($wageTypes) && is_array($wageTypes)) {
                foreach ($wageTypes as $wageType) {
                    $componentHours = $wageType['allocated_hours'] ?? 0;
                    $componentRate = $wageType['rate'] ?? 0;
                    $wageExpense += $componentHours * $componentRate;
                }
            }

            $gasExpense = $payout->gas_rate ?? 0;
            $totalExpense = $wageExpense + $gasExpense;

            // Calculate Income - use the split bill rate from pivot (consistent with Site Payouts)
            $billRate = $payout->bill_rate ?? 0;
            $payout->bill_rate = $billRate; // Update for view display
            $totalIncome = $hours * $billRate;

            $payout->hours = $hours;
            $payout->total_income = $totalIncome;
            $payout->total_expense = $totalExpense;

            if (!isset($groupedMargins[$siteKey])) {
                $groupedMargins[$siteKey] = [
                    'items' => [],
                    'total_hours' => 0,
                    'total_income' => 0,
                    'total_expense' => 0
                ];
            }

            $groupedMargins[$siteKey]['items'][] = $payout;
            $groupedMargins[$siteKey]['total_hours'] += $hours;
            $groupedMargins[$siteKey]['total_income'] += $totalIncome;
            $groupedMargins[$siteKey]['total_expense'] += $totalExpense;
        }

        return ['payouts' => $payouts, 'groupedMargins' => $groupedMargins];
    }

    public function exportIncidentsPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->buildIncidentsQuery($company_id, $request);
        $incidents = $query->orderBy('incidents.created_at', 'desc')->get();

        $company = \App\Models\Company::find($company_id);

        // Create a dedicated PDF view or reuse a generic one
        $pdf = Pdf::loadView('admin.reports.operational.incidents_pdf', compact('incidents', 'company'));
        return $pdf->download('Incidents_Report_' . now()->format('Ymd_Hi') . '.pdf');
    }

    public function exportIncidentsExcel(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->buildIncidentsQuery($company_id, $request);
        $incidents = $query->orderBy('incidents.created_at', 'desc')->get();

        $filename = "Incidents_Report_" . now()->format('Ymd_Hi') . ".csv";
        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];
        $columns = ['Date', 'Duty #', 'Employee', 'Site', 'Subject', 'Description', 'Created At'];

        $callback = function () use ($incidents, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($incidents as $row) {
                fputcsv($file, [
                    Carbon::parse($row->created_at)->format('Y-m-d H:i'),
                    $row->duty_number,
                    $row->first_name . ' ' . $row->last_name,
                    $row->site_name,
                    $row->subject,
                    $row->description,
                    $row->created_at
                ]);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function buildIncidentsQuery($company_id, Request $request)
    {
        $query = DB::table('incidents')
            ->join('schedules', 'incidents.schedule_id', '=', 'schedules.id')
            ->join('employees', 'incidents.employee_id', '=', 'employees.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->where('schedules.company_id', $company_id)
            ->select(
                'incidents.*',
                'schedules.schedule_date',
                'schedules.duty_number',
                'sites.name as site_name',
                'employees.first_name',
                'employees.last_name'
            );

        if ($request->filled('start_date')) {
            $query->whereDate('incidents.created_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('incidents.created_at', '<=', $request->end_date);
        }
        if ($request->filled('site_id')) {
            $query->where('schedules.site_id', $request->site_id);
        }
        if ($request->filled('employee_id')) {
            $query->where('incidents.employee_id', $request->employee_id);
        }

        return $query;
    }

    public function downloadIncidentImages($id)
    {
        // Use standard DB call or Model if available. Query used in incidents() uses joins, 
        // but for a single incident download, simple lookup is safer.
        // Assuming Incident model exists as it is used in EmployeeJobController.
        $incident = \App\Models\Incident::findOrFail($id);

        if (empty($incident->images)) {
            return back()->with('error', 'No images found for this incident.');
        }

        // images is cast to array in model? EmployeeJobController saves it as array but doesn't show cast in snippet.
        // But in EmployeeJobController: incidents()->create(['images' => $images]) where $images is array.
        // It's likely cast in Model or stored as JSON.
        // If stored as JSON string in DB, Eloquent 'casts' => ['images' => 'array'] handles it.
        // If not cast, we need json_decode.
        // Let's assume it is cast or decode it safely.
        $images = $incident->images;
        if (is_string($images)) {
            $images = json_decode($images, true);
        }

        if (!is_array($images) || count($images) === 0) {
            return back()->with('error', 'No images found for this incident.');
        }

        $zipFileName = 'incident_' . $incident->duty_number . '_' . $incident->id . '.zip';
        $zipPath = storage_path('app/public/temp/' . $zipFileName);

        // Ensure temp dir exists
        if (!file_exists(dirname($zipPath))) {
            mkdir(dirname($zipPath), 0755, true);
        }

        $zip = new \ZipArchive;
        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === TRUE) {
            foreach ($images as $imagePath) {
                // Image path is relative to storage/app/public usually (based on store('...', 'public'))
                // So full path is storage_path('app/public/' . $imagePath)
                $fullPath = storage_path('app/public/' . $imagePath);

                if (file_exists($fullPath)) {
                    $zip->addFile($fullPath, basename($imagePath));
                }
            }
            $zip->close();
        } else {
            return back()->with('error', 'Failed to create zip file.');
        }

        return response()->download($zipPath)->deleteFileAfterSend(true);
    }

    public function generateSinglePdf($id)
    {
        $reportType = request()->input('type', 'report');

        // Check if $id is schedule_id or employee_schedule_id
        // In index view, we passed schedule_id for scans_history.
        // For standard checkout/combined, we passed employee_schedule.id as report_id.
        // Let's standardise or detect.

        $query = DB::table('employee_schedule')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->leftJoin('job_roles', 'employees.job_role_id', '=', 'job_roles.id')
            ->select(
                'schedules.id as schedule_id',
                'schedules.schedule_date',
                'schedules.duty_number',
                'schedules.status',
                'schedules.from_datetime',
                'schedules.to_datetime',
                'sites.name as site_name',
                'sites.address_line_1 as site_address',
                'employees.first_name',
                'employees.last_name',
                'employees.last_name',
                'employees.employee_id',
                'employees.id as employee_pk',
                'job_roles.name as job_role',
                'employee_schedule.actual_start_at',
                'employee_schedule.actual_end_at',
                (DB::raw("CONCAT(employee_schedule.start_lat, ', ', employee_schedule.start_lng) as checkin_coordinates")),
                (DB::raw("CONCAT(employee_schedule.end_lat, ', ', employee_schedule.end_lng) as checkout_coordinates")),
                'employee_schedule.checkin_images',
                'employee_schedule.checkout_images as checkout_evidence',
                'employee_schedule.created_at as assigned_at'
            );

        // Always lookup by employee_schedule.id for precise employee targeting
        $query->where('employee_schedule.id', $id);

        $report = $query->first();

        if (!$report) {
            abort(404);
        }

        $scans = [];
        if ($reportType === 'scans_history') {
            // Fetch Scans for this Schedule
            // We can filter by employee_table.id if needed, but usually scans are linked to schedule.
            // CheckpointScan model has schedule_id.
            $scans = DB::table('checkpoint_scans')
                ->leftJoin('checkpoints', 'checkpoint_scans.checkpoint_id', '=', 'checkpoints.id')
                ->leftJoin('tour_routes', 'checkpoint_scans.tour_route_id', '=', 'tour_routes.id')
                ->where('checkpoint_scans.schedule_id', $report->schedule_id)
                ->where('checkpoint_scans.employee_id', $report->employee_pk) // Filter by specific employee
                ->select(
                    'checkpoint_scans.*',
                    'checkpoints.name as checkpoint_name',
                    'tour_routes.description as tour_route_name'
                )
                ->orderBy('checkpoint_scans.scanned_at', 'desc')
                ->get();
        }

        $company_id = session('selected_company_id');
        $company = \App\Models\Company::find($company_id);

        $pdf = Pdf::loadView('reports.operational.single_pdf', compact('report', 'scans', 'reportType', 'company'));
        return $pdf->download("Operational_Report_{$report->duty_number}_{$report->first_name}.pdf");
    }

    public function lateCheckin(Request $request)
    {
        $company_id = session('selected_company_id');
        $company = \App\Models\Company::find($company_id);
        $lateCheckins = $this->getLateCheckinData($company_id, $request);
        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();

        return view('reports.operational.late_checkin', compact('lateCheckins', 'sites', 'company'));
    }

    public function exportLateCheckinPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $company = \App\Models\Company::find($company_id);
        $request->merge(['export' => true]);
        $lateCheckins = $this->getLateCheckinData($company_id, $request);
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $pdf = Pdf::loadView('admin.reports.operational.late_checkin_pdf', compact('lateCheckins', 'startDate', 'endDate', 'company'))
            ->setPaper('a4', 'portrait');

        return $pdf->download("Late_Checkin_{$startDate}_{$endDate}.pdf");
    }

    public function exportLateCheckinExcel(Request $request)
    {
        $company_id = session('selected_company_id');
        $request->merge(['export' => true]);
        $lateCheckins = $this->getLateCheckinData($company_id, $request);
        $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));

        $filename = "Late_Checkin_{$startDate}_{$endDate}.xlsx";
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = ['Emp ID', 'Employee Name', 'Site', 'Scheduled Start', 'Actual Start', 'Difference', 'Date'];
        $sheet->fromArray($headers, NULL, 'A1');

        $headerStyle = [
            'font' => ['bold' => true, 'color' => ['argb' => 'FFFFFFFF']],
            'fill' => ['fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID, 'startColor' => ['argb' => 'FF1E293B']],
        ];
        $sheet->getStyle('A1:G1')->applyFromArray($headerStyle);

        $rowNumber = 2;
        foreach ($lateCheckins as $row) {
            $scheduled = Carbon::parse($row->scheduled_start);
            $actual = Carbon::parse($row->actual_start_at);
            $diff = $scheduled->diff($actual)->format('%H:%I:%S');

            $sheet->setCellValue('A' . $rowNumber, $row->duty_number);
            $sheet->setCellValue('B' . $rowNumber, $row->first_name . ' ' . $row->last_name);
            $sheet->setCellValue('C' . $rowNumber, $row->site_name);
            $sheet->setCellValue('D' . $rowNumber, $scheduled->format('H:i'));
            $sheet->setCellValue('E' . $rowNumber, $actual->format('H:i'));
            $sheet->setCellValue('F' . $rowNumber, $diff);
            $sheet->setCellValue('G' . $rowNumber, Carbon::parse($row->schedule_date)->format('Y-m-d'));
            $rowNumber++;
        }

        foreach (range('A', 'G') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $response = response()->stream(
            function () use ($writer) {
                $writer->save('php://output');
            },
            200,
            [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
                'Cache-Control' => 'max-age=0',
            ]
        );

        return $response;
    }

    private function getLateCheckinData($company_id, Request $request)
    {
        $query = DB::table('employee_schedule')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->where('schedules.company_id', $company_id)
            ->whereNotNull('employee_schedule.actual_start_at')
            ->whereRaw('employee_schedule.actual_start_at > schedules.from_datetime')
            ->select(
                'schedules.schedule_date',
                'schedules.duty_number',
                'schedules.from_datetime as scheduled_start',
                'employee_schedule.actual_start_at',
                'sites.name as site_name',
                'employees.first_name',
                'employees.last_name'
            )
            ->orderBy('schedules.schedule_date', 'desc');

        if ($request->filled('start_date')) {
            $query->whereDate('schedules.schedule_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('schedules.schedule_date', '<=', $request->end_date);
        }

        if ($request->has('export')) {
            return $query->get();
        }

        return $query->paginate(15)->withQueryString();
    }

    private function calculateWageRates($payouts, $calculationMode = 'actual')
    {
        $collection = ($payouts instanceof \Illuminate\Support\Collection) ? $payouts : collect($payouts);
        if ($collection->isEmpty())
            return;

        $employeeIds = $collection->pluck('employee_id')->unique()->toArray();
        $employeeRates = DB::table('employee_wage_type')
            ->join('wage_types', 'employee_wage_type.wage_type_id', '=', 'wage_types.id')
            ->whereIn('employee_id', $employeeIds)
            ->select('employee_id', 'wage_type_id', 'rate', 'wage_types.name as type_name')
            ->get()
            ->groupBy('employee_id')
            ->map(function ($items) {
                return $items->keyBy('wage_type_id');
            });

        foreach ($collection as $item) {
            $sinRate = 0;
            $otherRate = 0;
            $calculatedRate = 0;
            $displayTypes = [];
            $wageTypesData = json_decode($item->wage_types ?? '[]', true);

            // Calculate actual worked hours
            $actualWorkedHours = 0;
            $scheduledHours = 0;

            if ($item->actual_start_at && $item->actual_end_at) {
                $schedStart = \Carbon\Carbon::parse($item->from_datetime);
                $actualStart = \Carbon\Carbon::parse($item->actual_start_at);
                $end = \Carbon\Carbon::parse($item->actual_end_at);

                // Raw Actual Hours (Actual Out - Actual In)
                // Ensure end is after start to avoid negative/zero if data is bad, though DB should handle it.
                if ($end->greaterThan($actualStart)) {
                    $actualWorkedHours = $actualStart->diffInMinutes($end) / 60;
                }

                // Scheduled Hours
                $schedEnd = \Carbon\Carbon::parse($item->to_datetime);
                $scheduledHours = $schedStart->diffInMinutes($schedEnd) / 60;
            }

            // Determine Target Hours based on Mode
            $targetHours = ($calculationMode === 'scheduled') ? $scheduledHours : $actualWorkedHours;

            // Attach View Properties for Consistency
            $item->view_hours = $targetHours;
            $item->view_start = ($calculationMode === 'scheduled') ? $item->from_datetime : $item->actual_start_at;
            $item->view_end = ($calculationMode === 'scheduled') ? $item->to_datetime : $item->actual_end_at;

            // Calculate total allocated hours in DB to support proportional scaling
            $dbAllocatedTotal = 0;
            if (!empty($wageTypesData) && is_array($wageTypesData)) {
                foreach ($wageTypesData as $wtInfo) {
                    if (is_array($wtInfo)) {
                        $dbAllocatedTotal += ($wtInfo['allocated_hours'] ?? 0);
                    }
                }
            }

            $processedWageTypes = [];

            if (!empty($wageTypesData) && is_array($wageTypesData)) {
                $empRates = $employeeRates[$item->employee_id] ?? collect([]);

                // Sorting Logic: Priority for "sin".
                usort($wageTypesData, function ($a, $b) {
                    $nameA = is_array($a) ? strtolower($a['name'] ?? '') : '';
                    $nameB = is_array($b) ? strtolower($b['name'] ?? '') : '';
                    $isSinA = strpos($nameA, 'sin') !== false;
                    $isSinB = strpos($nameB, 'sin') !== false;
                    if ($isSinA && !$isSinB)
                        return -1;
                    if (!$isSinA && $isSinB)
                        return 1;
                    return 0;
                });

                $remainingHours = $targetHours;

                foreach ($wageTypesData as $wtInfo) {
                    $idToLookup = null;
                    $allocatedHours = 0;

                    if (is_array($wtInfo)) {
                        $idToLookup = $wtInfo['id'] ?? null;
                        $dbAllocated = $wtInfo['allocated_hours'] ?? 0;

                        if ($calculationMode == 'scheduled') {
                            // Universal Scaling Logic
                            if ($dbAllocatedTotal > 0) {
                                $allocatedHours = ($dbAllocated / $dbAllocatedTotal) * $targetHours;
                            } else {
                                $allocatedHours = $dbAllocated;
                            }
                        } else {
                            // Waterfall for Actuals
                            $allocatedHours = min($remainingHours, $dbAllocated);
                            $remainingHours -= $allocatedHours;
                            if ($remainingHours < 0)
                                $remainingHours = 0;
                        }
                    } elseif (is_scalar($wtInfo)) {
                        $idToLookup = $wtInfo;
                        // Scalar means "applies to whole shift"
                        $allocatedHours = $targetHours;
                        $remainingHours = 0;
                    }

                    if ($idToLookup) {
                        $record = $empRates->get($idToLookup);

                        // Get rate and name from pivot if available, fallback to record
                        $pivotRate = is_array($wtInfo) ? ($wtInfo['rate'] ?? null) : null;
                        $pivotName = is_array($wtInfo) ? ($wtInfo['name'] ?? null) : null;
                        $profileRate = $record ? $record->rate : null;
                        $profileName = $record ? $record->type_name : null;

                        // Mode-based prioritization
                        if ($calculationMode === 'scheduled') {
                            $effectiveRate = $profileRate ?? $pivotRate ?? 0;
                        } else {
                            $effectiveRate = $pivotRate ?? $profileRate ?? 0;
                        }

                        $effectiveName = $pivotName ?? $profileName ?? 'Unknown';

                        if ($effectiveRate > 0 || $record) {
                            $calculatedRate += $effectiveRate;
                            $displayTypes[] = $effectiveName . ' ($' . number_format($effectiveRate, 2) . ')';

                            if (stripos($effectiveName, 'sin') !== false) {
                                $sinRate += $effectiveRate;
                            } else {
                                $otherRate += $effectiveRate;
                            }

                            // Store wage type with allocated hours (not capped)
                            $processedWageTypes[] = [
                                'id' => $idToLookup,
                                'name' => $effectiveName,
                                'rate' => $effectiveRate,
                                'allocated_hours' => $allocatedHours
                            ];
                        }
                    }
                }
            }

            $multiplier = $item->stat_multiplier ?? 1;
            $item->sin_rate = $sinRate * $multiplier;
            $item->other_rate = $otherRate * $multiplier;

            $item->wage_type_display = !empty($displayTypes) ? implode(' + ', $displayTypes) : 'sin';

            if ($calculatedRate > 0) {
                $item->wage_rate = $calculatedRate * $multiplier;
            }

            // Store the processed wage types with allocated hours as JSON
            $item->wage_types = json_encode($processedWageTypes);

            // Calculate Total Payout for the row
            $amount = 0;
            if (!empty($processedWageTypes)) {
                foreach ($processedWageTypes as $wt) {
                    $amount += ($wt['allocated_hours'] * $wt['rate']);
                }
            } else {
                $amount = $targetHours * ($item->wage_rate ?? 0);
            }
            $amount += ($item->gas_rate ?? 0);
            $item->total_payout = $amount;
        }
    }

    public function emailShiftReport(Schedule $schedule, Employee $employee)
    {
        $user = auth()->user();

        // Load necessary relationships for the comprehensive report view
        $schedule->load([
            'site',
            'company',
            'incidents' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            }
        ]);

        // Get the pivot data for this specific employee on this schedule
        $empWithPivot = $schedule->employees()->where('employees.id', $employee->id)->first();
        if (!$empWithPivot) {
            return back()->with('error', 'Employee not found on this schedule.');
        }
        $pivot = $empWithPivot->pivot;

        $isPdf = true;
        // Reusing the comprehensive job report view
        $pdf = Pdf::loadView('employee.reports.complete', compact('schedule', 'employee', 'pivot', 'isPdf'));

        $pdfContent = $pdf->output();
        $filename = "Shift_Report_{$schedule->duty_number}.pdf";

        try {
            Mail::html("<p>Hello {$employee->first_name},</p><p>Please find attached the comprehensive shift report for <strong>Duty #{$schedule->duty_number}</strong>.</p><p>This report includes operative profile, tactical timeline, geospatial coordinates, and incident logs.</p><p>Regards,<br>Operations Team</p>", function ($message) use ($employee, $pdfContent, $filename, $schedule) {
                $message->to($employee->email)
                    ->subject("Shift Report: Duty #{$schedule->duty_number}")
                    ->attachData($pdfContent, $filename, [
                        'mime' => 'application/pdf',
                    ]);
            });

            return back()->with('success', 'Shift report has been emailed to ' . $employee->email);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Email Shift Report Error: ' . $e->getMessage());
            return back()->with('error', 'Failed to send email: ' . $e->getMessage());
        }
    }

    public function emailScansHistoryToClient($id)
    {
        $employeeSchedule = DB::table('employee_schedule')->find($id);
        if (!$employeeSchedule) {
            return back()->with('error', 'Report not found.');
        }

        $schedule = Schedule::with('site')->findOrFail($employeeSchedule->schedule_id);
        $employee = Employee::findOrFail($employeeSchedule->employee_id);

        $clientEmail = $schedule->site->contact_email;
        if (!$clientEmail) {
            return back()->with('error', 'Client email not found for this site.');
        }

        $reportType = 'scans_history';

        $query = DB::table('employee_schedule')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->leftJoin('job_roles', 'employees.job_role_id', '=', 'job_roles.id')
            ->select(
                'schedules.id as schedule_id',
                'schedules.schedule_date',
                'schedules.duty_number',
                'schedules.status',
                'schedules.from_datetime',
                'schedules.to_datetime',
                'sites.name as site_name',
                'sites.address_line_1 as site_address',
                'employees.first_name',
                'employees.last_name',
                'employees.employee_id',
                'employees.id as employee_pk',
                'job_roles.name as job_role',
                'employee_schedule.actual_start_at',
                'employee_schedule.actual_end_at',
                'employee_schedule.created_at as assigned_at'
            )
            ->where('employee_schedule.id', $id);

        $report = $query->first();

        $scans = DB::table('checkpoint_scans')
            ->leftJoin('checkpoints', 'checkpoint_scans.checkpoint_id', '=', 'checkpoints.id')
            ->leftJoin('tour_routes', 'checkpoint_scans.tour_route_id', '=', 'tour_routes.id')
            ->where('checkpoint_scans.schedule_id', $report->schedule_id)
            ->where('checkpoint_scans.employee_id', $report->employee_pk)
            ->select(
                'checkpoint_scans.*',
                'checkpoints.name as checkpoint_name',
                'tour_routes.description as tour_route_name'
            )
            ->orderBy('checkpoint_scans.scanned_at', 'desc')
            ->get();

        $company_id = session('selected_company_id');
        $company = \App\Models\Company::find($company_id);

        $pdf = Pdf::loadView('reports.operational.single_pdf', compact('report', 'scans', 'reportType', 'company'))
            ->setOption('isRemoteEnabled', true)
            ->setOption('isHtml5ParserEnabled', true);

        $pdfContent = $pdf->output();
        $filename = "Scan_History_{$schedule->duty_number}_{$employee->first_name}.pdf";

        try {
            Mail::html("<p>Hello,</p><p>Please find attached the Checkpoint Scan History report for <strong>Duty #{$schedule->duty_number}</strong> at <strong>{$schedule->site->name}</strong>.</p><p>Regards,<br>Operations Team</p>", function ($message) use ($clientEmail, $pdfContent, $filename, $schedule) {
                $message->to($clientEmail)
                    ->subject("Checkpoint Scan History Report: Duty #{$schedule->duty_number}")
                    ->attachData($pdfContent, $filename, [
                        'mime' => 'application/pdf',
                    ]);
            });

            return back()->with('success', 'Scan history report has been emailed to the client at ' . $clientEmail);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Email Scan History Error: ' . $e->getMessage());
            return back()->with('error', 'Failed to send email: ' . $e->getMessage());
        }
    }
}
