<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\Uniform;
use App\Models\UniformVariant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\EmployeeUniform;
use App\Models\Company;
use App\Services\ExcelExportService;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use Barryvdh\DomPDF\Facade\Pdf;

class EmployeeUniformController extends Controller
{
    public function index()
    {
        if (Auth::guard('web')->check()) {
            return $this->adminIndex();
        }

        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            abort(403, 'Unauthorized');
        }

        $inventory = UniformVariant::with('uniform')
            ->whereHas('uniform', function ($q) use ($employee) {
                $q->where('company_id', $employee->company_id)
                    ->where('active', true);
            })
            ->where('active', true)
            ->orderBy('uniform_id')
            ->get();

        $myUniforms = $employee->issuedUniforms()->with('variant.uniform')->latest('issued_at')->get();

        return view('employee.uniforms.index', compact('inventory', 'myUniforms'));
    }

    public function exportStockPdf()
    {
        $companyId = session('selected_company_id');

        $inventory = UniformVariant::with('uniform')
            ->whereHas('uniform', function ($q) use ($companyId) {
                $q->where('company_id', $companyId)
                    ->where('active', true);
            })
            ->where('active', true)
            ->orderBy('uniform_id')
            ->get();

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('reports.uniforms.stock_pdf', compact('inventory'));
        return $pdf->download('Uniform_Stock_Levels_' . now()->format('Ymd_Hi') . '.pdf');
    }

    private function adminIndex()
    {
        $companyId = session('selected_company_id');

        $inventory = UniformVariant::with('uniform')
            ->whereHas('uniform', function ($q) use ($companyId) {
                $q->where('company_id', $companyId)
                    ->where('active', true);
            })
            ->where('active', true)
            ->orderBy('uniform_id')
            ->get();

        // Fetch all issued uniforms for the company
        $allIssuedUniforms = \App\Models\EmployeeUniform::whereHas('employee', function ($q) use ($companyId) {
            $q->where('company_id', $companyId);
        })
            ->with(['employee', 'variant.uniform'])
            ->latest('issued_at')
            ->get();

        // We can reuse the view if we adapt it or create a new one. 
        // Given the request "show ... which uniform is assigned to which employee", a new view is better.
        // Let's create 'user.uniforms.index' view.
        return view('user.uniforms.index', compact('inventory', 'allIssuedUniforms'));
    }
    public function exportIssuedPdf(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = EmployeeUniform::whereHas('employee', function ($q) use ($companyId) {
            $q->where('company_id', $companyId);
        })->with(['employee', 'variant.uniform'])->latest('issued_at');

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('id', $ids);
        }

        $allIssuedUniforms = $query->get();
        $company = Company::find($companyId);
        $companyName = $company ? $company->name : 'ShiftMatrix';

        $pdf = Pdf::loadView('reports.uniforms.issued_pdf', compact('allIssuedUniforms', 'companyName'))
            ->setPaper('a4', 'landscape');

        return $pdf->download('Issued_Uniforms_Log_' . now()->format('Ymd_Hi') . '.pdf');
    }

    public function exportIssuedExcel(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = EmployeeUniform::whereHas('employee', function ($q) use ($companyId) {
            $q->where('company_id', $companyId);
        })->with(['employee', 'variant.uniform'])->latest('issued_at');

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('id', $ids);
        }

        $allIssuedUniforms = $query->get();
        
        $excelService = new ExcelExportService();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        $colCount = 6;
        $excelService->applyBrandedHeader($sheet, 'Issued Uniforms Log', 'All Assignments', $colCount);
        
        $currentRow = 5;
        $headers = ['Employee', 'Uniform Item', 'Size / Color', 'Quantity', 'Issued Date', 'Notes'];
        $sheet->fromArray($headers, NULL, "A{$currentRow}");
        $excelService->applyTableHeaders($sheet, "A{$currentRow}:F{$currentRow}");
        $currentRow++;

        $startDataRow = $currentRow;
        foreach ($allIssuedUniforms as $issued) {
            $variantDetails = ($issued->variant->size ? $issued->variant->size : '') . 
                             ($issued->variant->size && $issued->variant->color ? ' / ' : '') . 
                             ($issued->variant->color ? $issued->variant->color : '');
            
            $sheet->setCellValue("A{$currentRow}", $issued->employee->first_name . ' ' . $issued->employee->last_name);
            $sheet->setCellValue("B{$currentRow}", $issued->variant->uniform->name);
            $sheet->setCellValue("C{$currentRow}", $variantDetails ?: 'Standard');
            $sheet->setCellValue("D{$currentRow}", $issued->quantity);
            $sheet->setCellValue("E{$currentRow}", $issued->issued_at->format('M d, Y'));
            $sheet->setCellValue("F{$currentRow}", $issued->notes ?: '-');
            $currentRow++;
        }
        $endDataRow = $currentRow - 1;

        if ($endDataRow >= $startDataRow) {
            $excelService->applyDataStyles($sheet, "A{$startDataRow}:F{$endDataRow}", $colCount, [
                'center_columns' => [4, 5] // Quantity, Issued Date
            ]);
            
            // Zebra Striping
            for ($i = $startDataRow; $i <= $endDataRow; $i++) {
                if ($i % 2 == 0) {
                    $sheet->getStyle("A{$i}:F{$i}")->getFill()
                        ->setFillType(Fill::FILL_SOLID)
                        ->getStartColor()->setARGB('FFF1F5F9');
                }
            }
        }

        $excelService->applyFinalPolish($sheet, $colCount);

        $filename = "Issued_Uniforms_Log_" . now()->format('Ymd_His') . ".xlsx";

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename);
    }

    public function assign(Request $request)
    {
        $employee = Auth::guard('employee')->user();

        $request->validate([
            'uniform_variant_id' => 'required|exists:uniform_variants,id',
            'quantity' => 'required|integer|min:1',
            'issued_at' => 'required|date',
            'notes' => 'nullable|string'
        ]);

        $variant = UniformVariant::with('uniform')->findOrFail($request->uniform_variant_id);

        // Security check: ensure variant belongs to employee's company
        if ($variant->uniform->company_id != $employee->company_id) {
            abort(403, 'Unauthorized access to uniform item.');
        }

        if ($variant->stock_quantity < $request->quantity) {
            return back()->with('error', "Insufficient stock for {$variant->uniform->name}. Available: {$variant->stock_quantity}");
        }

        \Illuminate\Support\Facades\DB::transaction(function () use ($request, $employee, $variant) {
            // Deduct stock
            $variant->decrement('stock_quantity', $request->quantity);

            // Assign to employee
            $employee->issuedUniforms()->create([
                'uniform_variant_id' => $request->uniform_variant_id,
                'quantity' => $request->quantity,
                'issued_at' => $request->issued_at,
                'notes' => $request->notes
            ]);
        });

        return back()->with('success', 'Uniform assigned successfully.');
    }
}
