<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class EmployeeNotificationController extends Controller
{
    public function index()
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            return redirect()->route('employee.login');
        }

        $notifications = Notification::where('employee_id', $employee->id)
            ->orderBy('created_at', 'desc')
            ->paginate(100);

        return view('employee.notifications.index', compact('notifications'));
    }

    public function markAsRead($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $notification = Notification::where('employee_id', $employee->id)->findOrFail($id);

        $notification->update(['read_at' => now()]);

        return response()->json(['success' => true]);
    }

    /**
     * Mark notification as read and redirect
     */
    public function read($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            return redirect()->route('employee.login');
        }

        $notification = Notification::where('employee_id', $employee->id)->findOrFail($id);

        if (is_null($notification->read_at)) {
            $notification->update(['read_at' => now()]);
        }

        if ($notification->url) {
            return redirect($notification->url);
        }

        return redirect()->route('employee.notifications.index', ['#notification_' . $id]);
    }

    public function sendReminder(Request $request)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $request->validate([
            'job_id' => 'required',
            'job_type' => 'required|in:guard,patroller',
            'duty_number' => 'required|string'
        ]);

        // Only send if the job is still active
        if ($request->job_type === 'guard') {
            $job = \App\Models\Schedule::where('id', $request->job_id)
                ->where('status', 'active')
                ->first();
        } else {
            $job = \App\Models\PatrollerSchedule::where('id', $request->job_id)
                ->where('job_status', 'in_progress')
                ->first();
        }

        if (!$job) {
            return response()->json(['success' => false, 'message' => 'Job is no longer active.']);
        }

        Notification::create([
            'company_id' => $employee->company_id,
            'employee_id' => $employee->id,
            'title' => 'Mission Note Reminder',
            'message' => "Reminder: Please submit your mission note for Duty #{$request->duty_number}.",
            'url' => ($request->job_type === 'guard' ? route('employee.jobs.index') : route('employee.patroller-jobs.index')) . "?job_id={$request->job_id}&job_type={$request->job_type}",
        ]);

        return response()->json(['success' => true]);
    }
}
