<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class EmployeeDashboardController extends Controller
{
    use \App\Traits\CalculatesEarnings;

    public function index()
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            Auth::guard('employee')->logout();
            return redirect()->route('employee.login')->withErrors(['email' => 'Session invalid.']);
        }

        // Visibility Check
        $visibilityMinutes = 60; // Default fallback
        if ($employee->job_role_id) {
            $visibilitySetting = \App\Models\CompanyJobRoleVisibility::where('company_id', $employee->company_id)
                ->where('job_role_id', $employee->job_role_id)
                ->first();
            if ($visibilitySetting) {
                $visibilityMinutes = $visibilitySetting->visibility_minutes;
            }
        }
        $visibilityDeadline = Carbon::now()->addMinutes($visibilityMinutes);

        // Active jobs: Only those that have been explicitly started by this specific employee
        $activeJobs = $employee->schedules()
            ->with(['site', 'employees'])
            ->whereNotNull('employee_schedule.actual_start_at')
            ->whereNull('employee_schedule.actual_end_at')
            ->orderBy('from_datetime')
            ->get();

        // Today's jobs: Scheduled for today but NOT yet started by this employee
        $todayJobs = $employee->schedules()
            ->with(['site', 'employees'])
            ->whereDate('from_datetime', Carbon::today())
            ->where('to_datetime', '>', now())
            ->whereNull('employee_schedule.actual_start_at')
            ->whereNotIn('schedules.status', ['completed', 'inactive', 'cancelled'])
            ->orderBy('from_datetime')
            ->limit(2)
            ->get();

        // Completed jobs for reports
        $completedJobs = $employee->schedules()
            ->with(['site'])
            ->whereNotNull('actual_end_at')
            ->orderBy('from_datetime', 'desc')
            ->limit(5)
            ->get();

        // Fetch patroller jobs
        $activePatrollerJobs = $employee->patrollerSchedules()
            ->with(['route.sites'])
            ->where('job_status', 'in_progress')
            ->orderBy('from_time')
            ->get();

        $todayPatrollerJobs = $employee->patrollerSchedules()
            ->with(['route.sites'])
            ->whereDate('from_time', Carbon::today())
            ->where('to_time', '>', now())
            ->where('job_status', 'pending')
            ->orderBy('from_time')
            ->limit(2)
            ->get();

        $completedPatrollerJobs = $employee->patrollerSchedules()
            ->with(['route'])
            ->whereIn('job_status', ['completed', 'auto_ended'])
            ->orderBy('from_time', 'desc')
            ->limit(5)
            ->get();

        $categories = \App\Models\IncidentParentCategory::with([
            'incident_categories' => function ($q) {
                $q->where('status', true);
            }
        ])->where('status', true)->get();

        // --- Earnings Analytics for Dashboard ---
        $months = [];
        for ($i = 11; $i >= 0; $i--) {
            $months[] = now()->subMonths($i)->format('Y-m');
        }

        $incomeLabels = [];
        $incomeValues = [];
        $highestIncome = 0;
        $highestMonth = '';

        foreach ($months as $month) {
            $startOfMonth = Carbon::parse($month . '-01')->startOfMonth();
            $endOfMonth = Carbon::parse($month . '-01')->endOfMonth();
            $label = $startOfMonth->format('M');
            $incomeLabels[] = $label;

            // Guard Income - Only completed jobs
            $guardIncome = $employee->schedules()
                ->where('schedules.status', 'completed')
                ->whereBetween('from_datetime', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($j) {
                    return $this->calculateGuardEarnings($j);
                });

            // Patroller Income - Only completed jobs
            $patrollerIncome = $employee->patrollerSchedules()
                ->whereIn('patroller_schedules.job_status', ['completed', 'auto_ended'])
                ->whereBetween('patroller_schedules.from_time', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($j) {
                    return $this->calculatePatrollerEarnings($j);
                });

            $totalIncome = round($guardIncome + $patrollerIncome, 2);
            $incomeValues[] = $totalIncome;

            if ($totalIncome >= $highestIncome) {
                $highestIncome = $totalIncome;
                $highestMonth = $startOfMonth->format('F Y');
            }
        }

        $currentMonthIncome = end($incomeValues);

        $today = Carbon::today();

        $birthdayWish = null;
        if ($employee->dob && Carbon::parse($employee->dob)->format('m-d') === $today->format('m-d')) {
            $birthdayWish = "Happy Birthday, " . $employee->first_name . "! We hope you have a fantastic day!";
        }

        $hiringAnniversaryWish = null;
        if ($employee->employment_date && Carbon::parse($employee->employment_date)->format('m-d') === $today->format('m-d')) {
            $years = Carbon::parse($employee->employment_date)->age; // age gives diff in years from date to now
            if ($years > 0) {
                $hiringAnniversaryWish = "Happy Work Anniversary! Celebrating " . $years . " year" . ($years > 1 ? 's' : '') . " with us!";
            }
        }

        // Fetch issued uniforms
        $issuedUniforms = $employee->issuedUniforms()->with('variant.uniform')->latest('issued_at')->limit(5)->get();

        // --- Shift Hours 30d Calculation (Matching API) ---
        $last30Days = now()->subDays(30);

        $recentGuardSchedules = $employee->schedules()
            ->where('schedules.status', 'completed')
            ->where('from_datetime', '>=', $last30Days)
            ->get();

        $recentPatrollerSchedules = $employee->patrollerSchedules()
            ->whereIn('patroller_schedules.job_status', ['completed', 'auto_ended'])
            ->where('patroller_schedules.from_time', '>=', $last30Days)
            ->get();

        $shiftHours30d = 0;

        foreach ($recentGuardSchedules as $j) {
            $duration = $j->from_datetime->floatDiffInHours($j->to_datetime);
            $shiftHours30d += $duration;
        }

        foreach ($recentPatrollerSchedules as $j) {
            if ($j->from_time && $j->to_time) {
                $shiftHours30d += \Carbon\Carbon::parse($j->from_time)->floatDiffInHours(\Carbon\Carbon::parse($j->to_time));
            }
        }
        $shiftHours30d = round($shiftHours30d, 1);

        return view('employee.dashboard', compact(
            'employee',
            'activeJobs',
            'todayJobs',
            'completedJobs',
            'activePatrollerJobs',
            'todayPatrollerJobs',
            'completedPatrollerJobs',
            'categories',
            'incomeLabels',
            'incomeValues',
            'highestIncome',
            'highestMonth',
            'currentMonthIncome',
            'birthdayWish',
            'hiringAnniversaryWish',
            'issuedUniforms',
            'shiftHours30d'
        ));
    }

    public function support()
    {
        return view('employee.support');
    }

    public function submitSupport(Request $request)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'message' => 'required|string',
        ]);

        // Prepare email content
        $employee = Auth::guard('employee')->user();
        $emailSubject = "Technical Support Request from {$employee->first_name} {$employee->last_name}";
        $emailContent = "
            <p><strong>Employee:</strong> {$employee->first_name} {$employee->last_name} (#{$employee->employee_id})</p>
            <p><strong>Subject:</strong> {$request->subject}</p>
            <p><strong>Message:</strong></p>
            <div style='background-color: #f3f4f6; padding: 15px; border-radius: 5px; border: 1px solid #e5e7eb;'>
                " . nl2br(e($request->message)) . "
            </div>
            <p><small>Sent from Employee Portal Support</small></p>
        ";

        try {
            \Illuminate\Support\Facades\Mail::to('it@thefalconsecurity.com')->send(new \App\Mail\GenericEmail($emailSubject, $emailContent));
            return back()->with('success', 'Your technical support request has been submitted successfully to the IT team.');
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Support Email Failed: ' . $e->getMessage());
            return back()->with('error', 'Unable to send email at this time, but your request has been logged. Please try again later.');
        }
    }
}


