<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Uniform;
use App\Models\UniformVariant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class UniformController extends Controller
{
    public function index(Request $request)
    {
        $showInventory = $request->query('view') === 'inventory';

        if ($showInventory) {
            $companyId = session('selected_company_id') ?? auth()->user()->company_id ?? \App\Models\Company::first()->id;

            // Flatten variants for inventory view
            $inventory = UniformVariant::with('uniform')
                ->whereHas('uniform', function ($q) use ($companyId) {
                    $q->where('company_id', $companyId);
                })
                ->orderBy('uniform_id')
                ->get();

            return view('admin.uniforms.inventory', compact('inventory'));
        }

        // Standard index view
        $companyId = session('selected_company_id') ?? auth()->user()->company_id ?? \App\Models\Company::first()->id;
        $uniforms = Uniform::withCount('variants')->where('company_id', $companyId)->get();
        return view('admin.uniforms.index', compact('uniforms'));
    }

    public function create()
    {
        return view('admin.uniforms.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'variants' => 'nullable|array',
            'variants.*.size' => 'nullable|string|max:50',
            'variants.*.color' => 'nullable|string|max:50',
            'variants.*.stock_quantity' => 'required|integer|min:0',
        ]);

        DB::transaction(function () use ($request) {
            $uniform = Uniform::create([
                'name' => $request->name,
                'code' => $request->code,
                'description' => $request->description,
                'active' => true,
                'active' => true,
                'company_id' => session('selected_company_id') ?? auth()->user()->company_id ?? \App\Models\Company::first()->id,
            ]);

            if ($request->has('variants')) {
                foreach ($request->variants as $variant) {
                    if (empty($variant['size']) && empty($variant['color']))
                        continue;

                    $uniform->variants()->create([
                        'size' => $variant['size'],
                        'color' => $variant['color'],
                        'stock_quantity' => $variant['stock_quantity'],
                        'active' => true,
                    ]);
                }
            }
        });

        return redirect()->route('admin.uniforms.index')->with('success', 'Uniform created successfully');
    }

    public function edit(Uniform $uniform)
    {
        $uniform->load('variants');
        return view('admin.uniforms.edit', compact('uniform'));
    }

    public function show(Uniform $uniform)
    {
        $uniform->load('variants');
        $uniform->variants_count = $uniform->variants->count(); // load count manually or use loadCount() if needed, but simple load is enough for view loop
        return view('admin.uniforms.show', compact('uniform'));
    }

    public function update(Request $request, Uniform $uniform)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'variants' => 'nullable|array',
            'variants.*.id' => 'nullable|exists:uniform_variants,id',
            'variants.*.size' => 'nullable|string|max:50',
            'variants.*.color' => 'nullable|string|max:50',
            'variants.*.stock_quantity' => 'required|integer|min:0',
        ]);

        DB::transaction(function () use ($request, $uniform) {
            $uniform->update([
                'name' => $request->name,
                'code' => $request->code,
                'description' => $request->description,
            ]);

            // Handle Variants
            if ($request->has('variants')) {
                $submittedIds = [];
                foreach ($request->variants as $variantData) {
                    if (empty($variantData['size']) && empty($variantData['color']))
                        continue;

                    if (isset($variantData['id']) && $variantData['id']) {
                        $variant = UniformVariant::find($variantData['id']);
                        if ($variant && $variant->uniform_id == $uniform->id) {
                            $variant->update([
                                'size' => $variantData['size'],
                                'color' => $variantData['color'],
                                'stock_quantity' => $variantData['stock_quantity'],
                            ]);
                            $submittedIds[] = $variant->id;
                        }
                    } else {
                        $newVariant = $uniform->variants()->create([
                            'size' => $variantData['size'],
                            'color' => $variantData['color'],
                            'stock_quantity' => $variantData['stock_quantity'],
                            'active' => true,
                        ]);
                        $submittedIds[] = $newVariant->id;
                    }
                }

                // Delete variants not in submission
                $uniform->variants()->whereNotIn('id', $submittedIds)->delete();
            } else {
                // If no variants sent, maybe delete all? Or do nothing?
                // Usually if 'variants' key exists but empty array, delete all.
                // If key missing, do nothing?
                // In the view form, variants array will always exist if x-data submits it.
                // Let's assume if variants array is passed, we sync.
                if ($request->has('variants')) {
                    $uniform->variants()->delete();
                }
            }
        });

        return redirect()->route('admin.uniforms.index')->with('success', 'Uniform updated successfully');
    }

    public function toggleStatus(Uniform $uniform)
    {
        $uniform->active = !$uniform->active;
        $uniform->save();
        return response()->json(['message' => 'Status updated successfully']);
    }

    public function destroy(Uniform $uniform)
    {
        $uniform->delete();
        return redirect()->route('admin.uniforms.index')->with('success', 'Uniform deleted successfully');
    }
}
