<?php

namespace App\Http\Controllers;

use App\Models\Schedule;
use App\Models\PatrollerSchedule;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use App\Services\ImageService;
use Illuminate\Pagination\LengthAwarePaginator;

class AddonScheduleController extends Controller
{
    public function index(Request $request)
    {
        $companyId = session('selected_company_id');
        $now = \Carbon\Carbon::now();
        $hasFilter = $request->anyFilled(['search', 'start_date', 'end_date']);

        if (!$hasFilter) {
            return view('addon_schedules.index', ['schedules' => new LengthAwarePaginator([], 0, 100)]);
        }

        // 1. Guard Schedules Query
        $guardQuery = Schedule::with(['site', 'employees'])
            ->where('company_id', $companyId);

        $guardQuery->where(function ($q) use ($now) {
            $q->whereIn('status', ['missed', 'cancelled'])
                ->orWhere(function ($sq) use ($now) {
                    $sq->where('to_datetime', '<=', $now)
                        ->whereNotIn('status', ['active', 'completed', 'cancelled']);
                });
        });

        // 2. Patroller Schedules Query
        $patrolQuery = PatrollerSchedule::with(['route', 'employees'])
            ->where('company_id', $companyId);

        $patrolQuery->where(function ($q) use ($now) {
            $q->where('job_status', 'cancelled')
                ->orWhere(function ($sq) use ($now) {
                    $sq->where('to_time', '<=', $now)
                        ->whereNotIn('job_status', ['in_progress', 'completed', 'auto_ended', 'cancelled']);
                })
                ->orWhere(function ($sq) {
                    $sq->whereDate('from_time', '<', today())
                        ->where('job_status', 'pending');
                });
        });

        // Apply filters to both
        if ($request->filled('search')) {
            $search = $request->search;
            $filter = function ($q) use ($search) {
                $q->where('duty_number', 'like', "%{$search}%");
                if ($q->getModel()->getTable() === 'schedules') {
                    $q->orWhereHas('site', function ($subQ) use ($search) {
                        $subQ->where('name', 'like', "%{$search}%")
                            ->orWhere('site_id', 'like', "%{$search}%");
                    });
                } else {
                    $q->orWhereHas('route', function ($subQ) use ($search) {
                        $subQ->where('name', 'like', "%{$search}%");
                    });
                }
                $q->orWhereHas('employees', function ($subQ) use ($search) {
                    $subQ->where('employees.first_name', 'like', "%{$search}%")
                        ->orWhere('employees.last_name', 'like', "%{$search}%")
                        ->orWhere('employees.employee_id', 'like', "%{$search}%");
                });
            };
            $guardQuery->where($filter);
            $patrolQuery->where($filter);
        }

        if ($request->filled('start_date')) {
            $guardQuery->whereDate('from_datetime', '>=', $request->start_date);
            $patrolQuery->whereDate('from_time', '>=', $request->start_date);
        }

        if ($request->filled('end_date')) {
            $guardQuery->whereDate('from_datetime', '<=', $request->end_date);
            $patrolQuery->whereDate('from_time', '<=', $request->end_date);
        }

        // Fetch results
        $guardResults = $guardQuery->get();
        $patrolResults = $patrolQuery->get();

        // Merge and sort
        $combined = $guardResults->concat($patrolResults)
            ->sortByDesc(function ($item) {
                return $item->from_datetime; // PatrollerSchedule has from_datetime accessor now
            })->values();

        // Paginate manually
        $perPage = 100;
        $page = $request->get('page', 1);
        $schedules = new LengthAwarePaginator(
            $combined->forPage($page, $perPage),
            $combined->count(),
            $perPage,
            $page,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('addon_schedules.index', compact('schedules'));
    }

    /**
     * Show form to manually complete a schedule for a specific employee.
     * Since a schedule can have multiple employees, we need to pick which one to complete.
     * Or if it's 1:1 usually, we pick the first uncompleted one or list valid options.
     */
    public function create(Request $request, Schedule $schedule)
    {
        // Check if multiple employees, require selection?
        // For simplicity, let's assume the user selects the employee in the modal or we pass it.
        $employeeId = $request->input('employee_id');

        if (!$employeeId) {
            // If multiple employees, logic needed. For now, take first uncompleted.
            $employee = $schedule->employees()
                ->wherePivot('actual_start_at', null)
                ->first();
            if (!$employee) {
                return back()->with('error', 'No incomplete employee assignment found for this schedule.');
            }
            $employeeId = $employee->id;
        }

        $employee = \App\Models\Employee::find($employeeId);

        return view('addon_schedules.create', compact('schedule', 'employee'));
    }

    /**
     * Store the manual completion data.
     */
    public function store(Request $request, $id)
    {
        $request->validate([
            'type' => 'required|in:guard,patrol',
            'employees' => 'required|array|min:1',
            'employees.*.employee_id' => 'required|exists:employees,id',
            'employees.*.checkin_datetime' => 'required|date',
            'employees.*.checkout_datetime' => 'required|date|after:employees.*.checkin_datetime',
            'employees.*.checkin_image' => 'nullable|image|max:10240',
            'employees.*.checkout_image' => 'nullable|image|max:10240',
        ]);

        $type = $request->type;
        $schedule = $type === 'patrol'
            ? PatrollerSchedule::findOrFail($id)
            : Schedule::findOrFail($id);

        DB::transaction(function () use ($schedule, $request, $type) {
            $completedCount = 0;

            foreach ($request->employees as $employeeData) {
                $employee = $schedule->employees()->where('employees.id', $employeeData['employee_id'])->first();
                if (!$employee)
                    continue;

                $checkinPath = null;
                if (isset($employeeData['checkin_image']) && $employeeData['checkin_image'] instanceof \Illuminate\Http\UploadedFile) {
                    $timestamp = Carbon::parse($employeeData['checkin_datetime']);
                    $folderName = ($type === 'patrol' ? 'patrol-' : '') . 'check-in';
                    $siteName = $type === 'patrol' ? ($schedule->route->name ?? 'Route') : ($schedule->site->name ?? 'Site');
                    $folderPath = ImageService::getFolderPath($folderName, $schedule->duty_number, $siteName, $timestamp);
                    $checkinPath = ImageService::processAndStore($employeeData['checkin_image'], $folderPath);
                }

                $checkoutPath = null;
                if (isset($employeeData['checkout_image']) && $employeeData['checkout_image'] instanceof \Illuminate\Http\UploadedFile) {
                    $timestamp = Carbon::parse($employeeData['checkin_datetime']);
                    $folderName = ($type === 'patrol' ? 'patrol-' : '') . 'check-out';
                    $siteName = $type === 'patrol' ? ($schedule->route->name ?? 'Route') : ($schedule->site->name ?? 'Site');
                    $folderPath = ImageService::getFolderPath($folderName, $schedule->duty_number, $siteName, $timestamp);
                    $checkoutPath = ImageService::processAndStore($employeeData['checkout_image'], $folderPath);
                }

                $pivotData = [
                    'actual_start_at' => Carbon::parse($employeeData['checkin_datetime']),
                    'actual_end_at' => Carbon::parse($employeeData['checkout_datetime']),
                ];

                $pivotData['start_evidence_path'] = $checkinPath;
                $pivotData['end_evidence_path'] = $checkoutPath;
                $pivotData['is_addon'] = true;

                $schedule->employees()->updateExistingPivot($employeeData['employee_id'], $pivotData);
                $completedCount++;
            }

            if ($schedule->employees()->wherePivotNull('actual_start_at')->count() === 0) {
                if ($type === 'guard') {
                    $schedule->update(['status' => 'completed']);
                } else {
                    $schedule->update(['job_status' => 'completed']);
                }
            }

            \App\Models\ActivityLog::create([
                'user_id' => auth()->id(),
                'action' => 'addon_completed',
                'subject_type' => get_class($schedule),
                'subject_id' => $schedule->id,
                'description' => "Manually completed {$completedCount} employee(s) for " . ($type === 'patrol' ? 'Patroller' : 'Guard') . " schedule #{$schedule->duty_number} via Addon",
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);
        });

        return redirect()->route('addon-schedules.index')->with('success', 'Schedule manually marked as complete.');
    }

    public function history(Request $request)
    {
        $companyId = session('selected_company_id');

        // 1. Guard Schedules
        $guardSchedules = Schedule::where('company_id', $companyId)
            ->whereHas('employees', function ($q) {
                $q->where('employee_schedule.is_addon', true);
            })
            ->with([
                'employees' => function ($q) {
                    $q->where('employee_schedule.is_addon', true);
                },
                'site'
            ])
            ->get();

        // 2. Patroller Schedules
        $patrolSchedules = PatrollerSchedule::where('company_id', $companyId)
            ->whereHas('employees', function ($q) {
                $q->where('employee_patroller_schedule.is_addon', true);
            })
            ->with([
                'employees' => function ($q) {
                    $q->where('employee_patroller_schedule.is_addon', true);
                },
                'route'
            ])
            ->get();

        // Merge and sort
        $combined = $guardSchedules->concat($patrolSchedules)
            ->sortByDesc(function ($item) {
                return $item->from_datetime;
            })->values();

        // Paginate
        $perPage = 100;
        $page = $request->get('page', 1);
        $schedules = new LengthAwarePaginator(
            $combined->forPage($page, $perPage),
            $combined->count(),
            $perPage,
            $page,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('addon_schedules.history', compact('schedules'));
    }

    public function downloadEvidence(Request $request, $id, $employeeId)
    {
        $type = $request->query('type', 'guard');

        if ($type === 'patrol') {
            $schedule = PatrollerSchedule::findOrFail($id);
        } else {
            $schedule = Schedule::findOrFail($id);
        }
        $employee = $schedule->employees()
            ->where('employees.id', $employeeId)
            ->firstOrFail();

        $startPath = $employee->pivot->start_evidence_path;
        $endPath = $employee->pivot->end_evidence_path;

        if (!$startPath && !$endPath) {
            return back()->with('error', 'No evidence files found.');
        }

        $dutyId = $schedule->duty_number ?? $schedule->id;
        $timestamp = now()->format('YmdHis');
        $mainFolderName = "{$dutyId}_{$timestamp}";
        $empFolder = "{$employee->first_name}_{$employee->last_name}_{$employee->employee_id}";

        $zipFileName = "Evidence_{$mainFolderName}.zip";
        $zipPath = storage_path('app/public/temp_zips/' . $zipFileName);

        // Ensure temp dir exists
        if (!file_exists(dirname($zipPath))) {
            mkdir(dirname($zipPath), 0755, true);
        }

        $zip = new \ZipArchive;
        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === TRUE) {

            if ($startPath && Storage::disk('public')->exists($startPath)) {
                $fullPath = Storage::disk('public')->path($startPath);
                $extension = pathinfo($fullPath, PATHINFO_EXTENSION);

                // Determine folder based on file type (Image vs Video logic if present in main controller? 
                // ScheduleController just puts everything in "Checkin Images". Assuming images/videos go there.)
                $zip->addFile($fullPath, "{$mainFolderName}/{$empFolder}/Checkin Images/checkin_0.{$extension}");
            }

            if ($endPath && Storage::disk('public')->exists($endPath)) {
                $fullPath = Storage::disk('public')->path($endPath);
                $extension = pathinfo($fullPath, PATHINFO_EXTENSION);
                $zip->addFile($fullPath, "{$mainFolderName}/{$empFolder}/Checkout Images/checkout_0.{$extension}");
            }

            $zip->close();
        } else {
            return back()->with('error', 'Could not create zip file.');
        }

        return response()->download($zipPath)->deleteFileAfterSend(true);
    }
}
