<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ActivityLog;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class ActivityLogController extends Controller
{
    public function index(Request $request)
    {
        $query = ActivityLog::where('user_id', Auth::id())
            ->orderBy('created_at', 'desc');

        if ($request->has('range')) {
            $this->applyRangeFilter($query, $request->range);
        } else {
            // Default to today if explicitly requested or just show all paginated?
            // Usually logs show all, but prompt says "activities performed during the day".
            // Let's filter by today by default for the "day view"? 
            // "view his logs ie his activities performed during the day" -> sounds like default is Today.
            $query->whereDate('created_at', Carbon::today());
        }

        $logs = $query->paginate(20);

        return view('activity_logs.index', compact('logs'));
    }

    public function exportPdf(Request $request)
    {
        $query = ActivityLog::where('user_id', Auth::id())
            ->orderBy('created_at', 'desc');

        $range = $request->get('range', 'today');
        $this->applyRangeFilter($query, $range);

        $logs = $query->get();

        $pdf = Pdf::loadView('activity_logs.pdf', compact('logs', 'range'));
        return $pdf->download('Activity_Logs_' . $range . '_' . now()->format('Ymd') . '.pdf');
    }

    public function exportExcel(Request $request)
    {
        $query = ActivityLog::where('user_id', Auth::id())
            ->orderBy('created_at', 'desc');

        $range = $request->get('range', 'today');
        $this->applyRangeFilter($query, $range);

        $logs = $query->get();

        $filename = 'Activity_Logs_' . $range . '_' . now()->format('Ymd') . '.csv';
        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];

        $columns = ['Date', 'Action', 'Description', 'IP Address'];

        $callback = function () use ($logs, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($logs as $log) {
                fputcsv($file, [
                    $log->created_at->format('Y-m-d H:i:s'),
                    $log->action,
                    $log->description,
                    $log->ip_address
                ]);
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function applyRangeFilter($query, $range)
    {
        switch ($range) {
            case 'today':
                $query->whereDate('created_at', Carbon::today());
                break;
            case 'weekly':
                $query->whereBetween('created_at', [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()]);
                break;
            case 'monthly':
                $query->whereMonth('created_at', Carbon::now()->month)
                    ->whereYear('created_at', Carbon::now()->year);
                break;
            // Add 'all' if needed
        }
    }
}
