<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

class UpdateScheduleStatuses extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:update-schedule-statuses';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Updates schedule statuses to today, missed, or upcoming based on current date';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        // Get all distinct time zones from active/upcoming/today schedules to minimize queries
        // We only care about pending schedules
        $timeZones = \App\Models\Schedule::whereNotIn('status', ['completed', 'missed'])
            ->distinct()
            ->pluck('time_zone')
            ->filter()
            ->values();

        // If no timezones found (e.g. empty table or all completed), we might still want to default to app timezone or skip
        if ($timeZones->isEmpty()) {
            // Fallback for rows with null timezone? Assumed UTC or Config?
            // Let's explicitly add UTC if not present just in case
            $timeZones->push(config('app.timezone'));
        }

        foreach ($timeZones as $tz) {
            try {
                $now = \Carbon\Carbon::now($tz);
                $today = \Carbon\Carbon::today($tz); // Midnight of that TZ
            } catch (\Exception $e) {
                // Fallback if invalid TZ
                $now = \Carbon\Carbon::now();
                $today = \Carbon\Carbon::today();
            }

            // 1. Mark as 'missed' using Local NOW vs Local End Time
            \App\Models\Schedule::where('time_zone', $tz)
                ->where('to_datetime', '<', $now)
                ->whereNotIn('status', ['active', 'completed', 'missed'])
                ->update(['status' => 'missed']);

            // 2. Mark as 'today' using Local Date vs Local Start Date AND Local NOW vs Local End Time
            // (Start is today, AND it hasn't finished yet)
            \App\Models\Schedule::where('time_zone', $tz)
                ->whereDate('from_datetime', $today)
                ->where('to_datetime', '>=', $now)
                ->whereNotIn('status', ['active', 'completed', 'today', 'missed'])
                ->update(['status' => 'today']);

            // 3. Mark as 'upcoming' using Local Start Date > Local Today
            \App\Models\Schedule::where('time_zone', $tz)
                ->whereDate('from_datetime', '>', $today)
                ->whereNotIn('status', ['active', 'completed', 'upcoming'])
                ->update(['status' => 'upcoming']);
        }

        // Handle schedules with NULL timezone (Default to UTC or App Timezone)
        $defaultTz = config('app.timezone');
        $now = \Carbon\Carbon::now($defaultTz);
        $today = \Carbon\Carbon::today($defaultTz);

        \App\Models\Schedule::whereNull('time_zone')
            ->where('to_datetime', '<', $now)
            ->whereNotIn('status', ['active', 'completed', 'missed'])
            ->update(['status' => 'missed']);

        \App\Models\Schedule::whereNull('time_zone')
            ->whereDate('from_datetime', $today)
            ->where('to_datetime', '>=', $now)
            ->whereNotIn('status', ['active', 'completed', 'today', 'missed'])
            ->update(['status' => 'today']);

        \App\Models\Schedule::whereNull('time_zone')
            ->whereDate('from_datetime', '>', $today)
            ->whereNotIn('status', ['active', 'completed', 'upcoming'])
            ->update(['status' => 'upcoming']);

        $this->info('Schedule statuses updated successfully (Timezone Aware).');
    }
}
