<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Employee;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Illuminate\Support\Facades\File;

class ImportEmployees extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'import:employees {--file=public/users-5.csv : Path to the CSV file}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import employees from a CSV file into the database';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $filePath = $this->option('file');

        if (!File::exists($filePath)) {
            $this->error("File does not exist at path: $filePath");
            return 1;
        }

        $this->info("Starting import from $filePath...");

        $handle = fopen($filePath, 'r');
        if (!$handle) {
            $this->error("Could not open file: $filePath");
            return 1;
        }

        // Skip Header
        fgetcsv($handle);

        $count = 0;
        $updated = 0;
        $created = 0;

        while (($data = fgetcsv($handle)) !== false) {
            $count++;

            // Map CSV Data (Indices from generate_sql.php analysis)
            // 0: Email
            // 1: Employee ID
            // 2: First Name
            // 3: Last Name
            // 4: Gender
            // 5: DOB
            // 6: Employment Date
            // 7: Phone
            // 8: Perm Addr
            // 9: Corr Addr
            // 10: SIN Num
            // 11: Company ID (int)
            // 12: file_license (upload_license)
            // 13: upload_id (Ignored)
            // 14: License Number
            // 15: License Expiry
            // 16: Has Car (str)
            // 17-19: Wage (Ignored)
            // 20: file_contract (contract)

            if (count($data) < 21) {
                $this->warn("Skipping row $count due to insufficent columns.");
                continue;
            }

            try {
                $employeeId = trim($data[1]);
                $email = trim($data[0]);

                if (empty($employeeId) || empty($email)) {
                    $this->warn("Skipping row $count due to missing ID or Email.");
                    continue;
                }

                $dob = $this->parseDate($data[5]);
                $empDate = $this->parseDate($data[6]);
                $licExp = $this->parseDate($data[15]);

                $hasCarStr = strtolower(trim($data[16]));
                $hasCar = ($hasCarStr === 'yes') ? true : false;

                $empData = [
                    'company_id' => (int) $data[11],
                    'first_name' => trim($data[2]),
                    'last_name' => trim($data[3]),
                    'gender' => trim($data[4]),
                    'email' => $email,
                    'dob' => $dob,
                    'employment_date' => $empDate,
                    'phone_number' => trim($data[7]),
                    'permanent_address_line_1' => trim($data[8]),
                    'corresponding_address_line_1' => trim($data[9]),
                    'sin_number' => trim($data[10]),
                    'file_license' => trim($data[12]),
                    // upload_id ignored at index 13
                    'license_number' => trim($data[14]),
                    'license_expiry' => $licExp,
                    'has_car' => $hasCar, // Assuming 'has_car' column exists in migration/model to match SQL
                    // SQL has 'has_car', model might use 'metadata' or specific column. 
                    // I checked migration, column didn't explicitly show 'has_car'?
                    // Wait. In Step 958 migration view, I saw:
                    // 1-111 truncated? No, complete file.
                    // I didn't see 'has_car' column in lines 13-70!
                    // Let me double check migration content from Step 958.
                    // line 52: file_license
                    // line 53: file_contract
                    // line 56: reporting_manager
                    // No 'has_car'.
                    // Step 990 SQL output has `has_car` in INSERT statement (col index 16).
                    // This implies the DATABASE has the column, regardless of migration listing?
                    // Or maybe I missed it in migration view?
                    // I'll re-read migration snippet (Step 958).
                    // Lines 13-70.
                    // I see `file_license`, `file_contract`.
                    // I see `sin_number`.
                    // I DO NOT see `has_car`.
                    // BUT my generate_sql.php used it. And user approved it?
                    // If I try to insert `has_car` via Eloquent and column doesn't exist -> Error.
                    // If I generated SQL with it, maybe the user has a migration I didn't see?
                    // Or maybe I hallucinatae `has_car` in SQL?
                    // Step 990: `INSERT INTO ... file_contract, created_at) VALUES ...`
                    // Wait. Step 990 output Line 1:
                    // `... license_expiry, has_car, file_contract ...`
                    // IT IS THERE in SQL.
                    // So DB must have it.
                    // Maybe migration snippet was truncated/old? Or another file?
                    // `2025_12_27_110000_...`.
                    // Maybe `add_has_car_to_employees`?
                    // I'll assume it exists. If Eloquent fails, I'll know.
                    'file_contract' => trim($data[20]),
                    'password' => Hash::make($employeeId),
                ];

                // Upsert based on employee_id
                $employee = Employee::updateOrCreate(
                    ['employee_id' => $employeeId],
                    $empData
                );

                if ($employee->wasRecentlyCreated) {
                    $created++;
                } else {
                    $updated++;
                }

            } catch (\Exception $e) {
                $this->error("Error row $count: " . $e->getMessage());
            }
        }

        fclose($handle);

        $this->info("Import completed.");
        $this->info("Created: $created");
        $this->info("Updated: $updated");

        return 0;
    }

    private function parseDate($dateStr)
    {
        if (empty($dateStr) || $dateStr === 'NULL')
            return null;
        try {
            return Carbon::createFromFormat('d-m-Y', $dateStr)->format('Y-m-d');
        } catch (\Exception $e) {
            return null; // Fallback or log check
        }
    }
}
