<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Schedule;
use App\Models\PatrollerSchedule;

class AutoCompleteOverdueJobs extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'jobs:auto-complete-overdue';

    protected $description = 'Automatically complete jobs that have exceeded their scheduled end time';

    public function handle()
    {
        $now = \Carbon\Carbon::now();
        $this->info("Checking for overdue jobs at {$now->format('Y-m-d H:i:s')}...");

        // 1. Handle Guard Schedules
        $overdueSchedules = Schedule::where('status', 'active')
            ->where('to_datetime', '<', $now)
            ->with([
                'employees' => function ($query) {
                    $query->wherePivot('actual_start_at', '!=', null)
                        ->wherePivotNull('actual_end_at');
                }
            ])
            ->get();

        $completedSchedulesCount = 0;
        $guardEmployeeCount = 0;

        foreach ($overdueSchedules as $schedule) {
            foreach ($schedule->employees as $employee) {
                // Set actual_end_at to scheduled end time
                $schedule->employees()->updateExistingPivot($employee->id, [
                    'actual_end_at' => $schedule->to_datetime
                ]);

                $guardEmployeeCount++;
                $this->info("  ✓ Guard Auto-completed: {$employee->first_name} {$employee->last_name} - Duty #{$schedule->duty_number}");
            }

            // Check if all employees have completed
            $allCompleted = $schedule->employees()
                ->wherePivot('actual_start_at', '!=', null)
                ->wherePivotNull('actual_end_at')
                ->count() === 0;

            if ($allCompleted && $schedule->employees()->wherePivot('actual_start_at', '!=', null)->count() > 0) {
                $schedule->update(['status' => 'completed']);
                $completedSchedulesCount++;
                $this->info("  ✓ Guard Schedule #{$schedule->duty_number} marked as completed");
            }
        }

        // 2. Handle Patroller Schedules
        $overduePatrollerSchedules = PatrollerSchedule::where('job_status', 'in_progress')
            ->where('to_time', '<', $now)
            ->with([
                'employees' => function ($query) {
                    $query->wherePivot('actual_start_at', '!=', null)
                        ->wherePivotNull('actual_end_at');
                }
            ])
            ->get();

        $completedPatrollerSchedulesCount = 0;
        $patrollerEmployeeCount = 0;

        foreach ($overduePatrollerSchedules as $pSchedule) {
            foreach ($pSchedule->employees as $employee) {
                // Set actual_end_at to scheduled end time
                $pSchedule->employees()->updateExistingPivot($employee->id, [
                    'actual_end_at' => $pSchedule->to_time
                ]);

                $patrollerEmployeeCount++;
                $this->info("  ✓ Patroller Auto-completed: {$employee->first_name} {$employee->last_name} - Duty #{$pSchedule->duty_number}");
            }

            // Check if all employees have completed
            $allCompleted = $pSchedule->employees()
                ->wherePivot('actual_start_at', '!=', null)
                ->wherePivotNull('actual_end_at')
                ->count() === 0;

            if ($allCompleted && $pSchedule->employees()->wherePivot('actual_start_at', '!=', null)->count() > 0) {
                $pSchedule->update(['job_status' => 'auto_ended']);
                $completedPatrollerSchedulesCount++;
                $this->info("  ✓ Patroller Schedule #{$pSchedule->duty_number} marked as auto_ended");
            }
        }

        $this->info("Guard Summary: {$guardEmployeeCount} employee jobs across {$completedSchedulesCount} schedules.");
        $this->info("Patroller Summary: {$patrollerEmployeeCount} employee jobs across {$completedPatrollerSchedulesCount} schedules.");

        return 0;
    }
}
