# Employee Job Management API Documentation

This document describes the complete job lifecycle management APIs for the mobile application, matching all web panel validations and business logic.

## Overview

The job management APIs allow employees to:
1. Start a job (with visibility and concurrency checks)
2. Upload check-in evidence
3. Report incidents during active jobs
4. Upload checkout evidence
5. End a job (with evidence validation)
6. Delete evidence images

All endpoints require authentication via `Bearer` token in the `Authorization` header.

---

## API Endpoints

### 1. Start Job
**POST** `/api/employee/jobs/{id}/start`

Initiates a job with comprehensive validation matching the web panel.

#### Request Parameters
```json
{
  "latitude": 31.5204,  // Optional: Employee's current latitude
  "longitude": 74.3587  // Optional: Employee's current longitude
}
```

#### Validations
- **Visibility Check**: Job must be within the visibility window (configurable per employee)
- **Concurrent Job Check**: Employee cannot have another active job
- **Expiry Check**: Job's end time must be in the future
- **Already Active Check**: Job cannot be started twice

#### Success Response (200)
```json
{
  "status": "success",
  "message": "Job initiated. Deployment is now ACTIVE. Please submit check-in evidence.",
  "data": {
    "job_id": 123,
    "duty_number": "DTY0377",
    "started_at": "2026-01-26T12:01:20+05:30"
  }
}
```

#### Error Responses

**Visibility Window Not Met (422)**
```json
{
  "status": "error",
  "message": "Job is not yet cleared for initiation.",
  "details": {
    "scheduled": "2026-01-26T14:00:00+05:30",
    "deadline": "2026-01-26T13:30:00+05:30",
    "window_minutes": 30
  }
}
```

**Concurrent Job Active (422)**
```json
{
  "status": "error",
  "message": "You are already active on Duty #DTY0376. Please end that duty before starting a new one.",
  "active_job": {
    "id": 122,
    "duty_number": "DTY0376",
    "from_time": "10:00",
    "to_time": "18:00"
  }
}
```

---

### 2. Upload Check-in Evidence
**POST** `/api/employee/jobs/{id}/checkin`

Upload check-in images after starting a job.

#### Request Parameters
```
Content-Type: multipart/form-data

images[]: File (required, min: 1 image, max: 10MB per image)
```

#### Success Response (200)
```json
{
  "status": "success",
  "message": "Check-in evidence successfully archived. Job profile updated.",
  "data": {
    "images_count": 3,
    "images": [
      "images/Gleichner_PLC_Site_2026-01-26_12-01-20_DTY0377/checkin_images/photo1.jpg",
      "images/Gleichner_PLC_Site_2026-01-26_12-01-20_DTY0377/checkin_images/photo2.jpg",
      "images/Gleichner_PLC_Site_2026-01-26_12-01-20_DTY0377/checkin_images/photo3.jpg"
    ]
  }
}
```

---

### 3. Report Incident
**POST** `/api/employee/jobs/{id}/incident`

Report an incident during an active job.

#### Request Parameters
```
Content-Type: multipart/form-data

subject: string (required, max: 255 characters)
description: string (required)
incident_parent_category_id: integer (required, must exist)
incident_category_id: integer (required, must exist)
images[]: File (optional, max: 5MB per image)
```

#### Success Response (200)
```json
{
  "status": "success",
  "message": "Incident report filed and prioritized for review.",
  "data": {
    "incident_id": 45,
    "images_count": 2
  }
}
```

---

### 4. Upload Checkout Evidence
**POST** `/api/employee/jobs/{id}/checkout-evidence`

Upload checkout/extraction evidence before ending a job.

#### Request Parameters
```
Content-Type: multipart/form-data

images[]: File (required, min: 1 image, max: 5MB per image)
```

#### Notes
- This endpoint **appends** images to existing checkout evidence
- Multiple uploads are allowed before ending the job

#### Success Response (200)
```json
{
  "status": "success",
  "message": "Checkout images uploaded successfully.",
  "data": {
    "total_images": 5,
    "new_images": 2,
    "images": [
      "images/.../checkout_images/photo1.jpg",
      "images/.../checkout_images/photo2.jpg",
      "images/.../checkout_images/photo3.jpg",
      "images/.../checkout_images/photo4.jpg",
      "images/.../checkout_images/photo5.jpg"
    ]
  }
}
```

---

### 5. End Job
**POST** `/api/employee/jobs/{id}/end`

Terminate a job with evidence validation.

#### Request Parameters
```json
{
  "latitude": 31.5204,  // Optional: Employee's current latitude
  "longitude": 74.3587  // Optional: Employee's current longitude
}
```

#### Validations
- **Check-in Evidence Required**: At least one check-in image must be uploaded
- **Checkout Evidence Required**: At least one checkout image must be uploaded

#### Success Response (200)
```json
{
  "status": "success",
  "message": "Job Terminated. Extraction complete.",
  "data": {
    "job_id": 123,
    "duty_number": "DTY0377",
    "ended_at": "2026-01-26T18:30:00+05:30",
    "job_status": "completed"
  }
}
```

#### Error Response - Missing Evidence (422)
```json
{
  "status": "error",
  "message": "Job Clearance Denied",
  "errors": [
    "Infiltration Evidence (Check-in Photos) missing.",
    "Extraction Evidence (Checkout Photos) missing."
  ]
}
```

---

### 6. Delete Evidence
**DELETE** `/api/employee/jobs/{id}/evidence/{type}/{index}`

Delete a specific evidence image.

#### URL Parameters
- `id`: Job ID
- `type`: Evidence type (`checkin` or `checkout`)
- `index`: Image index (0-based)

#### Example
```
DELETE /api/employee/jobs/123/evidence/checkin/0
```

#### Success Response (200)
```json
{
  "status": "success",
  "message": "Evidence deleted successfully."
}
```

---

## Job Lifecycle Flow

```
1. GET /api/employee/jobs?type=today
   ↓ (Get today's jobs)
   
2. POST /api/employee/jobs/{id}/start
   ↓ (Start the job - validates visibility, concurrency, expiry)
   
3. POST /api/employee/jobs/{id}/checkin
   ↓ (Upload check-in evidence - required before ending)
   
4. POST /api/employee/jobs/{id}/incident (Optional, can be called multiple times)
   ↓ (Report incidents during the job)
   
5. POST /api/employee/jobs/{id}/checkout-evidence
   ↓ (Upload checkout evidence - required before ending)
   
6. POST /api/employee/jobs/{id}/end
   ↓ (End the job - validates all evidence is present)
   
✓ Job Completed
```

---

## Business Rules

### Visibility Window
- Each employee has a configurable visibility window (default: 30 minutes)
- Jobs can only be started within this window before the scheduled start time
- Example: If job starts at 14:00 and window is 30 mins, employee can start from 13:30

### Single Active Job Constraint
- An employee can only have ONE active job at a time
- Must end current job before starting a new one
- Prevents time-tracking conflicts

### Evidence Requirements
- **Check-in images**: Must be uploaded after starting the job
- **Checkout images**: Must be uploaded before ending the job
- Both are mandatory for job completion
- Images are stored with organized folder structure based on duty number and site

### Job Status Updates
- Starting a job sets status to `active`
- Ending a job:
  - Sets `actual_end_at` for the employee
  - If ALL employees on the job have ended, job status becomes `completed`
  - Otherwise, job remains `active` for other employees

---

## Error Handling

All endpoints follow consistent error response format:

```json
{
  "status": "error",
  "message": "Human-readable error message",
  "errors": ["Detailed error 1", "Detailed error 2"]  // Optional array
}
```

Common HTTP status codes:
- `200`: Success
- `404`: Job not found or not assigned to employee
- `422`: Validation error or business rule violation
- `500`: Server error

---

## Testing Examples

### Using cURL

**Start Job:**
```bash
curl -X POST http://192.168.29.141:8000/api/employee/jobs/2/start \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"latitude": 31.5204, "longitude": 74.3587}'
```

**Upload Check-in:**
```bash
curl -X POST http://192.168.29.141:8000/api/employee/jobs/2/checkin \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -F "images[]=@/path/to/photo1.jpg" \
  -F "images[]=@/path/to/photo2.jpg"
```

**End Job:**
```bash
curl -X POST http://192.168.29.141:8000/api/employee/jobs/2/end \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"latitude": 31.5204, "longitude": 74.3587}'
```

---

## Notes for Mobile Developers

1. **Location Permissions**: Request location permissions for start/end job endpoints
2. **Camera Permissions**: Required for check-in, checkout, and incident evidence
3. **Image Compression**: Consider compressing images before upload to save bandwidth
4. **Offline Support**: Queue job actions when offline and sync when online
5. **Real-time Updates**: Consider implementing WebSocket/FCM for job status updates
6. **Evidence Preview**: Show uploaded evidence to users before submission
7. **Validation Feedback**: Display clear error messages from API responses
