@extends('layouts.user')

@section('title', 'Edit Guard Schedule')

@section('content')
    <div class="p-6" x-data="scheduleForm()" x-init="init()">
        <div class="flex justify-between items-center mb-6">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">Edit Duty Schedule</h1>
                <p class="text-gray-500">Update duty details for #{{ $schedule->duty_number }}</p>
            </div>
            <a href="{{ route('schedules.index') }}"
                class="px-6 py-2 bg-gray-100 text-gray-600 font-bold rounded-lg hover:bg-gray-200 transition-all">
                Back to List
            </a>
        </div>

        <!-- Meta Alerts Container -->
        <div class="space-y-4 mb-6">
            <!-- License Expiry Alert Section -->
            <template x-for="emp in selectedEmployeesData.filter(e => e.is_license_expired)" :key="'exp-' + emp.id">
                <div class="p-6 bg-amber-50 border-2 border-amber-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-amber-500 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-amber-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                </path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-amber-800 mb-1">Warning: Guard License Expired!</h3>
                            <p class="text-amber-600 text-sm mb-4 font-medium"
                                x-text="'The security license for ' + emp.name + ' expired on ' + emp.license_expiry + '.'">
                            </p>
                            <button type="button" @click="toggleEmployee(emp.id)"
                                class="px-4 py-2 bg-amber-600 text-white text-xs font-bold rounded-lg hover:bg-amber-700">Remove
                                This Guard</button>
                        </div>
                    </div>
                </div>
            </template>

            <!-- Conflict Alert Section -->
            <template x-if="allConflicts.length > 0">
                <div class="p-6 bg-rose-50 border-2 border-rose-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-rose-500 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-rose-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                </path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-rose-800 mb-1">Schedule Conflict Detected!</h3>
                            <p class="text-rose-600 text-sm mb-4 font-medium">One or more selected guards are already
                                scheduled during this period.</p>
                            <div class="space-y-3 max-h-40 overflow-y-auto pr-2">
                                <template x-for="c in allConflicts" :key="c.duty_number + c.guard_name">
                                    <div
                                        class="bg-white/80 p-3 rounded-xl flex items-center justify-between border border-rose-200">
                                        <div>
                                            <div class="text-[10px] font-bold text-rose-500 uppercase"
                                                x-text="c.guard_name"></div>
                                            <div class="text-sm font-bold text-gray-900" x-text="c.site_name"></div>
                                        </div>
                                        <div class="text-right">
                                            <div class="text-[10px] text-gray-500" x-text="c.from + ' - ' + c.to"></div>
                                            <div class="text-[10px] font-bold text-gray-400" x-text="c.duty_number"></div>
                                        </div>
                                    </div>
                                </template>
                            </div>
                        </div>
                    </div>
                </div>
            </template>

            <!-- Max Shift Hours Alert Section -->
            <template x-if="isUserRole && calculateDuration() > maxShiftHours">
                <div class="p-6 bg-red-50 border-2 border-red-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-red-600 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-red-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z">
                                </path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-red-800 mb-1">Shift Duration Error!</h3>
                            <p class="text-red-600 text-sm font-medium"
                                x-text="'Total shift duration (' + getDurationString() + ') exceeds the maximum allowed shift hours of ' + maxShiftHours + ' hours.'">
                            </p>
                        </div>
                    </div>
                </div>
            </template>

            <!-- Validation Errors Section -->
            @if ($errors->any())
                <div class="p-6 bg-red-50 border-2 border-red-100 rounded-3xl animate-in fade-in zoom-in duration-300">
                    <div class="flex items-start gap-4">
                        <div
                            class="w-12 h-12 bg-red-500 rounded-2xl flex items-center justify-center text-white shadow-lg shadow-red-200 flex-shrink-0">
                            <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                        </div>
                        <div class="flex-1">
                            <h3 class="text-lg font-bold text-red-800 mb-2">Input Validation Errors</h3>
                            <ul class="list-disc list-inside text-red-600 text-sm space-y-1 font-medium">
                                @foreach ($errors->all() as $error)
                                    <li>{{ $error }}</li>
                                @endforeach
                            </ul>
                        </div>
                    </div>
                </div>
            @endif
        </div>

        <form action="{{ route('schedules.update', $schedule) }}" method="POST"
            class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            @csrf
            @method('PUT')

            <!-- Left Column: Form Fields -->
            <div class="lg:col-span-2 space-y-6">
                <!-- Site & Timing Section -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-8">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Duty # -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Duty Number</label>
                            <input type="text" value="{{ $schedule->duty_number }}" readonly
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl text-gray-500 font-mono font-bold outline-none cursor-not-allowed">
                        </div>

                        <!-- Schedule Date -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Schedule Date <span
                                    class="text-red-500">*</span></label>
                            <input type="date" name="schedule_date" x-model="scheduleDate" @change="fetchEmployeesInfo"
                                required
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                        </div>

                        <!-- Select Site (Searchable) -->
                        <div class="md:col-span-2 relative" @click.away="siteSearchOpen = false">
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Select Site <span
                                    class="text-red-500">*</span></label>
                            <div class="relative">
                                <input type="text" x-model="siteSearch" @focus="siteSearchOpen = true; $el.select()"
                                    @input="siteSearchOpen = true" placeholder="Search by Site Name or ID..."
                                    class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                                <input type="hidden" name="site_id" :value="selectedSiteId" required>

                                <!-- Dropdown Results -->
                                <div x-show="siteSearchOpen" x-cloak
                                    class="absolute z-50 w-full mt-2 bg-white border border-gray-100 rounded-2xl shadow-2xl max-h-64 overflow-y-auto p-2 space-y-1 animate-in fade-in zoom-in duration-200">
                                    <template x-for="site in getFilteredSites()" :key="site.id">
                                        <button type="button" @click="selectSite(site)"
                                            class="w-full text-left px-4 py-3 hover:bg-blue-50 rounded-xl transition-all group">
                                            <div class="flex items-center justify-between">
                                                <div>
                                                    <div class="text-sm font-bold text-gray-900 group-hover:text-blue-700"
                                                        x-text="site.name"></div>
                                                    <div class="text-[10px] text-gray-400 font-bold uppercase tracking-wider"
                                                        x-text="'ID: ' + site.site_id"></div>
                                                </div>
                                                <div x-show="selectedSiteId == site.id">
                                                    <svg class="w-5 h-5 text-blue-500" fill="none" stroke="currentColor"
                                                        viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round"
                                                            stroke-width="3" d="M5 13l4 4L19 7"></path>
                                                    </svg>
                                                </div>
                                            </div>
                                        </button>
                                    </template>
                                    <div x-show="getFilteredSites().length === 0"
                                        class="p-4 text-center text-gray-400 text-sm italic">
                                        No sites found matching "<span x-text="siteSearch"></span>"
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Site Rate -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Site Rate</label>
                            <div class="relative">
                                <span class="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 font-bold">$</span>
                                <input type="number" step="0.01" name="site_rate" x-model="siteRate" required
                                    class="w-full pl-8 pr-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                            </div>
                        </div>

                        <!-- Assign Guards -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Assign Guards <span
                                    class="text-red-500">*</span></label>
                            <div class="relative" x-data="{ open: false }" @click.away="open = false">
                                <button type="button" @click="open = !open"
                                    class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl text-left flex items-center justify-between">
                                    <span class="text-sm font-bold text-gray-700"
                                        x-text="selectedEmployeeIds.length > 0 ? selectedEmployeeIds.length + ' Selected' : 'Choose Guard(s)'"></span>
                                    <svg class="w-4 h-4 text-gray-400 transition-transform"
                                        :class="open ? 'rotate-180' : ''" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M19 9l-7 7-7-7"></path>
                                    </svg>
                                </button>
                                <div x-show="open" x-cloak
                                    class="absolute z-10 w-full mt-2 bg-white border border-gray-100 rounded-2xl shadow-2xl max-h-60 overflow-y-auto p-2 space-y-1">
                                    @foreach($employees as $employee)
                                        <label
                                            class="flex items-center gap-3 p-2 hover:bg-gray-50 rounded-lg cursor-pointer transition-colors"
                                            :class="selectedEmployeeIds.includes('{{ $employee->id }}') ? 'bg-blue-50' : ''">
                                            <input type="checkbox" value="{{ $employee->id }}" x-model="selectedEmployeeIds"
                                                class="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500">
                                            <div class="flex-1">
                                                <div class="text-sm font-bold text-gray-900">{{ $employee->first_name }}
                                                    {{ $employee->last_name }}
                                                </div>
                                                <div class="text-[10px] text-gray-400 font-semibold uppercase tracking-wider">
                                                    {{ $employee->employee_id }}
                                                </div>
                                            </div>
                                        </label>
                                    @endforeach
                                </div>
                            </div>
                        </div>

                        <!-- From & To Date/Time -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">From Date & Time <span
                                    class="text-red-500">*</span></label>
                            <input type="datetime-local" x-model="fromDatetime" @change="fetchEmployeesInfo"
                                name="from_datetime" required
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                        </div>
                        <div>
                            <div class="flex items-center justify-between mb-2">
                                <label class="block text-xs font-bold text-gray-400 uppercase">To Date & Time <span
                                        class="text-red-500">*</span></label>
                                <template x-if="getDurationString()">
                                    <span
                                        class="text-[10px] font-black px-2 py-0.5 bg-blue-50 text-blue-600 rounded-lg uppercase tracking-wider animate-in fade-in slide-in-from-right-2"
                                        x-text="'Duration: ' + getDurationString()"></span>
                                </template>
                                <template
                                    x-if="fromDatetime && toDatetime && new Date(toDatetime) <= new Date(fromDatetime)">
                                    <span
                                        class="text-[10px] font-black px-2 py-0.5 bg-red-50 text-red-600 rounded-lg uppercase tracking-wider animate-pulse">
                                        End time must be after start time
                                    </span>
                                </template>
                            </div>
                            <input type="datetime-local" x-model="toDatetime" @change="fetchEmployeesInfo"
                                name="to_datetime" required
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold"
                                :class="fromDatetime && toDatetime && new Date(toDatetime) <= new Date(fromDatetime) ? 'border-red-300 ring-red-50' : ''">
                        </div>

                        <!-- Time Zone -->
                        <div>
                            <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Time Zone</label>
                            <select name="time_zone"
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                                @foreach($timeZones as $tz)
                                    <option value="{{ $tz->name }}" {{ $schedule->time_zone == $tz->name ? 'selected' : '' }}>
                                        {{ $tz->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>
                </div>

                <!-- Guard Individual Settings Section -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-8">
                    <h2 class="text-lg font-bold text-gray-800 mb-6 flex items-center gap-2">
                        <svg class="w-5 h-5 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z">
                            </path>
                        </svg>
                        Guard Individual Pricing & Wages
                    </h2>

                    <div class="space-y-8">
                        <template x-for="(emp, empIndex) in selectedEmployeesData" :key="'cfg-' + emp.id">
                            <div class="p-6 bg-gray-50 rounded-3xl border border-gray-100 relative overflow-hidden">
                                <div class="relative z-10">
                                    <div class="flex items-center gap-4 mb-6">
                                        <img :src="emp.profile_picture"
                                            class="w-12 h-12 rounded-full object-cover ring-4 ring-white shadow-sm">
                                        <div class="flex-1">
                                            <div class="font-bold text-gray-900" x-text="emp.name"></div>
                                            <div class="flex items-center gap-2">
                                                <div class="text-[10px] text-gray-400 font-bold uppercase tracking-widest"
                                                    x-text="emp.employee_id"></div>
                                                <template x-if="emp.stat_holiday">
                                                    <span
                                                        class="px-2 py-0.5 bg-amber-100 text-amber-700 text-[9px] font-black rounded-lg uppercase tracking-tight border border-amber-200"
                                                        :title="'Stat Holiday: ' + emp.stat_holiday.name + ' (' + emp.stat_holiday.multiplier + 'x Multiplier)'">
                                                        Stat Holiday: <span x-text="emp.stat_holiday.name"></span>
                                                    </span>
                                                </template>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6" x-show="calculateDuration() > 0">
                                        <!-- GAS Rate -->
                                        <div class="md:col-span-2">
                                            <label class="block text-[10px] font-bold text-gray-400 uppercase mb-2">GAS Rate
                                                (Optional)</label>
                                            <div class="relative">
                                                <span
                                                    class="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 font-bold">$</span>
                                                <input type="number" step="0.01"
                                                    :name="'employees[' + emp.id + '][gas_rate]'" x-model="emp.gas_rate"
                                                    class="w-full pl-8 pr-4 py-3 bg-white border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                                            </div>
                                        </div>

                                        <!-- Wage Types for this employee -->
                                        <div class="md:col-span-2">
                                            <label
                                                class="block text-[10px] font-black text-slate-400 uppercase tracking-[0.2em] mb-4">Select
                                                Wage Types</label>
                                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                <template x-for="(wt, wtIndex) in emp.wage_types"
                                                    :key="wt.virtual_id || wt.id">
                                                    <div class="p-6 bg-white border border-slate-100 rounded-[2rem] shadow-sm hover:shadow-md transition-all space-y-4"
                                                        :class="wt.selected ? 'ring-2 ring-blue-500/10 border-blue-500/20' : 'opacity-60'">
                                                        <div class="flex items-center justify-between">
                                                            <div class="flex items-center gap-3">
                                                                <input type="checkbox" x-model="wt.selected"
                                                                    class="w-6 h-6 rounded-lg border-slate-300 text-blue-600 focus:ring-blue-500/20 transition-all cursor-pointer">
                                                                <div>
                                                                    <p class="text-xs font-black text-slate-900 uppercase tracking-tight"
                                                                        x-text="wt.name"></p>
                                                                    <template
                                                                        x-if="wt.virtual_id && wt.virtual_id.includes('_hol')">
                                                                        <span
                                                                            class="text-[8px] font-black px-1.5 py-0.5 bg-amber-100 text-amber-700 rounded-md uppercase tracking-tighter">Stat
                                                                            Multiplier Applied</span>
                                                                    </template>
                                                                    <p class="text-[10px] font-bold text-slate-400 uppercase tracking-widest mt-0.5"
                                                                        x-text="'ID: ' + wt.id"></p>
                                                                </div>
                                                            </div>
                                                            <div class="w-24 relative">
                                                                <span
                                                                    class="absolute left-3 top-1/2 -translate-y-1/2 text-[10px] font-black text-slate-300">$</span>
                                                                <input type="number" step="0.01" x-model="wt.rate"
                                                                    class="w-full pl-6 pr-3 py-2 bg-slate-50 border border-slate-100 rounded-xl text-xs font-black text-slate-900 focus:ring-4 focus:ring-blue-500/5 focus:border-blue-500 outline-none transition-all placeholder:text-slate-200"
                                                                    placeholder="Rate">
                                                            </div>
                                                        </div>

                                                        <!-- Hour Input if multiple are selected OR multiple guards are present -->
                                                        <template
                                                            x-if="wt.selected && (emp.wage_types.filter(w => w.selected).length > 1 || selectedEmployeeIds.length > 1)">
                                                            <div
                                                                class="pt-4 border-t border-slate-50 flex items-center justify-between gap-4 animate-in slide-in-from-top-2 duration-300">
                                                                <label
                                                                    class="text-[10px] font-black text-slate-400 uppercase">Hours
                                                                    to Calculate</label>
                                                                <div class="relative w-28">
                                                                    <input type="number" step="0.1"
                                                                        x-model="wt.allocated_hours"
                                                                        class="w-full px-4 py-2 bg-blue-50/50 border border-blue-100 rounded-xl text-xs font-black text-blue-900 focus:ring-4 focus:ring-blue-500/5 focus:border-blue-500 outline-none transition-all"
                                                                        placeholder="0.0">
                                                                    <span
                                                                        class="absolute right-3 top-1/2 -translate-y-1/2 text-[10px] font-black text-blue-300">HRS</span>
                                                                </div>
                                                            </div>
                                                        </template>

                                                        <!-- Hidden inputs for JSON/Pivot storage -->
                                                        <template x-if="wt.selected">
                                                            <div>
                                                                <input type="hidden"
                                                                    :name="'employees[' + emp.id + '][wage_types][' + wtIndex + '][id]'"
                                                                    :value="wt.id">
                                                                <input type="hidden"
                                                                    :name="'employees[' + emp.id + '][wage_types][' + wtIndex + '][rate]'"
                                                                    :value="wt.rate">
                                                                <input type="hidden"
                                                                    :name="'employees[' + emp.id + '][wage_types][' + wtIndex + '][name]'"
                                                                    :value="wt.name">
                                                                <input type="hidden"
                                                                    :name="'employees[' + emp.id + '][wage_types][' + wtIndex + '][allocated_hours]'"
                                                                    :value="wt.allocated_hours">
                                                            </div>
                                                        </template>
                                                    </div>
                                                </template>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <input type="hidden" name="employee_ids[]" :value="emp.id">
                            </div>
                    </div>
                    </template>
                </div>
            </div>

            <!-- Footer Remarks -->
            <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-8">
                <div class="grid grid-cols-1 gap-6">
                    <div>
                        <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Customer Reference</label>
                        <input type="text" name="customer_reference" value="{{ $schedule->customer_reference }}"
                            class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold">
                    </div>
                    <div>
                        <label class="block text-xs font-bold text-gray-400 uppercase mb-2">Comments / Special
                            Instructions</label>
                        <textarea name="comments" rows="3"
                            class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-medium">{{ $schedule->comments }}</textarea>
                    </div>
                </div>

                <!-- Hour Allocation Validation -->
                <!-- Hour Allocation Validation Removed -->

                <div class="mt-10 pt-8 border-t border-gray-100 flex gap-4">
                    <button type="submit"
                        class="px-10 py-4 bg-blue-600 text-white font-bold rounded-2xl hover:bg-blue-700 shadow-xl shadow-blue-500/25 transition-all active:scale-95 disabled:opacity-50 disabled:cursor-not-allowed disabled:hover:bg-blue-600"
                        :disabled="(fromDatetime && toDatetime && new Date(toDatetime) <= new Date(fromDatetime)) || (isUserRole && calculateDuration() > maxShiftHours)">Update
                        Duty Schedule</button>
                    <a href="{{ route('schedules.index') }}"
                        class="px-8 py-4 bg-gray-100 text-gray-500 font-bold rounded-2xl hover:bg-gray-200 transition-all text-center">Cancel</a>
                </div>
            </div>
    </div>

    <!-- Previews -->
    <div class="space-y-6">
        <!-- Site Details -->
        <div x-show="siteInfo"
            class="bg-white rounded-[2rem] shadow-sm border border-slate-100 overflow-hidden sticky top-6">
            <div class="p-6 bg-slate-800 text-white">
                <h3 class="text-xs font-bold uppercase tracking-widest opacity-75">Site Preview</h3>
            </div>
            <div class="p-8">
                <h4 class="text-xl font-bold text-gray-900 mb-2" x-text="siteInfo.name"></h4>
                <p class="text-sm text-gray-500 mb-8 flex items-start gap-2">
                    <svg class="w-4 h-4 text-gray-400 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor"
                        viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z">
                        </path>
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                            d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"></path>
                    </svg>
                    <span x-text="siteInfo.address"></span>
                </p>
            </div>

            <!-- Economics Section -->
            <div class="border-t border-slate-50" x-show="financialSummary.revenue > 0 || financialSummary.payout > 0">
                <div class="p-6 bg-slate-900">
                    <div class="flex items-center justify-between">
                        <h3 class="text-[10px] font-black text-slate-400 uppercase tracking-[0.2em]">Duty Economics
                        </h3>
                        <div class="px-2 py-1 bg-emerald-500/10 rounded-lg">
                            <span class="text-[9px] font-black text-emerald-500 uppercase tracking-wider"
                                x-text="financialSummary.marginPercent + '% Margin'"></span>
                        </div>
                    </div>
                </div>
                <div class="p-6 space-y-4">
                    <div class="flex justify-between">
                        <span class="text-[10px] font-black text-slate-400 uppercase">Gross Revenue</span>
                        <span class="text-sm font-black text-slate-900" x-text="'$' + financialSummary.revenue"></span>
                    </div>
                    <div class="flex justify-between items-center">
                        <div class="flex items-center gap-2">
                            <span class="text-[10px] font-black text-slate-400 uppercase">Guard Payout</span>
                            <template
                                x-if="selectedEmployeesData.some(e => e.holiday_split && e.holiday_split.holiday_hours > 0)">
                                <span
                                    class="px-1.5 py-0.5 bg-amber-500/10 text-amber-600 text-[8px] font-black rounded uppercase">Holiday
                                    Alert</span>
                            </template>
                        </div>
                        <span class="text-sm font-black text-slate-900" x-text="'$' + financialSummary.payout"></span>
                    </div>
                    <div class="pt-4 border-t border-slate-100 flex justify-between items-center">
                        <span class="text-[10px] font-black text-slate-400 uppercase">Net Margin</span>
                        <span class="text-lg font-black text-emerald-600" x-text="'$' + financialSummary.margin"></span>
                    </div>
                </div>
            </div>

            <!-- Force Preview -->
            <div class="p-6 border-t border-slate-50" x-show="selectedEmployeesData.length > 0">
                <label class="block text-[10px] font-black text-slate-400 uppercase tracking-widest mb-4">Assigned
                    Force (<span x-text="selectedEmployeesData.length"></span>)</label>
                <div class="space-y-3">
                    <template x-for="emp in selectedEmployeesData" :key="emp.id">
                        <div class="flex items-center gap-4 p-3 bg-slate-50 rounded-2xl group transition-all">
                            <img :src="emp.profile_picture" class="w-8 h-8 rounded-xl object-cover ring-2 ring-white">
                            <div class="flex-1 min-w-0">
                                <div class="text-[10px] font-black text-slate-900 uppercase truncate" x-text="emp.name">
                                </div>
                                <div class="flex items-center gap-2">
                                    <span class="text-[8px] font-black text-blue-600 uppercase"
                                        x-text="'$' + calculateGuardPayout(emp).toFixed(2)"></span>
                                    <span class="w-1 h-1 rounded-full bg-slate-300"></span>
                                    <template x-if="emp.holiday_split && emp.holiday_split.has_holiday">
                                        <span class="text-[8px] font-bold text-amber-600"
                                            x-text="'(' + emp.holiday_split.holiday_hours.toFixed(1) + 'h Hol)'"></span>
                                    </template>
                                    <span class="text-[8px] font-black text-slate-400 uppercase"
                                        x-text="getDurationString()"></span>
                                </div>
                            </div>
                            <button type="button" @click="toggleEmployee(emp.id)"
                                class="text-slate-300 hover:text-rose-500 transition-colors">
                                <svg class="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2.5"
                                        d="M6 18L18 6M6 6l12 12"></path>
                                </svg>
                            </button>
                        </div>
                    </template>
                </div>
            </div>
        </div>
    </div>
    </form>
    </div>

    @push('scripts')
        <script>
            function scheduleForm() {
                return {
                    selectedSiteId: '{{ $schedule->site_id }}',
                    siteSearch: '{{ $schedule->site->site_id }} - {{ $schedule->site->name }}',
                    siteSearchOpen: false,
                    allSites: @json($sites),
                    selectedEmployeeIds: @js($schedule->employees->pluck('id')->map(fn($id) => (string) $id)),
                    selectedEmployeesData: [],
                    siteRate: '{{ $schedule->site_rate }}',
                    siteInfo: null,
                    isInitialLoad: true,
                    baseSiteRate: '{{ $schedule->site->rate }}',
                    fromDatetime: '{{ $schedule->from_datetime->format("Y-m-d\TH:i") }}',
                    toDatetime: '{{ $schedule->to_datetime->format("Y-m-d\TH:i") }}',
                    scheduleDate: '{{ $schedule->schedule_date->format("Y-m-d") }}',
                    maxShiftHours: {{ $maxShiftHours }},
                    isUserRole: {{ auth()->user()->role === 'user' ? 'true' : 'false' }},
                    allConflicts: [],

                    getFilteredSites() {
                        if (!this.siteSearch) return this.allSites;

                        const search = this.siteSearch.toLowerCase();
                        // If it matches exactly what we have displayed for the current site, 
                        // show everything so the user sees all options on focus.
                        if (this.selectedSiteId) {
                            const currentSite = this.allSites.find(s => String(s.id) === String(this.selectedSiteId));
                            if (currentSite && this.siteSearch === `${currentSite.site_id} - ${currentSite.name}`) {
                                return this.allSites;
                            }
                        }

                        return this.allSites.filter(site => {
                            const name = (site.name || '').toLowerCase();
                            const sid = (site.site_id || '').toLowerCase();
                            return name.includes(search) || sid.includes(search);
                        });
                    },

                    selectSite(site) {
                        this.selectedSiteId = String(site.id);
                        this.siteSearch = `${site.site_id} - ${site.name}`;
                        this.siteSearchOpen = false;
                        this.baseSiteRate = site.rate;
                        this.updateSiteRate();
                        this.fetchSiteInfo();
                    },

                    calculateDuration() {
                        if (!this.fromDatetime || !this.toDatetime) return 0;
                        const start = new Date(this.fromDatetime);
                        const end = new Date(this.toDatetime);
                        const diffMs = end - start;
                        if (diffMs <= 0) return 0;
                        return diffMs / (1000 * 60 * 60); // Decimal hours
                    },

                    getDurationString() {
                        const hrs = this.calculateDuration();
                        if (!hrs) return null;
                        const h = Math.floor(hrs);
                        const m = Math.round((hrs - h) * 60);
                        return `${h}h ${m}m`;
                    },

                    getTotalAllocatedHours() {
                        let total = 0;
                        this.selectedEmployeesData.forEach(emp => {
                            const selectedWages = emp.wage_types.filter(w => w.selected);
                            if (selectedWages.length === 0) return;

                            if (selectedWages.length === 1 && this.selectedEmployeeIds.length === 1) {
                                total += this.calculateDuration();
                            } else {
                                selectedWages.forEach(w => {
                                    total += parseFloat(w.allocated_hours || 0);
                                });
                            }
                        });
                        return total;
                    },

                    isOverAllocated() {
                        const duration = this.calculateDuration();
                        if (duration <= 0) return false;
                        return this.getTotalAllocatedHours() > (duration + 0.01);
                    },

                    isAllocationMismatch() {
                        const duration = this.calculateDuration();
                        if (duration <= 0) return false;
                        const total = this.getTotalAllocatedHours();
                        return Math.abs(total - duration) > 0.01;
                    },

                    // Calculate payout for a specific employee based on their selected wage types and hour allocations
                    calculateGuardPayout(emp) {
                        const totalHrs = this.calculateDuration();
                        if (totalHrs <= 0) return 0;

                        const selectedWages = emp.wage_types.filter(w => w.selected);
                        if (selectedWages.length === 0) return 0;

                        // If only one is selected and only one guard, it gets all hours automatically
                        if (selectedWages.length === 1 && this.selectedEmployeeIds.length === 1) {
                            return totalHrs * parseFloat(selectedWages[0].rate || 0);
                        }

                        // If multiple, sum up the allocated hours * rates
                        let total = 0;
                        selectedWages.forEach(w => {
                            total += (parseFloat(w.allocated_hours) || 0) * parseFloat(w.rate || 0);
                        });
                        return total;
                    },

                    get financialSummary() {
                        const hrs = this.calculateDuration();
                        const revenue = hrs * (parseFloat(this.siteRate) || 0);

                        let totalPayout = 0;
                        this.selectedEmployeesData.forEach(emp => {
                            totalPayout += this.calculateGuardPayout(emp);
                            totalPayout += (parseFloat(emp.gas_rate) || 0);
                        });

                        const margin = revenue - totalPayout;
                        const marginPercent = revenue > 0 ? (margin / revenue) * 100 : 0;

                        return {
                            revenue: revenue.toFixed(2),
                            payout: totalPayout.toFixed(2),
                            margin: margin.toFixed(2),
                            marginPercent: marginPercent.toFixed(1)
                        };
                    },

                    // Pivot data from existing database records
                    initialEmployeeMeta: @js($schedule->employees->keyBy('id')->map(fn($e) => [
                        'wage_rate' => $e->pivot->wage_rate,
                        'gas_rate' => $e->pivot->gas_rate,
                        'wage_types' => json_decode($e->pivot->wage_types, true) ?: []
                    ])),

                    init() {
                        this.$nextTick(() => {
                            this.fetchSiteInfo();
                            // Pre-initialize to ensure fields render immediately (fixes validation "required" error)
                            this.selectedEmployeesData = this.selectedEmployeeIds.map(id => ({
                                id: id,
                                name: 'Loading...',
                                wage_types: [],
                                conflicts: []
                            }));
                            this.fetchEmployeesInfo();
                            this.isInitialLoad = false;

                            this.$watch('selectedEmployeeIds', () => {
                                this.updateSiteRate();
                                this.fetchEmployeesInfo();
                            });
                        });
                    },

                    toggleEmployee(id) {
                        const sId = id.toString();
                        const index = this.selectedEmployeeIds.indexOf(sId);
                        if (index > -1) {
                            this.selectedEmployeeIds.splice(index, 1);
                        } else {
                            this.selectedEmployeeIds.push(sId);
                        }
                    },

                    updateSiteRate() {
                        const count = this.selectedEmployeeIds.length;
                        const multiplier = count > 0 ? count : 1;
                        if (this.baseSiteRate) {
                            this.siteRate = (parseFloat(this.baseSiteRate) * multiplier).toFixed(2);
                        }
                    },

                    async fetchSiteInfo() {
                        if (!this.selectedSiteId) return;
                        try {
                            const response = await fetch(`/schedules/site-info/${this.selectedSiteId}`);
                            const data = await response.json();
                            this.siteInfo = data;
                            this.baseSiteRate = data.rate;

                            if (!this.isInitialLoad) {
                                this.updateSiteRate();
                            }
                        } catch (e) { }
                    },

                    async fetchEmployeesInfo() {
                        if (this.selectedEmployeeIds.length === 0) {
                            this.selectedEmployeesData = [];
                            this.allConflicts = [];
                            return;
                        }

                        try {
                            const fetchPromises = this.selectedEmployeeIds.map(async (id) => {
                                let url = `/schedules/employee-info/${id}?`;
                                let params = new URLSearchParams();
                                if (this.fromDatetime) params.append('start', this.fromDatetime);
                                if (this.toDatetime) params.append('end', this.toDatetime);
                                if (this.scheduleDate) params.append('date', this.scheduleDate);

                                url += params.toString();
                                const res = await fetch(url);
                                return await res.json();
                            });

                            const employeesData = await Promise.all(fetchPromises);

                            this.selectedEmployeesData = employeesData.map(newData => {
                                // Priority: Current Input > Database Record > Defaults
                                const existingInUI = this.selectedEmployeesData.find(e => e.id == newData.id);
                                const dbMeta = this.initialEmployeeMeta[newData.id];
                                const duration = this.calculateDuration() || 0;

                                let processedWageTypes = [];
                                (newData.wage_types || []).forEach(wt => {
                                    const isSin = wt.name.toUpperCase() === 'SIN';
                                    const hasSplit = newData.holiday_split && newData.holiday_split.holiday_hours > 0 && newData.holiday_split.regular_hours > 0;

                                    // Determine selection state, rate, and allocated hours
                                    let isSelected = false;
                                    let rate = (isSin && newData.holiday_split && newData.holiday_split.holiday_hours > 0)
                                        ? wt.holiday_rate : wt.base_rate;
                                    let allocatedHours = duration.toFixed(1);

                                    // Check database records robustly (could be array or object)
                                    const dbWtList = dbMeta && dbMeta.wage_types ? (Array.isArray(dbMeta.wage_types) ? dbMeta.wage_types : Object.values(dbMeta.wage_types)) : [];
                                    const dbWt = dbWtList.find(dw => dw && dw.id == wt.id);

                                    // Check if we have current UI state to preserve
                                    let prev = null;
                                    if (existingInUI && Array.isArray(existingInUI.wage_types) && existingInUI.wage_types.length > 0) {
                                        prev = existingInUI.wage_types.find(w => w.id == wt.id);
                                    }

                                    if (prev) {
                                        // Manual UI choices take top priority
                                        isSelected = prev.selected;
                                        allocatedHours = prev.allocated_hours;
                                        rate = prev.rate;
                                    } else if (dbWt) {
                                        // Database records are the secondary source of truth
                                        isSelected = true;
                                        allocatedHours = dbWt.allocated_hours;
                                        rate = dbWt.rate;
                                    } else {
                                        // Default for new assignments only (not in DB and not yet in UI)
                                        isSelected = isSin;
                                    }

                                    if (isSin && hasSplit) {
                                        // Push Regular Part
                                        processedWageTypes.push({
                                            ...wt,
                                            virtual_id: wt.id + '_reg',
                                            name: 'SIN (Regular Part)',
                                            rate: rate,
                                            allocated_hours: newData.holiday_split.regular_hours.toFixed(1),
                                            selected: isSelected
                                        });
                                        // Push Holiday Part
                                        processedWageTypes.push({
                                            ...wt,
                                            virtual_id: wt.id + '_hol',
                                            name: 'SIN (Stat Holiday: ' + newData.holiday_split.holiday_name + ')',
                                            rate: rate,
                                            allocated_hours: newData.holiday_split.holiday_hours.toFixed(1),
                                            selected: isSelected
                                        });
                                    } else {
                                        processedWageTypes.push({
                                            ...wt,
                                            virtual_id: wt.id,
                                            rate: rate,
                                            allocated_hours: allocatedHours,
                                            selected: isSelected
                                        });
                                    }
                                });

                                return {
                                    ...newData,
                                    wage_rate: (existingInUI && existingInUI.wage_rate !== undefined) ? existingInUI.wage_rate : (dbMeta ? dbMeta.wage_rate : newData.wage_rate || ''),
                                    gas_rate: (existingInUI && existingInUI.gas_rate !== undefined) ? existingInUI.gas_rate : (dbMeta ? dbMeta.gas_rate : ''),
                                    wage_types: processedWageTypes
                                };
                            });

                            this.allConflicts = [];
                            employeesData.forEach(emp => {
                                (emp.conflicts || []).forEach(c => {
                                    if (c.duty_number !== '{{ $schedule->duty_number }}') {
                                        this.allConflicts.push({ ...c, guard_name: emp.name });
                                    }
                                });
                            });

                        } catch (e) { console.error(e); }
                    }
                }
            }
        </script>
        <style>
            [x-cloak] {
                display: none !important;
            }
        </style>
    @endpush
@endsection