@extends('layouts.user')

@section('title', 'Create New Quotation')

@section('content')
    <div class="p-6" x-data="quotationForm()">
        <div class="flex justify-between items-center mb-6">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">New Quotation</h1>
                <p class="text-gray-500">Create a price estimate for a client.</p>
            </div>
            <a href="{{ route('quotations.index') }}"
                class="px-6 py-2 bg-gray-100 text-gray-600 font-bold rounded-lg hover:bg-gray-200 transition-all">
                Cancel
            </a>
        </div>

        <form action="{{ route('quotations.store') }}" method="POST" class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            @csrf

            <!-- Left Column: Quote Details -->
            <div class="lg:col-span-2 space-y-6">

                <!-- Client Info Card -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-8">
                    <h3
                        class="text-xs font-black text-gray-400 uppercase tracking-widest mb-6 border-b border-gray-100 pb-2">
                        Client Details</h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase mb-2">Client Name <span
                                    class="text-red-500">*</span></label>
                            <input type="text" name="client_name" required
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold text-gray-800">
                        </div>
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase mb-2">Email Address</label>
                            <input type="email" name="client_email"
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold text-gray-800">
                        </div>
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase mb-2">Phone Number</label>
                            <input type="tel" name="client_phone"
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-bold text-gray-800">
                        </div>
                        <div class="md:col-span-2">
                            <label class="block text-xs font-bold text-gray-500 uppercase mb-2">Billing Address</label>
                            <textarea name="client_address" rows="2"
                                class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-medium text-gray-700"></textarea>
                        </div>
                    </div>
                </div>

                <!-- Line Items Card -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-8 overflow-hidden">
                    <div class="flex items-center justify-between mb-6 border-b border-gray-100 pb-2">
                        <h3 class="text-xs font-black text-gray-400 uppercase tracking-widest">Quote Items</h3>
                        <button type="button" @click="addItem()"
                            class="text-xs font-bold text-blue-600 hover:text-blue-700 hover:underline flex items-center gap-1">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4">
                                </path>
                            </svg>
                            Add Item
                        </button>
                    </div>

                    <div class="space-y-4">
                        <template x-for="(item, index) in items" :key="index">
                            <div
                                class="p-4 bg-gray-50 rounded-xl border border-gray-100 relative group animate-in slide-in-from-bottom-2 duration-300">
                                <!-- Remove Button -->
                                <button type="button" @click="removeItem(index)" x-show="items.length > 1"
                                    class="absolute top-2 right-2 p-1 text-gray-300 hover:text-red-500 transition-colors">
                                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M6 18L18 6M6 6l12 12"></path>
                                    </svg>
                                </button>

                                <div class="grid grid-cols-1 md:grid-cols-12 gap-4">
                                    <!-- Type Selector -->
                                    <div class="md:col-span-2">
                                        <label class="block text-[10px] font-bold text-gray-400 uppercase mb-1">Type</label>
                                        <select :name="'items['+index+'][type]'" x-model="item.type"
                                            class="w-full px-2 py-2 bg-white border border-gray-200 rounded-lg text-xs font-bold focus:border-blue-500 outline-none">
                                            <option value="service" selected>Service</option>
                                            <option value="product">Product</option>
                                        </select>
                                    </div>

                                    <!-- Description -->
                                    <div class="md:col-span-5">
                                        <label
                                            class="block text-[10px] font-bold text-gray-400 uppercase mb-1">Description</label>
                                        <input type="text" :name="'items['+index+'][description]'"
                                            x-model="item.description" placeholder="e.g. Unarmed Night Guard" required
                                            class="w-full px-3 py-2 bg-white border border-gray-200 rounded-lg text-sm font-bold text-gray-800 focus:border-blue-500 outline-none">
                                    </div>

                                    <!-- Quantity -->
                                    <div class="md:col-span-1">
                                        <label class="block text-[10px] font-bold text-gray-400 uppercase mb-1">Qty</label>
                                        <input type="number" :name="'items['+index+'][quantity]'" x-model="item.quantity"
                                            min="1" required
                                            class="w-full px-2 py-2 bg-white border border-gray-200 rounded-lg text-sm font-bold text-center focus:border-blue-500 outline-none">
                                    </div>

                                    <!-- Hours (Only for Service) -->
                                    <div class="md:col-span-2">
                                        <label class="block text-[10px] font-bold text-gray-400 uppercase mb-1"
                                            x-text="item.type === 'service' ? 'Hours' : '---'"></label>
                                        <input type="number" :name="'items['+index+'][hours]'" x-model="item.hours" min="0"
                                            step="0.5" :disabled="item.type === 'product'"
                                            :class="item.type === 'product' ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-white text-gray-800'"
                                            class="w-full px-2 py-2 border border-gray-200 rounded-lg text-sm font-bold text-center focus:border-blue-500 outline-none">
                                    </div>

                                    <!-- Unit Price -->
                                    <div class="md:col-span-2">
                                        <label class="block text-[10px] font-bold text-gray-400 uppercase mb-1">Rate
                                            ($)</label>
                                        <input type="number" :name="'items['+index+'][unit_price]'"
                                            x-model="item.unit_price" min="0" step="0.01" required
                                            class="w-full px-2 py-2 bg-white border border-gray-200 rounded-lg text-sm font-bold text-right focus:border-blue-500 outline-none">
                                    </div>
                                </div>

                                <!-- Row Total -->
                                <div class="mt-2 text-right">
                                    <span class="text-[10px] font-bold text-slate-400 uppercase tracking-widest mr-2">Line
                                        Total:</span>
                                    <span class="text-sm font-black text-slate-800"
                                        x-text="'$' + calculateLineTotal(item).toFixed(2)"></span>
                                </div>
                            </div>
                        </template>
                    </div>
                </div>

                <!-- Notes Section -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-8">
                    <label class="block text-xs font-bold text-gray-500 uppercase mb-2">Terms & Conditions / Notes</label>
                    <textarea name="notes" rows="4"
                        class="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:ring-4 focus:ring-blue-500/10 focus:border-blue-500 outline-none transition-all font-medium text-gray-700"
                        placeholder="Enter any specific terms, conditions, or payment instructions here..."></textarea>
                </div>

                <div class="flex gap-4">
                    <button type="submit"
                        class="px-8 py-4 bg-blue-600 text-white font-bold rounded-xl hover:bg-blue-700 shadow-xl shadow-blue-500/20 transition-all active:scale-95 flex-1">
                        Create Quotation
                    </button>
                </div>
            </div>

            <!-- Right Column: Summary & Meta -->
            <div class="space-y-6">
                <!-- Quote Meta -->
                <div class="bg-slate-800 rounded-2xl shadow-lg shadow-slate-900/10 border border-slate-700 p-6 text-white">
                    <h3 class="text-xs font-bold uppercase tracking-widest text-slate-400 mb-6">Quote Information</h3>

                    <div class="space-y-4">
                        <div>
                            <label class="block text-[10px] font-bold text-slate-400 uppercase mb-1">Quote Number</label>
                            <input type="text" name="quotation_number" value="{{ $nextNumber }}" readonly
                                class="w-full px-3 py-2 bg-slate-700/50 border border-slate-600 rounded-lg text-gray-200 font-mono font-bold outline-none cursor-not-allowed">
                        </div>
                        <div>
                            <label class="block text-[10px] font-bold text-slate-400 uppercase mb-1">Issue Date</label>
                            <input type="date" name="issue_date" value="{{ date('Y-m-d') }}" required
                                class="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-lg text-white font-bold focus:border-blue-500 outline-none">
                        </div>
                        <div>
                            <label class="block text-[10px] font-bold text-slate-400 uppercase mb-1">Valid Until</label>
                            <input type="date" name="valid_until" value="{{ date('Y-m-d', strtotime('+30 days')) }}"
                                class="w-full px-3 py-2 bg-slate-700 border border-slate-600 rounded-lg text-white font-bold focus:border-blue-500 outline-none">
                        </div>
                    </div>
                </div>

                <!-- Financial Summary -->
                <div class="bg-white rounded-2xl shadow-sm border border-gray-100 p-6 sticky top-6">
                    <h3 class="text-xs font-black text-gray-400 uppercase tracking-widest mb-6">Financial Summary</h3>

                    <div class="space-y-3 mb-6">
                        <div class="flex justify-between items-center text-sm">
                            <span class="font-bold text-gray-500">Subtotal</span>
                            <span class="font-black text-gray-900" x-text="'$' + calculateSubtotal().toFixed(2)"></span>
                        </div>

                        <div class="flex justify-between items-center text-sm">
                            <span class="font-bold text-gray-500 flex items-center gap-1">
                                Tax Rate (%)
                                <input type="number" name="tax_rate" x-model="taxRate" min="0" max="100" step="0.1"
                                    class="w-16 px-1 py-0.5 bg-gray-50 border border-gray-200 rounded text-center text-xs ml-1">
                            </span>
                            <span class="font-black text-gray-900" x-text="'$' + calculateTax().toFixed(2)"></span>
                        </div>

                        <div class="h-px bg-gray-100 my-2"></div>

                        <div class="flex justify-between items-center text-lg">
                            <span class="font-black text-slate-800">Total</span>
                            <span class="font-black text-blue-600" x-text="'$' + calculateTotal().toFixed(2)"></span>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>

    @push('scripts')
        <script>
            function quotationForm() {
                return {
                    taxRate: 13.0, // Default tax rate
                    items: [
                        { type: 'service', description: '', quantity: 1, hours: 8, unit_price: 25.00 }
                    ],

                    addItem() {
                        this.items.push({ type: 'service', description: '', quantity: 1, hours: 0, unit_price: 0 });
                    },

                    removeItem(index) {
                        if (this.items.length > 1) {
                            this.items.splice(index, 1);
                        }
                    },

                    calculateLineTotal(item) {
                        const qty = parseFloat(item.quantity) || 0;
                        const rate = parseFloat(item.unit_price) || 0;

                        if (item.type === 'product') {
                            return qty * rate;
                        } else {
                            const hours = parseFloat(item.hours) || 0;
                            return qty * hours * rate;
                        }
                    },

                    calculateSubtotal() {
                        return this.items.reduce((sum, item) => sum + this.calculateLineTotal(item), 0);
                    },

                    calculateTax() {
                        return this.calculateSubtotal() * (parseFloat(this.taxRate) || 0) / 100;
                    },

                    calculateTotal() {
                        return this.calculateSubtotal() + this.calculateTax();
                    }
                }
            }
        </script>
    @endpush
@endsection