<?php

namespace App\Http\Controllers;

use App\Models\ScheduleTemplate;
use App\Models\Site;
use App\Models\TemplateShift;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ScheduleTemplateController extends Controller
{
    /**
     * Get the schedule template for a specific site (JSON API).
     */
    /**
     * Get all schedule templates for a specific site (JSON API).
     */
    public function index(Site $site)
    {
        $templates = ScheduleTemplate::where('site_id', $site->id)
            ->with(['shifts.jobRole', 'shifts.employee'])
            ->get();

        return response()->json([
            'templates' => $templates
        ]);
    }

    /**
     * Create or Update a Schedule Template.
     */
    public function store(Request $request)
    {
        $request->validate([
            'site_id' => 'required|exists:sites,id',
            'id' => 'nullable|exists:schedule_templates,id', // Optional ID for update
            'name' => 'required|string',
            'cycle_duration_days' => 'required|integer|in:7,14,28',
            'start_date' => 'nullable|date',
            'settings' => 'nullable|array'
        ]);

        if ($request->id) {
            $template = ScheduleTemplate::find($request->id);
            $template->update([
                'name' => $request->name,
                'cycle_duration_days' => $request->cycle_duration_days,
                'start_date' => $request->start_date,
                'settings' => $request->settings
            ]);
        } else {
            // Create New
            $template = ScheduleTemplate::create([
                'site_id' => $request->site_id,
                'name' => $request->name,
                'cycle_duration_days' => $request->cycle_duration_days,
                'start_date' => $request->start_date,
                'settings' => $request->settings
            ]);
        }

        return response()->json(['message' => 'Template saved.', 'template' => $template]);
    }

    /**
     * Delete the entire Schedule Template.
     */
    public function destroy(ScheduleTemplate $template)
    {
        $template->delete();
        return response()->json(['message' => 'Template deleted successfully.']);
    }

    /**
     * Add or Update a Shift in the Template.
     */
    public function storeShift(Request $request, ScheduleTemplate $template)
    {
        $request->validate([
            'day_offset' => 'required|integer|min:1',
            'start_time' => 'required', // HH:mm
            'end_time' => 'required', // HH:mm
            'job_role_id' => 'required|exists:job_roles,id',
            'employee_id' => 'nullable|exists:employees,id',
            'break_duration_minutes' => 'nullable|integer'
        ]);

        // Basic overlap check could go here (Phase 4 validation)

        $shift = $template->shifts()->create([
            'day_offset' => $request->day_offset,
            'start_time' => $request->start_time,
            'end_time' => $request->end_time,
            'job_role_id' => $request->job_role_id,
            'employee_id' => $request->employee_id,
            'break_duration_minutes' => $request->break_duration_minutes ?? 0,
        ]);

        return response()->json(['message' => 'Shift added to template.', 'shift' => $shift->load('jobRole', 'employee')]);
    }

    /**
     * Delete a Template Shift.
     */
    public function destroyShift(TemplateShift $shift)
    {
        $shift->delete();
        return response()->json(['message' => 'Shift removed from template.']);
    }

    /**
     * Roll the template into active schedules for a date range.
     */
    public function roll(Request $request, ScheduleTemplate $template)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        $site = $template->site;
        $periodStart = \Carbon\Carbon::parse($request->start_date);
        $periodEnd = \Carbon\Carbon::parse($request->end_date);
        $cycleStart = $template->start_date ? \Carbon\Carbon::parse($template->start_date) : $periodStart;

        $cycleStart = $template->start_date ? \Carbon\Carbon::parse($template->start_date) : $periodStart;

        // 1. Validation & Conflict Check Phase
        $conflicts = [];
        $shifts = $template->shifts;

        for ($date = $periodStart->copy(); $date->lte($periodEnd); $date->addDay()) {
            $diffInDays = $cycleStart->diffInDays($date);
            $dayOffset = ($diffInDays % $template->cycle_duration_days) + 1;
            $dayShifts = $shifts->where('day_offset', $dayOffset);

            foreach ($dayShifts as $tShift) {
                if (!$tShift->employee_id)
                    continue;

                $startDateTime = $date->copy()->setTimeFromTimeString($tShift->start_time);
                $endDateTime = $date->copy()->setTimeFromTimeString($tShift->end_time);

                if ($endDateTime->lt($startDateTime)) {
                    $endDateTime->addDay();
                }

                // Check for overlapping schedules for this employee
                $hasConflict = \App\Models\Schedule::whereHas('employees', function ($q) use ($tShift) {
                    $q->where('employees.id', $tShift->employee_id);
                })
                    ->where(function ($q) use ($startDateTime, $endDateTime) {
                        $q->where('from_datetime', '<', $endDateTime)
                            ->where('to_datetime', '>', $startDateTime);
                    })
                    ->where('status', '!=', 'cancelled') // Assuming 'cancelled' schedules don't block
                    ->exists();

                if ($hasConflict) {
                    $employeeName = $tShift->employee ? $tShift->employee->first_name . ' ' . $tShift->employee->last_name : 'Employee #' . $tShift->employee_id;
                    $conflicts[] = "Conflict for $employeeName on " . $date->format('Y-m-d') . " (" . $tShift->start_time . " - " . $tShift->end_time . ")";
                }
            }
        }

        if (!empty($conflicts)) {
            return response()->json(['message' => 'Scheduling conflicts detected.', 'errors' => $conflicts], 422);
        }

        // 2. Execution Phase
        DB::transaction(function () use ($template, $site, $periodStart, $periodEnd, $cycleStart, $shifts) {

            for ($date = $periodStart->copy(); $date->lte($periodEnd); $date->addDay()) {
                // Determine day offset (1-based)
                // If cycle is 7 days, and we are on day X.
                $diffInDays = $cycleStart->diffInDays($date);
                $dayOffset = ($diffInDays % $template->cycle_duration_days) + 1;

                $dayShifts = $shifts->where('day_offset', $dayOffset);

                foreach ($dayShifts as $tShift) {
                    $startDateTime = $date->copy()->setTimeFromTimeString($tShift->start_time);
                    $endDateTime = $date->copy()->setTimeFromTimeString($tShift->end_time);

                    if ($endDateTime->lt($startDateTime)) {
                        $endDateTime->addDay();
                    }

                    // Create Schedule Record
                    $schedule = \App\Models\Schedule::create([
                        'company_id' => $site->company_id,
                        'duty_number' => 'SCH-' . $startDateTime->format('YmdHis') . '-' . mt_rand(1000, 9999),
                        'schedule_date' => $startDateTime->format('Y-m-d'),
                        'site_id' => $site->id,
                        'from_datetime' => $startDateTime,
                        'to_datetime' => $endDateTime,
                        'site_rate' => 0, // Defaults
                        'wage_rate' => 0, // Defaults
                        'status' => 'draft',
                    ]);

                    // Assign Employee if present in template
                    if ($tShift->employee_id) {
                        $schedule->employees()->attach($tShift->employee_id, [
                            // Add any pivot fields if necessary
                        ]);
                    }
                }
            }
        });

        return response()->json(['message' => 'Template rolled out successfully.']);
    }
}
