<?php

namespace App\Http\Controllers;

use App\Models\JobNote;
use App\Models\Employee;
use App\Models\Schedule;
use App\Models\PatrollerSchedule;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class JobNoteController extends Controller
{
    public function index(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = JobNote::with(['employee'])
            ->whereHas('employee', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            });

        // Search by Employee Name
        if ($request->filled('employee_name')) {
            $query->whereHas('employee', function ($q) use ($request) {
                $q->where(function ($q2) use ($request) {
                    $q2->where('first_name', 'like', '%' . $request->employee_name . '%')
                        ->orWhere('last_name', 'like', '%' . $request->employee_name . '%');
                });
            });
        }

        // Search by Duty Number
        if ($request->filled('duty_number')) {
            $dutyNumber = $request->duty_number;
            $query->where(function ($q) use ($dutyNumber) {
                // Guard jobs
                $q->where(function ($q2) use ($dutyNumber) {
                    $q2->where('job_type', 'guard')
                        ->whereIn('job_id', function ($sub) use ($dutyNumber) {
                            $sub->select('id')->from('schedules')->where('duty_number', 'like', '%' . $dutyNumber . '%');
                        });
                })->orWhere(function ($q2) use ($dutyNumber) {
                    // Patroller jobs
                    $q2->where('job_type', 'patroller')
                        ->whereIn('job_id', function ($sub) use ($dutyNumber) {
                            $sub->select('id')->from('patroller_schedules')->where('duty_number', 'like', '%' . $dutyNumber . '%');
                        });
                });
            });
        }

        // Filter by Site/Route
        if ($request->filled('site_name')) {
            $siteName = $request->site_name;
            $query->where(function ($q) use ($siteName) {
                // Guard jobs (Sites)
                $q->where(function ($q2) use ($siteName) {
                    $q2->where('job_type', 'guard')
                        ->whereIn('job_id', function ($sub) use ($siteName) {
                            $sub->select('schedules.id')
                                ->from('schedules')
                                ->join('sites', 'schedules.site_id', '=', 'sites.id')
                                ->where('sites.name', 'like', '%' . $siteName . '%');
                        });
                })->orWhere(function ($q2) use ($siteName) {
                    // Patroller jobs (Routes)
                    $q2->where('job_type', 'patroller')
                        ->whereIn('job_id', function ($sub) use ($siteName) {
                            $sub->select('patroller_schedules.id')
                                ->from('patroller_schedules')
                                ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
                                ->where('routes.name', 'like', '%' . $siteName . '%');
                        });
                });
            });
        }

        $notes = $query->orderBy('job_type')
            ->orderByDesc('job_id')
            ->latest()
            ->paginate(50)
            ->withQueryString();

        // Attach job details to prevent N+1 in view
        $scheduleIds = $notes->where('job_type', 'guard')->pluck('job_id')->unique();
        $patrollerIds = $notes->where('job_type', 'patroller')->pluck('job_id')->unique();

        $schedules = Schedule::with('site')->whereIn('id', $scheduleIds)->get()->keyBy('id');
        $patrollers = PatrollerSchedule::with('route')->whereIn('id', $patrollerIds)->get()->keyBy('id');

        foreach ($notes as $note) {
            if ($note->job_type === 'guard') {
                $note->setRelation('job_details', $schedules->get($note->job_id));
            } else {
                $note->setRelation('job_details', $patrollers->get($note->job_id));
            }
        }

        return view('job-notes.index', compact('notes'));
    }

    public function exportPdf(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = JobNote::with(['employee'])
            ->whereHas('employee', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            });

        // Apply same filters as index
        if ($request->filled('employee_name')) {
            $query->whereHas('employee', function ($q) use ($request) {
                $q->where(function ($q2) use ($request) {
                    $q2->where('first_name', 'like', '%' . $request->employee_name . '%')
                        ->orWhere('last_name', 'like', '%' . $request->employee_name . '%');
                });
            });
        }

        if ($request->filled('duty_number')) {
            $dutyNumber = $request->duty_number;
            $query->where(function ($q) use ($dutyNumber) {
                $q->where(function ($q2) use ($dutyNumber) {
                    $q2->where('job_type', 'guard')
                        ->whereIn('job_id', function ($sub) use ($dutyNumber) {
                            $sub->select('id')->from('schedules')->where('duty_number', 'like', '%' . $dutyNumber . '%');
                        });
                })->orWhere(function ($q2) use ($dutyNumber) {
                    $q2->where('job_type', 'patroller')
                        ->whereIn('job_id', function ($sub) use ($dutyNumber) {
                            $sub->select('id')->from('patroller_schedules')->where('duty_number', 'like', '%' . $dutyNumber . '%');
                        });
                });
            });
        }

        if ($request->filled('site_name')) {
            $siteName = $request->site_name;
            $query->where(function ($q) use ($siteName) {
                $q->where(function ($q2) use ($siteName) {
                    $q2->where('job_type', 'guard')
                        ->whereIn('job_id', function ($sub) use ($siteName) {
                            $sub->select('schedules.id')
                                ->from('schedules')
                                ->join('sites', 'schedules.site_id', '=', 'sites.id')
                                ->where('sites.name', 'like', '%' . $siteName . '%');
                        });
                })->orWhere(function ($q2) use ($siteName) {
                    $q2->where('job_type', 'patroller')
                        ->whereIn('job_id', function ($sub) use ($siteName) {
                            $sub->select('patroller_schedules.id')
                                ->from('patroller_schedules')
                                ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
                                ->where('routes.name', 'like', '%' . $siteName . '%');
                        });
                });
            });
        }

        $notes = $query->orderBy('job_type')
            ->orderByDesc('job_id')
            ->latest()
            ->get();

        // Attach job details
        $scheduleIds = $notes->where('job_type', 'guard')->pluck('job_id')->unique();
        $patrollerIds = $notes->where('job_type', 'patroller')->pluck('job_id')->unique();

        $schedules = Schedule::with('site')->whereIn('id', $scheduleIds)->get()->keyBy('id');
        $patrollers = PatrollerSchedule::with('route')->whereIn('id', $patrollerIds)->get()->keyBy('id');

        $groupedNotes = [];
        foreach ($notes as $note) {
            $key = $note->job_type . '_' . $note->job_id;

            if (!isset($groupedNotes[$key])) {
                $siteRoute = 'N/A';
                $dutyNumber = 'N/A';

                if ($note->job_type === 'guard') {
                    $item = $schedules->get($note->job_id);
                    $siteRoute = $item && $item->site ? $item->site->name : 'N/A';
                    $dutyNumber = $item ? $item->duty_number : 'N/A';
                } else {
                    $item = $patrollers->get($note->job_id);
                    $siteRoute = $item && $item->route ? $item->route->name : 'N/A';
                    $dutyNumber = $item ? $item->duty_number : 'N/A';
                }

                $groupedNotes[$key] = [
                    'job_type' => $note->job_type,
                    'duty_number' => $dutyNumber,
                    'site_route' => $siteRoute,
                    'notes' => []
                ];
            }
            $groupedNotes[$key]['notes'][] = $note;
        }

        $pdf = Pdf::loadView('pdf.job-notes', [
            'groupedNotes' => $groupedNotes
        ])->setPaper('a4', 'portrait');

        return $pdf->download('job-notes-' . now()->format('Y-m-d') . '.pdf');
    }

    public function exportExcel(Request $request)
    {
        $companyId = session('selected_company_id');
        $query = JobNote::with(['employee'])
            ->whereHas('employee', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            });

        // Apply filters
        if ($request->filled('employee_name')) {
            $query->whereHas('employee', function ($q) use ($request) {
                $q->where(function ($q2) use ($request) {
                    $q2->where('first_name', 'like', '%' . $request->employee_name . '%')
                        ->orWhere('last_name', 'like', '%' . $request->employee_name . '%');
                });
            });
        }

        if ($request->filled('duty_number')) {
            $dutyNumber = $request->duty_number;
            $query->where(function ($q) use ($dutyNumber) {
                $q->where(function ($q2) use ($dutyNumber) {
                    $q2->where('job_type', 'guard')
                        ->whereIn('job_id', function ($sub) use ($dutyNumber) {
                            $sub->select('id')->from('schedules')->where('duty_number', 'like', '%' . $dutyNumber . '%');
                        });
                })->orWhere(function ($q2) use ($dutyNumber) {
                    $q2->where('job_type', 'patroller')
                        ->whereIn('job_id', function ($sub) use ($dutyNumber) {
                            $sub->select('id')->from('patroller_schedules')->where('duty_number', 'like', '%' . $dutyNumber . '%');
                        });
                });
            });
        }

        if ($request->filled('site_name')) {
            $siteName = $request->site_name;
            $query->where(function ($q) use ($siteName) {
                $q->where(function ($q2) use ($siteName) {
                    $q2->where('job_type', 'guard')
                        ->whereIn('job_id', function ($sub) use ($siteName) {
                            $sub->select('schedules.id')
                                ->from('schedules')
                                ->join('sites', 'schedules.site_id', '=', 'sites.id')
                                ->where('sites.name', 'like', '%' . $siteName . '%');
                        });
                })->orWhere(function ($q2) use ($siteName) {
                    $q2->where('job_type', 'patroller')
                        ->whereIn('job_id', function ($sub) use ($siteName) {
                            $sub->select('patroller_schedules.id')
                                ->from('patroller_schedules')
                                ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
                                ->where('routes.name', 'like', '%' . $siteName . '%');
                        });
                });
            });
        }

        $notes = $query->orderBy('job_type')
            ->orderByDesc('job_id')
            ->latest()
            ->get();

        $filename = 'job-notes-' . now()->format('Y-m-d') . '.csv';
        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];

        $columns = ['Date', 'Employee', 'Job Type', 'Duty #', 'Site/Route', 'Note', 'Image URL'];

        $callback = function () use ($notes, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            // Pre-fetch for efficiency
            $scheduleIds = $notes->where('job_type', 'guard')->pluck('job_id')->unique();
            $patrollerIds = $notes->where('job_type', 'patroller')->pluck('job_id')->unique();
            $schedules = Schedule::with('site')->whereIn('id', $scheduleIds)->get()->keyBy('id');
            $patrollers = PatrollerSchedule::with('route')->whereIn('id', $patrollerIds)->get()->keyBy('id');

            foreach ($notes as $note) {
                $siteRoute = 'N/A';
                $dutyNumber = 'N/A';

                if ($note->job_type === 'guard') {
                    $item = $schedules->get($note->job_id);
                    $siteRoute = $item && $item->site ? $item->site->name : 'N/A';
                    $dutyNumber = $item ? $item->duty_number : 'N/A';
                } else {
                    $item = $patrollers->get($note->job_id);
                    $siteRoute = $item && $item->route ? $item->route->name : 'N/A';
                    $dutyNumber = $item ? $item->duty_number : 'N/A';
                }

                fputcsv($file, [
                    $note->created_at->format('Y-m-d H:i'),
                    $note->employee->full_name ?? ($note->employee->first_name . ' ' . $note->employee->last_name),
                    ucfirst($note->job_type),
                    $dutyNumber,
                    $siteRoute,
                    $note->note,
                    $note->image ? asset('storage/' . $note->image) : ''
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
