<?php

namespace App\Http\Controllers;

use App\Models\EmailTemplate;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Mail\CustomEmail; // We will need to create this Mailable

class EmailTemplateController extends Controller
{
    public function index()
    {
        $companyId = session('selected_company_id');
        if (!$companyId) {
            return redirect()->route('select.company');
        }

        $templates = EmailTemplate::where('company_id', $companyId)->get();

        return view('email_templates.index', compact('templates'));
    }

    public function create()
    {
        $companyId = session('selected_company_id');
        // Fetch default signature for the active company
        $defaultSignature = \App\Models\EmailSignature::where('company_id', $companyId)
            ->where('is_default', true)
            ->first();

        // If no default is set, maybe grab the first one
        if (!$defaultSignature) {
            $defaultSignature = \App\Models\EmailSignature::where('company_id', $companyId)->first();
        }

        return view('email_templates.create', compact('defaultSignature'));
    }

    public function store(Request $request)
    {
        // Debug: Log what we're receiving
        \Log::info('Email Template Store Request', [
            'all_data' => $request->all(),
            'has_content' => $request->has('content'),
            'content_length' => strlen($request->input('content', '')),
            'company_id' => session('selected_company_id')
        ]);

        $request->validate([
            'name' => 'required|string|max:255',
            'subject' => 'required|string|max:255',
            'content' => 'required|string',
        ]);

        $companyId = session('selected_company_id');

        if (!$companyId) {
            return redirect()->route('select.company')->with('error', 'Please select a company first.');
        }

        $template = EmailTemplate::create([
            'company_id' => $companyId,
            'name' => $request->name,
            'subject' => $request->subject,
            'content' => $request->input('content'),
            'is_active' => true,
        ]);

        \Log::info('Email Template Created', ['id' => $template->id]);

        return redirect()->route('email-templates.index')->with('success', 'Email template created successfully.');
    }

    public function edit(EmailTemplate $emailTemplate)
    {
        if ($emailTemplate->company_id != session('selected_company_id')) {
            abort(403);
        }
        return view('email_templates.edit', compact('emailTemplate'));
    }

    public function update(Request $request, EmailTemplate $emailTemplate)
    {
        if ($emailTemplate->company_id != session('selected_company_id')) {
            abort(403);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'subject' => 'required|string|max:255',
            'content' => 'required|string',
        ]);

        $emailTemplate->update($request->only('name', 'subject', 'content'));

        return redirect()->route('email-templates.index')->with('success', 'Email template updated successfully.');
    }

    public function destroy(EmailTemplate $emailTemplate)
    {
        if ($emailTemplate->company_id != session('selected_company_id')) {
            abort(403);
        }
        $emailTemplate->delete();
        return back()->with('success', 'Email template deleted successfully.');
    }

    public function preview(EmailTemplate $emailTemplate)
    {
        if ($emailTemplate->company_id != session('selected_company_id')) {
            abort(403);
        }

        // Dummy data for preview
        $previewContent = $emailTemplate->content;
        $previewContent = str_replace('@{{name}}', 'John Doe', $previewContent);
        $previewContent = str_replace('{{name}}', 'John Doe', $previewContent);

        $previewContent = str_replace('@{{first_name}}', 'John', $previewContent);
        $previewContent = str_replace('{{first_name}}', 'John', $previewContent);

        $previewContent = str_replace('@{{last_name}}', 'Doe', $previewContent);
        $previewContent = str_replace('{{last_name}}', 'Doe', $previewContent);

        return view('emails.generic', [
            'subjectLine' => $emailTemplate->subject,
            'content' => $previewContent
        ]);
    }
}
