<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use App\Models\PatrollerSchedule;
use App\Models\PatrollerJobSite;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Traits\PatrollerJobTransformer;

class PatrollerAPIController extends Controller
{
    use PatrollerJobTransformer;

    /**
     * Start a patroller job
     */
    public function start(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = PatrollerSchedule::with(['employees', 'route.sites', 'jobSites'])->findOrFail($id);

        // Validate employee is assigned to this schedule
        if (!$schedule->employees->contains($employee->id)) {
            return response()->json(['message' => 'You are not assigned to this schedule'], 403);
        }

        $pivot = $schedule->employees->where('id', $employee->id)->first()->pivot ?? null;

        // Check if individual already started
        if ($pivot && !empty($pivot->actual_start_at)) {
            return response()->json(['message' => 'You have already started this job'], 400);
        }

        // Check if job is in a startable overall status
        if (!in_array($schedule->job_status, ['pending', 'in_progress'])) {
            return response()->json(['message' => 'Job cannot be started in its current status: ' . $schedule->job_status], 400);
        }

        // Validate location is provided
        $request->validate([
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);

        $now = now();
        $actualStartAt = $now->gt($schedule->from_time) ? $now : $schedule->from_time;

        // Start the global job if not already started
        if ($schedule->job_status === 'pending') {
            $schedule->update([
                'job_started_at' => $actualStartAt,
                'job_status' => 'in_progress',
            ]);
        }

        // Always update the individual deployment start for this employee
        $schedule->employees()->updateExistingPivot($employee->id, [
            'actual_start_at' => $actualStartAt,
        ]);

        // Create individual job site records for THIS employee if they don't exist
        $existingSitesCount = PatrollerJobSite::where('patroller_schedule_id', $schedule->id)
            ->where('employee_id', $employee->id)
            ->count();

        if ($existingSitesCount === 0) {
            foreach ($schedule->route->sites as $site) {
                PatrollerJobSite::create([
                    'patroller_schedule_id' => $schedule->id,
                    'site_id' => $site->id,
                    'employee_id' => $employee->id,
                    'status' => 'pending',
                ]);
            }
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Patrol job started successfully',
            'data' => [
                'job_id' => $schedule->id,
                'started_at' => $schedule->job_started_at->format('Y-m-d H:i:s'),
                'status' => 'active'
            ]
        ]);
    }

    /**
     * List all patroller jobs with status filtering
     */
    public function index(Request $request)
    {
        $employee = $request->user();

        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $type = $request->query('type', 'all'); // all, active, today, upcoming, completed, missed, pending
        $now = Carbon::now();

        $query = $employee->patrollerSchedules()->with(['company', 'route.sites', 'employees', 'jobSites.site', 'issueTickets']);

        switch ($type) {
            case 'active':
                $query->active();
                break;
            case 'today':
                $query->todayUpcoming();
                break;
            case 'upcoming':
                $query->upcomingFuture();
                break;
            case 'completed':
                $query->completed();
                break;
            case 'missed':
                $query->missed();
                break;
            case 'pending':
                $query->pendingApproval();
                break;
            case 'cancelled':
                $query->cancelled();
                break;
        }

        $jobs = $query->orderBy('patroller_schedules.from_time')->paginate($request->query('per_page', 50));

        // Transform results
        $jobs->getCollection()->transform(function ($job) use ($employee, $now) {
            return $this->transformPatrollerJob($job, $employee, $now);
        });

        // Counts using standardized scopes
        $counts = [
            'all' => $employee->patrollerSchedules()->count(),
            'active' => $employee->patrollerSchedules()->active()->count(),
            'today' => $employee->patrollerSchedules()->todayUpcoming()->count(),
            'upcoming' => $employee->patrollerSchedules()->upcomingFuture()->count(),
            'completed' => $employee->patrollerSchedules()->completed()->count(),
            'missed' => $employee->patrollerSchedules()->missed()->count(),
            'pending' => $employee->patrollerSchedules()->pendingApproval()->count(),
            'cancelled' => $employee->patrollerSchedules()->cancelled()->count(),
        ];

        return response()->json([
            'status' => 'success',
            'data' => [
                'jobs' => $jobs,
                'counts' => $counts,
                'current_type' => $type,
            ],
        ]);
    }

    /**
     * GET /employee/patroller-jobs/{id}/report
     *
     * Get comprehensive patroller job report
     */
    public function getJobReport(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $job = $employee->patrollerSchedules()->with([
            'company',
            'route',
            'jobSites' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id)->with('site');
            },
            'issueTickets' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id)->with('site');
            }
        ])->findOrFail($id);

        $pivot = $job->employees->where('id', $employee->id)->first()->pivot ?? null;
        $now = now();
        $status = $this->getPatrollerCalculatedStatus($job, $now);

        // Individual Status: If this specific employee has finished, status is completed.
        // If they have started but not finished, status is active.
        if ($pivot) {
            if (!empty($pivot->actual_end_at)) {
                $status = 'completed';
            } elseif (!empty($pivot->actual_start_at)) {
                $status = 'active';
            }
        }

        $statusColor = $this->getPatrollerStatusColor($status);

        // Format Timings (Use individual pivot times)
        $timings = [
            'scheduled_start' => $job->from_time->format('Y-m-d H:i'),
            'scheduled_end' => $job->to_time->format('Y-m-d H:i'),
            'actual_start' => ($pivot && $pivot->actual_start_at) ? Carbon::parse($pivot->actual_start_at)->format('Y-m-d H:i') : 'N/A',
            'actual_end' => ($pivot && $pivot->actual_end_at) ? Carbon::parse($pivot->actual_end_at)->format('Y-m-d H:i') : 'N/A',
        ];

        // Process Site Visits (Grouped by Site)
        $siteVisits = $job->jobSites->groupBy('site_id')->map(function ($visits) {
            $firstVisit = $visits->first();
            return [
                'site_id' => $firstVisit->site_id,
                'site_name' => $firstVisit->site->name ?? 'N/A',
                'status' => $visits->contains('status', 'checked_in') ? 'checked_in' : ($visits->every('status', 'pending') ? 'pending' : 'checked_out'),
                'visit_count' => $visits->where('status', '!=', 'pending')->count(),
                'logs' => $visits->where('status', '!=', 'pending')->map(function ($visit) {
                    return [
                        'id' => $visit->id,
                        'checkin' => $visit->checked_in_at ? $visit->checked_in_at->format('H:i') : '--:--',
                        'checkout' => $visit->checked_out_at ? $visit->checked_out_at->format('H:i') : '--:--',
                        'duration' => $visit->getVisitDuration() ? $visit->getVisitDuration() . ' mins' : 'N/A',
                    ];
                })->values()
            ];
        })->values();

        // Incidents (Issue Tickets)
        $incidents = $job->issueTickets->map(function ($ticket) {
            return [
                'id' => $ticket->id,
                'site_id' => $ticket->site_id,
                'site_name' => $ticket->site->name ?? 'N/A',
                'ticket_number' => $ticket->ticket_number,
                'description' => $ticket->description,
                'status' => $ticket->status,
                'images' => is_array($ticket->images) ? array_map(function ($img) {
                    return asset(str_starts_with($img, 'storage/') ? $img : 'storage/' . $img);
                }, $ticket->images) : [],
                'created_at' => $ticket->created_at->format('Y-m-d H:i'),
            ];
        });

        // Gallery
        $gallery = [
            'site_visits' => [],
            'incidents' => []
        ];

        foreach ($job->jobSites as $visit) {
            if ($visit->checkin_images) {
                foreach ($visit->checkin_images as $img) {
                    $gallery['site_visits'][] = asset(str_starts_with($img, 'storage/') ? $img : 'storage/' . $img);
                }
            }
        }

        foreach ($incidents as $incident) {
            foreach ($incident['images'] as $imgUrl) {
                $gallery['incidents'][] = $imgUrl;
            }
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'job_details' => [
                    'id' => $job->id,
                    'duty_number' => $job->duty_number,
                    'route_name' => $job->route->name ?? 'N/A',
                    'status' => strtoupper($status),
                    'status_color' => $statusColor,
                    'sites_count' => $job->jobSites->count(),
                ],
                'operative' => [
                    'name' => $employee->first_name . ' ' . $employee->last_name,
                    'id' => $employee->employee_id,
                ],
                'timings' => $timings,
                'site_visits' => $siteVisits,
                'incidents' => $incidents,
                'gallery' => $gallery
            ]
        ]);
    }

    /**
     * Check in to a site
     */
    public function siteCheckin(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = PatrollerSchedule::findOrFail($id);

        // Validate request
        $request->validate([
            'site_id' => 'required|exists:sites,id',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);
        $siteId = $request->site_id;

        // Prevent concurrent check-ins
        $activeCheckin = PatrollerJobSite::where('employee_id', $employee->id)
            ->where('status', 'checked_in')
            ->where('site_id', '!=', $siteId)
            ->with(['site'])
            ->first();

        if ($activeCheckin) {
            $siteName = $activeCheckin->site->name ?? 'another site';
            return response()->json([
                'status' => 'error',
                'message' => "You are currently checked in at {$siteName}. Please checkout there first."
            ], 400);
        }

        // Find the latest job site record
        $latestRecord = PatrollerJobSite::where('patroller_schedule_id', $schedule->id)
            ->where('site_id', $siteId)
            ->where('employee_id', $employee->id)
            ->latest()
            ->first();

        $now = now();
        $actualStartAt = $now->gt($schedule->from_time) ? $now : $schedule->from_time;

        // Auto-start job if needed
        if ($schedule->job_status === 'pending') {
            $schedule->update([
                'job_started_at' => $actualStartAt,
                'job_status' => 'in_progress'
            ]);
        }

        // Ensure this individual employee's deployment is marked as started
        $pivot = $schedule->employees()->where('employees.id', $employee->id)->first()->pivot;
        if (!$pivot->actual_start_at) {
            $schedule->employees()->updateExistingPivot($employee->id, [
                'actual_start_at' => $actualStartAt,
            ]);
        }

        if (!$latestRecord) {
            $jobSite = PatrollerJobSite::create([
                'patroller_schedule_id' => $schedule->id,
                'site_id' => $siteId,
                'employee_id' => $employee->id,
                'status' => 'checked_in',
                'checked_in_at' => now(),
                'checkin_latitude' => $request->latitude,
                'checkin_longitude' => $request->longitude,
            ]);
        } elseif ($latestRecord->status === 'pending') {
            $latestRecord->update([
                'status' => 'checked_in',
                'checked_in_at' => now(),
                'checkin_latitude' => $request->latitude,
                'checkin_longitude' => $request->longitude,
            ]);
            $jobSite = $latestRecord;
        } elseif ($latestRecord->status === 'checked_out') {
            $jobSite = PatrollerJobSite::create([
                'patroller_schedule_id' => $schedule->id,
                'site_id' => $siteId,
                'employee_id' => $employee->id,
                'status' => 'checked_in',
                'checked_in_at' => now(),
                'checkin_latitude' => $request->latitude,
                'checkin_longitude' => $request->longitude,
            ]);
        } else {
            return response()->json(['status' => 'error', 'message' => 'Already checked in to this site'], 400);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Checked in successfully',
            'data' => [
                'site_visit_id' => $jobSite->id,
                'site_id' => $jobSite->site_id,
                'status' => 'active',
                'checked_in_at' => $jobSite->checked_in_at->format('H:i'),
            ]
        ]);
    }

    /**
     * Check out from a site
     */
    public function siteCheckout(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = PatrollerSchedule::findOrFail($id);

        $request->validate([
            'site_id' => 'required|exists:sites,id',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);
        $siteId = $request->site_id;

        // Find the active job site record
        $jobSite = PatrollerJobSite::where('patroller_schedule_id', $schedule->id)
            ->where('site_id', $siteId)
            ->where('employee_id', $employee->id)
            ->where('status', 'checked_in')
            ->latest()
            ->first();

        if (!$jobSite) {
            return response()->json(['status' => 'error', 'message' => 'Site not checked in'], 400);
        }

        $jobSite->update([
            'checked_out_at' => now(),
            'checkout_latitude' => $request->latitude,
            'checkout_longitude' => $request->longitude,
            'status' => 'checked_out',
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Checked out successfully',
            'data' => [
                'site_visit_id' => $jobSite->id,
                'status' => 'completed',
                'duration' => $jobSite->getVisitDuration() . ' mins',
            ]
        ]);
    }

    /**
     * Report an incident (Create Ticket)
     */
    public function reportIncident(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = PatrollerSchedule::findOrFail($id);

        $request->validate([
            'site_id' => 'required|exists:sites,id',
            'ticket_number' => 'required|string|max:50',
            'description' => 'required|string|max:1000',
            'status' => 'required|in:good,bad',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
            'images' => 'nullable|array',
            'images.*' => 'image|max:5120',
        ]);

        // Find the active job site record
        $jobSite = PatrollerJobSite::where('patroller_schedule_id', $schedule->id)
            ->where('site_id', $request->site_id)
            ->where('employee_id', $employee->id)
            ->where('status', 'checked_in')
            ->with('site')
            ->latest()
            ->first();

        if (!$jobSite) {
            return response()->json(['status' => 'error', 'message' => 'You must be checked into the site to report an incident.'], 403);
        }

        // Upload images
        $imagePaths = [];
        if ($request->hasFile('images')) {
            $folderPath = \App\Services\ImageService::getPatrollerFolderPath($schedule->duty_number, $jobSite->site->name);
            foreach ($request->file('images') as $image) {
                $path = \App\Services\ImageService::processAndStore($image, $folderPath);
                if ($path) {
                    $imagePaths[] = $path;
                }
            }
        }

        $ticket = \App\Models\PatrollerIssueTicket::create([
            'patroller_job_site_id' => $jobSite->id,
            'patroller_schedule_id' => $schedule->id,
            'site_id' => $request->site_id,
            'employee_id' => $employee->id,
            'ticket_number' => $request->ticket_number,
            'description' => $request->description,
            'status' => $request->status,
            'images' => $imagePaths,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Incident reported successfully',
            'data' => [
                'ticket_id' => $ticket->id,
                'ticket_number' => $ticket->ticket_number
            ]
        ]);
    }

    /**
     * End the patroller job
     */
    public function endJob(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = PatrollerSchedule::with('employees')->findOrFail($id);

        if (!$schedule->employees->contains($employee->id)) {
            return response()->json(['status' => 'error', 'message' => 'You are not assigned to this schedule'], 403);
        }

        if (!$schedule->isJobInProgress()) {
            return response()->json(['status' => 'error', 'message' => 'Job is not in progress'], 400);
        }

        // Check if employee is still checked in anywhere
        $activeCheckin = PatrollerJobSite::where('employee_id', $employee->id)
            ->where('status', 'checked_in')
            ->exists();

        if ($activeCheckin) {
            return response()->json(['status' => 'error', 'message' => 'You must checkout from current site before ending the job.'], 400);
        }

        // Update the pivot table for the current employee
        $schedule->employees()->updateExistingPivot($employee->id, [
            'actual_end_at' => now(),
        ]);

        // Check if ALL employees have finished
        $activeEmployeesCount = $schedule->employees()
            ->wherePivot('actual_end_at', null)
            ->count();

        if ($activeEmployeesCount === 0) {
            $schedule->update([
                'job_ended_at' => now(),
                'job_status' => 'completed',
            ]);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Job ended successfully',
            'data' => [
                'job_id' => $schedule->id,
                'status' => 'completed',
                'ended_at' => now()->format('Y-m-d H:i:s')
            ]
        ]);
    }

    /**
     * Get incidents for a patroller job
     */
    public function getIncidents(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = PatrollerSchedule::with(['issueTickets.site'])->findOrFail($id);

        if (!$schedule->employees->contains($employee->id)) {
            return response()->json(['status' => 'error', 'message' => 'You are not assigned to this schedule'], 403);
        }

        $incidents = $schedule->issueTickets->map(function ($ticket) {
            return [
                'id' => $ticket->id,
                'ticket_number' => $ticket->ticket_number,
                'site_name' => $ticket->site->name ?? 'N/A',
                'description' => $ticket->description,
                'status' => $ticket->status,
                'images' => is_array($ticket->images) ? array_map(function ($img) {
                    return asset(str_starts_with($img, 'storage/') ? $img : 'storage/' . $img);
                }, $ticket->images) : [],
                'created_at' => $ticket->created_at->format('Y-m-d H:i'),
                'latitude' => $ticket->latitude,
                'longitude' => $ticket->longitude,
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => [
                'incidents' => $incidents
            ]
        ]);
    }

    /**
     * Update an issue ticket
     */
    public function updateIssueTicket(Request $request, $id, $ticketId)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = PatrollerSchedule::findOrFail($id);

        $ticket = \App\Models\PatrollerIssueTicket::where('id', $ticketId)
            ->where('patroller_schedule_id', $schedule->id)
            ->where('employee_id', $employee->id)
            ->firstOrFail();

        $request->validate([
            'ticket_number' => 'required|string|max:50',
            'description' => 'required|string|max:1000',
            'status' => 'required|in:good,bad',
            'images' => 'nullable|array',
            'images.*' => 'image|max:5120',
        ]);

        $imagePaths = $ticket->images ?? [];

        // Add new images if provided
        if ($request->hasFile('images')) {
            $folderPath = \App\Services\ImageService::getPatrollerFolderPath($schedule->duty_number, $ticket->site->name ?? 'Unknown');
            foreach ($request->file('images') as $image) {
                $path = \App\Services\ImageService::processAndStore($image, $folderPath);
                if ($path) {
                    $imagePaths[] = $path;
                }
            }
        }

        $ticket->update([
            'ticket_number' => $request->ticket_number,
            'description' => $request->description,
            'status' => $request->status,
            'images' => $imagePaths,
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Ticket updated successfully',
            'data' => [
                'ticket_id' => $ticket->id,
                'ticket_number' => $ticket->ticket_number
            ]
        ]);
    }

    /**
     * Delete an image from an issue ticket
     */
    public function deleteIssueTicketImage(Request $request, $id, $ticketId)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $ticket = \App\Models\PatrollerIssueTicket::where('id', $ticketId)
            ->where('patroller_schedule_id', $id)
            ->where('employee_id', $employee->id)
            ->firstOrFail();

        $request->validate([
            'image_index' => 'required|integer',
        ]);

        $index = $request->image_index;
        $images = $ticket->images ?? [];

        if (isset($images[$index])) {
            // Optional: Delete file from storage if immediate cleanup is desired
            // \Illuminate\Support\Facades\Storage::delete($images[$index]);

            unset($images[$index]);
            $ticket->update(['images' => array_values($images)]); // Re-index array
        } else {
            return response()->json(['status' => 'error', 'message' => 'Image not found'], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Image deleted successfully'
        ]);
    }

    /**
     * Delete an issue ticket
     */
    public function deleteIssueTicket(Request $request, $id, $ticketId)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $ticket = \App\Models\PatrollerIssueTicket::where('id', $ticketId)
            ->where('patroller_schedule_id', $id)
            ->where('employee_id', $employee->id)
            ->firstOrFail();

        $ticket->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Ticket deleted successfully'
        ]);
    }

    /**
     * POST /employee/patroller-jobs/{id}/cancel
     * 
     * Request cancellation for a patroller job
     */
    public function requestCancellation(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $schedule = $employee->patrollerSchedules()->findOrFail($id);
        if ($schedule->from_time->isPast()) {
            return response()->json(['message' => 'Cancellation requests are only allowed for future patroller schedules that have not started.'], 400);
        }

        // Check if job already started for this specific employee
        $pivot = $schedule->employees()->where('employees.id', $employee->id)->first()->pivot;
        if (!empty($pivot->actual_start_at)) {
            return response()->json(['message' => 'Cannot cancel a job that has already started.'], 400);
        }

        $schedule->employees()->updateExistingPivot($employee->id, [
            'cancellation_requested_at' => \Carbon\Carbon::now(),
            'cancellation_reason' => $request->reason,
            'cancellation_status' => 'pending'
        ]);

        // Notify Company Users
        if ($employee->company) {
            foreach ($employee->company->users as $user) {
                \App\Models\Notification::create([
                    'company_id' => $employee->company_id,
                    'user_id' => $user->id,
                    'title' => 'New Patroller Cancellation Request',
                    'message' => "Employee {$employee->first_name} {$employee->last_name} has requested to cancel Patroller Duty #{$schedule->duty_number}.",
                    'url' => route('patroller-schedules.index', ['search' => $schedule->duty_number]),
                ]);
            }
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Cancellation request submitted for review.'
        ]);
    }

    /**
     * Export patroller job report as PDF
     */
    public function exportPdf(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = PatrollerSchedule::with(['employees', 'route.sites', 'jobSites.site', 'issueTickets.site'])->findOrFail($id);

        if (!$schedule->employees->contains($employee->id)) {
            return response()->json(['status' => 'error', 'message' => 'You are not assigned to this schedule'], 403);
        }

        // Statistics
        $totalTickets = $schedule->issueTickets->count();
        $goodTickets = $schedule->issueTickets->where('status', 'good')->count();
        $badTickets = $schedule->issueTickets->where('status', 'bad')->count();

        // Group tickets by site
        $ticketsBySite = $schedule->issueTickets->groupBy('site_id');

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('employee.patroller-jobs.pdf', compact(
            'schedule',
            'totalTickets',
            'goodTickets',
            'badTickets',
            'ticketsBySite'
        ))->setOptions([
                    'tempDir' => public_path(),
                    'chroot' => public_path(),
                    'isHtml5ParserEnabled' => true,
                    'isRemoteEnabled' => true,
                    'defaultFont' => 'DejaVu Sans',
                    'dpi' => 72, // Reduced DPI to decrease image file size
                    'defaultCompression' => true // Enable internal PDF compression
                ]);

        return $pdf->download("Patrol_Report_{$schedule->duty_number}.pdf");
    }
}
