<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class EmployeeAuthController extends Controller
{
    /**
     * /employee/login
     *
     * Validates credentials and returns a Bearer token for API authentication.
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
            'device_name' => 'nullable|string',
            'fcm_token' => 'nullable|string',
        ]);

        $employee = Employee::where('email', $request->email)->first();

        if (!$employee || !Hash::check($request->password, $employee->password)) {
            throw ValidationException::withMessages([
                'email' => ['The provided credentials are incorrect.'],
            ]);
        }

        if (!$employee->active) {
            return response()->json([
                'message' => 'Your account is currently inactive. Please contact your administrator.'
            ], 403);
        }

        $deviceName = $request->device_name ?? 'mobile_app';
        $token = $employee->createToken($deviceName)->plainTextToken;

        // Handle FCM Token
        if ($request->filled('fcm_token')) {
            $employee->fcmTokens()->updateOrCreate(
                ['token' => $request->fcm_token],
                [
                    'device_id' => $request->device_id ?? null,
                    'device_type' => $request->device_type ?? 'mobile',
                ]
            );
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Login successful',
            'data' => [
                'token' => $token,
                'employee' => [
                    'id' => $employee->id,
                    'first_name' => $employee->first_name,
                    'last_name' => $employee->last_name,
                    'email' => $employee->email,
                    'company_id' => $employee->company_id,
                    'job_role' => $employee->jobRole?->name,
                ]
            ]
        ]);
    }

    /**
     * /employee/logout
     *
     * Revokes the current access token.
     */
    public function logout(Request $request)
    {
        // Remove FCM Token if provided
        if ($request->filled('fcm_token')) {
            $request->user()->fcmTokens()->where('token', $request->fcm_token)->delete();
        }

        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Logged out successfully'
        ]);
    }

    /**
     * Store/Update FCM Token (e.g. on app open)
     */
    public function updateFcmToken(Request $request)
    {
        $request->validate([
            'fcm_token' => 'required|string',
            'device_id' => 'nullable|string',
            'device_type' => 'nullable|string',
        ]);

        $authenticatable = $request->user();

        $data = [
            'token' => $request->fcm_token,
            'device_id' => $request->device_id ?? null,
            'device_type' => $request->device_type ?? 'mobile',
        ];

        if ($authenticatable instanceof \App\Models\User) {
            \App\Models\FcmToken::updateOrCreate(
                ['user_id' => $authenticatable->id, 'token' => $request->fcm_token],
                $data
            );
        } elseif ($authenticatable instanceof \App\Models\Employee) {
            \App\Models\FcmToken::updateOrCreate(
                ['employee_id' => $authenticatable->id, 'token' => $request->fcm_token],
                $data
            );
        }

        return response()->json([
            'status' => 'success',
            'message' => 'FCM Token updated successfully'
        ]);
    }

    /**
     * Send password reset link (forgot password) via API
     */
    public function forgotPassword(Request $request)
    {
        $request->validate(['email' => 'required|email']);

        $employee = Employee::where('email', $request->email)->first();

        if (!$employee) {
            return response()->json(['message' => 'We can\'t find a guard with that email address.'], 404);
        }

        $token = \Illuminate\Support\Str::random(60);

        \Illuminate\Support\Facades\DB::table('password_reset_tokens')->updateOrInsert(
            ['email' => $request->email],
            ['token' => Hash::make($token), 'created_at' => now()]
        );

        // Send Email (using same web route for reset link mostly, but can be deep link for mobile)
        // Here we point to the web reset form, assuming mobile app will open webview or handle deep link
        $resetLink = route('employee.password.reset', ['token' => $token, 'email' => $request->email]);

        try {
            \Illuminate\Support\Facades\Mail::send('emails.employee.password_reset', ['resetLink' => $resetLink, 'employee' => $employee], function ($message) use ($request) {
                $message->to($request->email);
                $message->subject('Reset Password Notification');
            });
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('API Employee Password Reset Email Error: ' . $e->getMessage());
            return response()->json(['message' => 'Failed to send reset link.'], 500);
        }

        return response()->json(['message' => 'Password reset link sent to your email.']);
    }

    /**
     * Reset password via TOKEN (API)
     */
    public function resetPassword(Request $request)
    {
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|confirmed|min:8',
        ]);

        $record = \Illuminate\Support\Facades\DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->first();

        if (!$record || !Hash::check($request->token, $record->token)) {
            return response()->json(['message' => 'Invalid or expired password reset token.'], 400);
        }

        $employee = Employee::where('email', $request->email)->first();

        if (!$employee) {
            return response()->json(['message' => 'Employee not found.'], 404);
        }

        $employee->password = Hash::make($request->password);
        $employee->save();

        // Delete token
        \Illuminate\Support\Facades\DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        return response()->json(['message' => 'Password has been reset successfully.']);
    }
}
