<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Barryvdh\DomPDF\Facade\Pdf;

class EmployeeAPIController extends Controller
{
    use \App\Traits\CalculatesEarnings;
    use \App\Traits\PatrollerJobTransformer;

    /**
     * /employee/jobs
     *
     * Returns a paginated list of assigned jobs (Guard or Patroller).
     */
    public function jobs(Request $request)
    {
        $employee = $request->user();

        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $type = $request->query('type', 'all'); // all, active, today, upcoming, completed, missed, pending
        $today = Carbon::today();

        // Base query with relationships matching the web controller
        $query = $employee->schedules()->with([
            'site',
            'employees',
            'incidents' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            },
            'tourRoutes.checkpoints',
            'checkpointScans' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            }
        ]);

        $now = \Carbon\Carbon::now();

        switch ($type) {
            case 'active':
                $query->where('schedules.status', 'active');
                break;

            case 'today':
                $query->whereDate('schedules.from_datetime', $today)
                    ->where('schedules.to_datetime', '>', $now)
                    ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled']);
                break;

            case 'upcoming':
                $query->whereDate('schedules.from_datetime', '>', $today)
                    ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled']);
                break;

            case 'completed':
                $query->where('schedules.status', 'completed');
                break;

            case 'missed':
                $query->where(function ($q) use ($now) {
                    $q->where('schedules.status', 'missed')
                        ->orWhere(function ($sq) use ($now) {
                            $sq->where('schedules.to_datetime', '<=', $now)
                                ->whereNotIn('schedules.status', ['active', 'completed', 'cancelled']);
                        });
                });
                break;

            case 'pending':
                $query->where('employee_schedule.cancellation_status', 'pending');
                break;

            case 'cancelled':
                $query->where('schedules.status', 'cancelled');
                break;
        }

        $jobs = $query->orderBy('schedules.from_datetime')->paginate($request->query('per_page', 50));

        // Transform jobs to match mobile needs, similar to blade mapping
        $jobs->getCollection()->transform(function ($job) use ($employee) {
            $pivot = $job->pivot;

            return [
                'id' => $job->id,
                'duty_number' => $job->duty_number,
                'status' => (function () use ($job) {
                    $now = \Carbon\Carbon::now();

                    // Check for implicit missed status
                    // If to_datetime is in the past, and status is NOT active, completed, or cancelled
                    if ($job->to_datetime->lte($now) && !in_array($job->status, ['active', 'completed', 'cancelled'])) {
                        return 'missed';
                    }

                    // Dynamically elevate to 'today' if the date is today but DB still says upcoming
                    if ($job->status === 'upcoming' && $job->from_datetime->isToday()) {
                        return 'today';
                    }
                    return $job->status;
                })(),
                'status_color' => (function () use ($job, $pivot) {
                    // Determine status for color mapping
                    $displayStatus = $job->status;
                    $now = \Carbon\Carbon::now();

                    // Check for implicit missed status
                    if ($job->to_datetime->lte($now) && !in_array($job->status, ['active', 'completed', 'cancelled'])) {
                        $displayStatus = 'missed';
                    } elseif ($displayStatus === 'upcoming' && $job->from_datetime->isToday()) {
                        $displayStatus = 'today';
                    }

                    // Base color based on Schedule Status
                    $color = match ($displayStatus) {
                        'active' => '#2563EB',    // Blue-600
                        'completed' => '#16A34A', // Green-600
                        'missed' => '#DC2626',    // Red-600
                        'cancelled' => '#EF4444', // Red-500
                        'today' => '#D97706',     // Amber-600
                        default => '#D97706',     // Amber-600 (Upcoming/Scheduled)
                    };

                    // Override based on Employee Personal Status (Cancellation)
                    if ($pivot->cancellation_status === 'pending') {
                        $color = '#F59E0B'; // Amber-500
                    } elseif ($pivot->cancellation_status === 'approved') {
                        $color = '#EF4444'; // Red-500
                    }

                    return $color;
                })(),
                'from_datetime' => $job->from_datetime->format('Y-m-d H:i:s'),
                'to_datetime' => $job->to_datetime->format('Y-m-d H:i:s'),
                'site' => [
                    'id' => $job->site->id,
                    'name' => $job->site->name,
                    'address' => $job->site->address,
                    'lat' => $job->site->lat,
                    'lng' => $job->site->lng,
                ],
                'deployment_status' => [
                    'has_started' => !empty($pivot->actual_start_at),
                    'actual_start_at' => $pivot->actual_start_at,
                    'actual_end_at' => $pivot->actual_end_at,
                    'checkin_images' => !empty($pivot->checkin_images) ? json_decode($pivot->checkin_images, true) : [],
                    'checkout_images' => !empty($pivot->checkout_images) ? json_decode($pivot->checkout_images, true) : [],
                ],
                'companions' => $job->employees->where('id', '!=', $employee->id)->map(function ($comp) {
                    return [
                        'id' => $comp->id,
                        'name' => $comp->first_name . ' ' . $comp->last_name,
                        'profile_picture_url' => $comp->profile_picture_url,
                    ];
                })->values(),
                'incidents_count' => $job->incidents->count(),
                'checkpoints' => $job->tourRoutes->flatMap->checkpoints->map(function ($c) {
                    return [
                        'name' => $c->name,
                        'code' => $c->checkpoint_id_code,
                        'scan_interval' => $c->scan_interval_minutes,
                        'options' => is_string($c->extra_scan_options) ? json_decode($c->extra_scan_options, true) : $c->extra_scan_options,
                    ];
                })->values(),
                'can_start' => (function () use ($job, $pivot, $employee) {
                    $now = \Carbon\Carbon::now();
                    // 1. Status Check
                    if (in_array($job->status, ['inactive', 'cancelled', 'missed', 'completed'])) {
                        return false;
                    }
                    if ($pivot->cancellation_status === 'approved') {
                        return false;
                    }
                    // 2. Already Started Check
                    if (!empty($pivot->actual_start_at)) {
                        return false;
                    }
                    // 3. Expiry Check
                    if ($job->to_datetime <= $now) {
                        return false;
                    }
                    // 4. Timing/Visibility Check
                    $visibilityMinutes = $employee->getVisibilityMinutes(); // Get cached or DB value
                    $visibilityDeadline = $now->copy()->addMinutes($visibilityMinutes);

                    if ($job->from_datetime > $visibilityDeadline) {
                        return false; // Too early
                    }

                    return true;
                })(),
                'available_actions' => (function () use ($pivot, $job) {
                    // Determine which actions are available for THIS specific guard
                    $hasStarted = !empty($pivot->actual_start_at);
                    $hasEnded = !empty($pivot->actual_end_at);
                    $isCancelled = $pivot->cancellation_status === 'approved';
                    $jobCompleted = in_array($job->status, ['completed', 'cancelled', 'missed']);

                    return [
                        'can_view_tour_progress' => $hasStarted && !$hasEnded && !$isCancelled,
                        'can_report_incident' => $hasStarted && !$hasEnded && !$isCancelled,
                        'can_add_duty_notes' => $hasStarted && !$hasEnded && !$isCancelled,
                        'can_view_mission_gallery' => $hasStarted && !$hasEnded && !$isCancelled,
                        'can_upload_checkout_images' => $hasStarted && !$hasEnded && !$isCancelled,
                        'can_end_shift' => $hasStarted && !$hasEnded && !$isCancelled,
                        'shift_ended' => $hasEnded,
                        'shift_cancelled' => $isCancelled,
                        'can_cancel_shift' => !$hasStarted && !$isCancelled && $pivot->cancellation_status !== 'pending' && $job->from_datetime->isFuture(),
                    ];
                })(),
                'cancellation_details' => [
                    'status' => $pivot->cancellation_status, // null, pending, approved, rejected
                    'reason' => $pivot->cancellation_reason,
                    'requested_at' => $pivot->cancellation_requested_at,
                ],
            ];
        });

        // Get counts matching the web portal's tab counts
        $counts = [
            'all' => $employee->schedules()->count(),
            'active' => $employee->schedules()->where('schedules.status', 'active')->count(),
            'today' => $employee->schedules()
                ->whereDate('schedules.from_datetime', $today)
                ->where('schedules.to_datetime', '>', $now)
                ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled'])
                ->count(),
            'upcoming' => $employee->schedules()
                ->whereDate('schedules.from_datetime', '>', $today)
                ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled'])
                ->count(),
            'completed' => $employee->schedules()->where('schedules.status', 'completed')->count(),
            'missed' => $employee->schedules()->where(function ($q) use ($now) {
                $q->where('schedules.status', 'missed')
                    ->orWhere(function ($sq) use ($now) {
                        $sq->where('schedules.to_datetime', '<=', $now)
                            ->whereNotIn('schedules.status', ['active', 'completed', 'cancelled']);
                    });
            })->count(),
            'pending' => $employee->schedules()->where('employee_schedule.cancellation_status', 'pending')->count(),
            'cancelled' => $employee->schedules()->where('schedules.status', 'cancelled')->count(),
        ];

        return response()->json([
            'status' => 'success',
            'data' => [
                'jobs' => $jobs,
                'counts' => $counts,
                'current_type' => $type,
            ],
        ]);
    }

    /**
     * /employee/profile
     *
     * Returns detailed information about the logged-in employee, including
     * personal details and available metadata for profile updates (countries, states, etc.).
     */
    public function profile(Request $request)
    {
        $employee = $request->user();

        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $employee->load([
            'jobRole',
            'department',
            'permanentCountry',
            'permanentState',
            'correspondingCountry',
            'correspondingState',
            'licenceIssuingProvinceState',
            'skills',
            'identifications',
        ]);

        return response()->json([
            'status' => 'success',
            'data' => [
                'personal_logistics' => [
                    'employee_id' => $employee->employee_id,
                    'email' => $employee->email,
                    'designation' => $employee->jobRole->name ?? 'Unassigned',
                    'first_name' => $employee->first_name,
                    'middle_name' => $employee->middle_name,
                    'last_name' => $employee->last_name,
                    'emergency_number' => $employee->emergency_number,
                    'gender' => $employee->gender,
                    'dob' => $employee->dob ? $employee->dob->format('Y-m-d') : null,
                    'marital_status' => $employee->marital_status,
                ],
                'permanent_address' => [
                    'line_1' => $employee->permanent_address_line_1,
                    'line_2' => $employee->permanent_address_line_2,
                    'city' => $employee->permanent_city,
                    'state' => $employee->permanentState->name ?? 'N/A',
                    'state_id' => $employee->permanent_state_id,
                    'country' => $employee->permanentCountry->name ?? 'N/A',
                    'country_id' => $employee->permanent_country_id,
                    'zip_code' => $employee->permanent_zip_code,
                ],
                'corresponding_address' => [
                    'line_1' => $employee->corresponding_address_line_1,
                    'line_2' => $employee->corresponding_address_line_2,
                    'city' => $employee->corresponding_city,
                    'state' => $employee->correspondingState->name ?? 'N/A',
                    'state_id' => $employee->corresponding_state_id,
                    'country' => $employee->correspondingCountry->name ?? 'N/A',
                    'country_id' => $employee->corresponding_country_id,
                    'zip_code' => $employee->corresponding_zip_code,
                ],
                'operational_qualifications' => [
                    'skills' => $employee->skills->map(function ($skill) {
                        return [
                            'id' => $skill->id,
                            'name' => $skill->name,
                        ];
                    }),
                ],
                'credential_verification' => [
                    'contact_numbers' => [
                        'phone_number' => $employee->phone_number,
                        'cell_number' => $employee->cell_number,
                    ],
                    'security_license' => [
                        'number' => $employee->license_number,
                        'expiry' => $employee->license_expiry,
                        'is_expired' => $employee->license_expiry && \Carbon\Carbon::parse($employee->license_expiry)->isPast(),
                        'file_url' => $employee->file_license ? asset('storage/' . $employee->file_license) : null,
                        'issuing_authority' => $employee->licence_issuing_province ?? ($employee->licenceIssuingProvinceState->name ?? 'N/A'),
                    ],
                    'additional_identifications' => $employee->identifications->map(function ($id) {
                        return [
                            'type' => $id->type,
                            'identification_number' => $id->identification_number,
                            'expiry_date' => $id->expiry_date,
                            'is_expired' => $id->expiry_date && \Carbon\Carbon::parse($id->expiry_date)->isPast(),
                        ];
                    }),
                ],
                'work_fields' => [
                    'hiring_date' => $employee->employment_date,
                    'termination_date' => $employee->terminated_date ?? 'Active',
                    'reporting_manager' => $employee->reporting_manager,
                    'sin_number' => $employee->sin_number,
                    'fax' => $employee->fax,
                    'tags' => $employee->tags,
                ],
                'profile_picture_url' => $employee->profile_picture_url,
            ],
            'meta' => [
                'options' => [
                    'countries' => \App\Models\Country::where('active', true)->orderBy('name')->get(['id', 'name']),
                    'states' => \App\Models\State::where('active', true)->orderBy('name')->get(['id', 'name', 'country_id']), // Added country_id for filtering
                    'genders' => ['Male', 'Female', 'Other'],
                    'marital_statuses' => ['Single', 'Married', 'Divorced', 'Widowed'],
                ],
            ],
        ]);
    }

    /**
     * /employee/profile (Update)
     *
     * Updates the employee's personal and address information. Supporting
     * multipart/form-data for license file uploads.
     */
    public function updateProfile(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $validated = $request->validate([
            // Personal Info
            'first_name' => 'required|string|max:255',
            'middle_name' => 'nullable|string|max:255',
            'last_name' => 'required|string|max:255',
            'gender' => 'required|in:Male,Female,Other',
            'dob' => 'required|date',
            'marital_status' => 'required|string',

            // Contact & Credential
            'phone_number' => 'required|string|max:20',
            'cell_number' => 'nullable|string|max:20',
            'emergency_number' => 'nullable|string|max:20',
            'license_number' => 'nullable|string|max:50',
            'license_expiry' => 'nullable|date',
            'file_license' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120',
            'profile_picture' => 'nullable|image|max:5120', // New validation for profile pic

            // Permanent Address
            'permanent_address_line_1' => 'required|string|max:255',
            'permanent_address_line_2' => 'nullable|string|max:255',
            'permanent_city' => 'required|string|max:100',
            'permanent_state_id' => 'required|exists:states,id',
            'permanent_country_id' => 'required|exists:countries,id',
            'permanent_zip_code' => 'required|string|max:20',

            // Corresponding Address
            'corresponding_address_line_1' => 'nullable|string|max:255',
            'corresponding_address_line_2' => 'nullable|string|max:255',
            'corresponding_city' => 'nullable|string|max:100',
            'corresponding_state_id' => 'nullable|exists:states,id',
            'corresponding_country_id' => 'nullable|exists:countries,id',
            'corresponding_zip_code' => 'nullable|string|max:20',
        ]);

        $updateData = [
            'first_name' => $validated['first_name'],
            'middle_name' => $validated['middle_name'],
            'last_name' => $validated['last_name'],
            'gender' => $validated['gender'],
            'dob' => $validated['dob'],
            'birthday' => $validated['dob'], // Sync legacy column
            'marital_status' => $validated['marital_status'],
            'phone_number' => $validated['phone_number'],
            'cell_number' => $validated['cell_number'],
            'emergency_number' => $validated['emergency_number'],
            'permanent_address_line_1' => $validated['permanent_address_line_1'],
            'permanent_address_line_2' => $validated['permanent_address_line_2'],
            'permanent_city' => $validated['permanent_city'],
            'permanent_state_id' => $validated['permanent_state_id'],
            'permanent_country_id' => $validated['permanent_country_id'],
            'permanent_zip_code' => $validated['permanent_zip_code'],
            'corresponding_address_line_1' => $validated['corresponding_address_line_1'],
            'corresponding_address_line_2' => $validated['corresponding_address_line_2'],
            'corresponding_city' => $validated['corresponding_city'],
            'corresponding_state_id' => $validated['corresponding_state_id'],
            'corresponding_country_id' => $validated['corresponding_country_id'],
            'corresponding_zip_code' => $validated['corresponding_zip_code'],
            'license_number' => $validated['license_number'],
            'license_expiry' => $validated['license_expiry'],
        ];

        if ($request->hasFile('file_license')) {
            $path = $request->file('file_license')->store('employees/licenses', 'public');
            $updateData['file_license'] = $path;
        }

        if ($request->hasFile('profile_picture')) {
            // Using 'employees/photos' to match standard conventions or specific folder if needed
            $path = $request->file('profile_picture')->store('employees/photos', 'public');
            $updateData['profile_picture'] = $path;
        }

        $employee->update($updateData);

        return response()->json([
            'status' => 'success',
            'message' => 'Profile updated successfully',
            'data' => $employee->load([
                'jobRole',
                'department',
                'permanentCountry',
                'permanentState',
                'correspondingCountry',
                'correspondingState',
                'licenceIssuingProvinceState',
                'skills',
                'identifications',
            ]),
        ]);
    }

    /**
     * /employee/dashboard
     *
     * Returns a consolidated view of current jobs, incident categories,
     * earnings history, and active wishes/issued uniforms.
     */
    public function dashboard(Request $request)
    {
        $employee = $request->user();

        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        // Visibility Check
        $visibilityMinutes = $employee->getVisibilityMinutes();
        $visibilityDeadline = \Carbon\Carbon::now()->addMinutes($visibilityMinutes);

        // Active jobs: Only those started by this specific employee
        $activeJobs = $employee->schedules()
            ->with(['site', 'employees'])
            ->whereNotNull('employee_schedule.actual_start_at')
            ->whereNull('employee_schedule.actual_end_at')
            ->orderBy('from_datetime')
            ->get();

        $todayJobs = $employee->schedules()
            ->with(['site', 'employees'])
            ->whereDate('from_datetime', \Carbon\Carbon::today())
            ->where('to_datetime', '>', now())
            ->whereNull('employee_schedule.actual_start_at')
            ->whereNotIn('schedules.status', ['completed', 'inactive', 'cancelled'])
            ->orderBy('from_datetime')
            ->limit(2)
            ->get()
            ->map(function ($job) {
                if ($job->status === 'upcoming') {
                    $job->status = 'today';
                }
                return $job;
            });

        // Completed jobs
        $completedJobs = $employee->schedules()
            ->with(['site'])
            ->whereNotNull('actual_end_at')
            ->orderBy('from_datetime', 'desc')
            ->limit(5)
            ->get();

        // Patroller jobs
        $activePatrollerJobs = $employee->patrollerSchedules()
            ->with(['route.sites'])
            ->where('job_status', 'in_progress')
            ->orderBy('from_time')
            ->get();

        $todayPatrollerJobs = $employee->patrollerSchedules()
            ->with(['route.sites', 'employees'])
            ->whereDate('from_time', \Carbon\Carbon::today())
            ->where('to_time', '>', now())
            ->where('job_status', 'pending')
            ->orderBy('from_time')
            ->limit(2)
            ->get();

        $completedPatrollerJobs = $employee->patrollerSchedules()
            ->with(['route'])
            ->whereIn('job_status', ['completed', 'auto_ended'])
            ->orderBy('from_time', 'desc')
            ->limit(5)
            ->get();

        $now = \Carbon\Carbon::now();
        $activePatrollerJobs = $activePatrollerJobs->map(fn($j) => $this->transformPatrollerJob($j, $employee, $now));
        $todayPatrollerJobs = $todayPatrollerJobs->map(fn($j) => $this->transformPatrollerJob($j, $employee, $now));
        $completedPatrollerJobs = $completedPatrollerJobs->map(fn($j) => $this->transformPatrollerJob($j, $employee, $now));

        $categories = \App\Models\IncidentParentCategory::with([
            'incident_categories' => function ($q) {
                $q->where('status', true);
            },
        ])->where('status', true)->get();

        // --- Earnings Analytics ---
        $months = [];
        for ($i = 11; $i >= 0; $i--) { // Reduced to 6 months for API efficiency
            $months[] = now()->subMonths($i)->format('Y-m');
        }

        $incomeValues = [];
        foreach ($months as $month) {
            $startOfMonth = \Carbon\Carbon::parse($month . '-01')->startOfMonth();
            $endOfMonth = \Carbon\Carbon::parse($month . '-01')->endOfMonth();

            $guardIncome = $employee->schedules()
                ->where('schedules.status', 'completed')
                ->whereBetween('from_datetime', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($j) {
                    return $this->calculateGuardEarnings($j);
                });

            $patrollerIncome = $employee->patrollerSchedules()
                ->whereIn('patroller_schedules.job_status', ['completed', 'auto_ended'])
                ->whereBetween('patroller_schedules.from_time', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($j) {
                    return $this->calculatePatrollerEarnings($j);
                });

            $incomeValues[] = [
                'month' => $startOfMonth->format('M'),
                'total' => round($guardIncome + $patrollerIncome, 2),
            ];
        }

        $today = \Carbon\Carbon::today();
        $birthdayWish = null;
        if ($employee->dob && \Carbon\Carbon::parse($employee->dob)->format('m-d') === $today->format('m-d')) {
            $birthdayWish = 'Happy Birthday, ' . $employee->first_name . '! We hope you have a fantastic day!';
        }

        $hiringAnniversaryWish = null;
        if ($employee->employment_date && \Carbon\Carbon::parse($employee->employment_date)->format('m-d') === $today->format('m-d')) {
            $years = \Carbon\Carbon::parse($employee->employment_date)->age;
            if ($years > 0) {
                $hiringAnniversaryWish = 'Happy Work Anniversary! Celebrating ' . $years . ' year' . ($years > 1 ? 's' : '') . ' with us!';
            }
        }

        $issuedUniforms = $employee->issuedUniforms()->with('variant.uniform')->latest('issued_at')->limit(5)->get();

        // --- Dashboard Counts ---
        $unreadNotificationsCount = \App\Models\Notification::where('employee_id', $employee->id)->whereNull('read_at')->count();
        $totalDeployments = $employee->schedules()->count() + $employee->patrollerSchedules()->count();
        $activeDeployments = $employee->schedules()->where('status', 'active')->count() + $employee->patrollerSchedules()->where('job_status', 'in_progress')->count();

        $last30Days = now()->subDays(30);

        $recentGuardSchedules = $employee->schedules()
            ->where('schedules.status', 'completed')
            ->where('from_datetime', '>=', $last30Days)
            ->get();

        $recentPatrollerSchedules = $employee->patrollerSchedules()
            ->whereIn('patroller_schedules.job_status', ['completed', 'auto_ended'])
            ->where('patroller_schedules.from_time', '>=', $last30Days)
            ->get();

        $shiftHours30d = 0;
        $income30d = 0;

        foreach ($recentGuardSchedules as $j) {
            $duration = $j->from_datetime->floatDiffInHours($j->to_datetime);
            $shiftHours30d += $duration;
            $income30d += $this->calculateGuardEarnings($j);
        }

        foreach ($recentPatrollerSchedules as $j) {
            $income30d += $this->calculatePatrollerEarnings($j);
            // If we need hours for patrollers, it would depend on actual start/end or scheduled
            if ($j->from_time && $j->to_time) {
                $shiftHours30d += \Carbon\Carbon::parse($j->from_time)->floatDiffInHours(\Carbon\Carbon::parse($j->to_time));
            }
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'counts' => [
                    'messages' => $unreadNotificationsCount,
                    'deployments' => $totalDeployments,
                    'active' => $activeDeployments,
                    'shift_hours' => round($shiftHours30d, 1),
                    'income_30d' => round($income30d, 2),
                ],
                'active_jobs' => $activeJobs,
                'today_jobs' => $todayJobs,
                'completed_jobs' => $completedJobs,
                'active_patroller_jobs' => $activePatrollerJobs,
                'today_patroller_jobs' => $todayPatrollerJobs,
                'completed_patroller_jobs' => $completedPatrollerJobs,
                'incident_categories' => $categories,
                'earnings_history' => $incomeValues,
                'wishes' => [
                    'birthday' => $birthdayWish,
                    'anniversary' => $hiringAnniversaryWish,
                ],
                'issued_uniforms' => $issuedUniforms,
            ],
        ]);
    }

    /**
     * /employee/settings
     *
     * Returns notification preferences and language settings.
     */
    public function settings(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'email_notifications' => true, // Placeholder for future settings
                'push_notifications' => true,
                'language' => 'en',
            ],
        ]);
    }

    /**
     * /employee/settings/password
     *
     * Changes the authenticated employee's password. Requires current password validation.
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => ['required', 'current_password'],
            'password' => ['required', 'confirmed', Password::min(8)->letters()->mixedCase()->numbers()->symbols()],
        ]);

        $user = $request->user();
        $user->update([
            'password' => Hash::make($request->password),
        ]);

        // Sync with employee record if separate
        $employee = Employee::where('email', $user->email)->first();
        if ($employee) {
            $employee->update([
                'password' => Hash::make($request->password),
            ]);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Password updated successfully',
        ]);
    }

    /**
     * /employee/analytics
     *
     * Provides comprehensive metrics for the last 30 days and 12-month
     * trends for income and reporting.
     */
    public function analytics(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $rangeStart = now()->subDays(30);
        $lateTolerance = (int) \App\Models\Setting::where('key', 'late_checkin_tolerance')->value('value') ?: 15;

        // --- Guard Job Stats (30 days) ---
        $guardSchedules = $employee->schedules()
            ->where('from_datetime', '>=', $rangeStart)
            ->get();

        $guardShiftCount = $guardSchedules->count();
        $guardCompletedCount = $guardSchedules->whereNotNull('pivot.actual_end_at')->count();
        $guardMissedCount = $guardSchedules->where('to_datetime', '<', now())->whereNull('pivot.actual_start_at')->count();
        // --- Guard Job Stats (30 days) ---
        $guardSchedules = $employee->schedules()
            ->where('from_datetime', '>=', $rangeStart)
            ->get();

        $guardShiftCount = $guardSchedules->count();
        $guardCompletedCount = $guardSchedules->where('status', 'completed')->count();
        $guardMissedCount = $guardSchedules->where('status', 'missed')->count();

        // Matching Web Logic for Cancelled
        $guardCancelledCount = $guardSchedules->filter(fn($j) => $j->status === 'cancelled' || $j->pivot->cancellation_status === 'approved')->count();

        // Matching Web Logic for Upcoming
        $guardUpcomingCount = $guardSchedules->whereIn('status', ['published', 'scheduled', 'upcoming', 'today'])
            ->where('from_datetime', '>', now())
            ->where('pivot.cancellation_status', '!=', 'approved')
            ->count();

        // --- Patroller Job Stats (30 days) ---
        $patrollerSchedules = $employee->patrollerSchedules()
            ->where('patroller_schedules.from_time', '>=', $rangeStart)
            ->get();

        $patrollerShiftCount = $patrollerSchedules->count();
        $patrollerCompletedCount = $patrollerSchedules->whereIn('job_status', ['completed', 'auto_ended'])->count();
        $patrollerMissedCount = $patrollerSchedules->filter(fn($j) => $j->to_time->isPast() && $j->job_status === 'pending')->count();
        $patrollerUpcomingCount = $patrollerSchedules->filter(fn($j) => $j->from_time->isFuture() && $j->job_status === 'pending')->count();

        // --- Attendance & Duration (Aggregated) ---
        $totalPastShifts = 0;
        $onTimeCount = 0;
        $lateCount = 0;
        $totalDurationHours = 0;
        $shiftsWithDuration = 0;

        // Process Guard Jobs
        foreach ($guardSchedules->where('from_datetime', '<=', now())->filter(fn($j) => $j->status !== 'cancelled' && $j->pivot->cancellation_status !== 'approved') as $shift) {
            $totalPastShifts++;
            if ($shift->pivot && $shift->pivot->actual_start_at) {
                $scheduledStart = $shift->from_datetime;
                $actualStart = \Carbon\Carbon::parse($shift->pivot->actual_start_at);
                if ($actualStart->lte($scheduledStart->copy()->addMinutes($lateTolerance))) {
                    $onTimeCount++;
                } else {
                    $lateCount++;
                }
                if ($shift->pivot->actual_end_at) {
                    $totalDurationHours += $actualStart->floatDiffInHours(\Carbon\Carbon::parse($shift->pivot->actual_end_at));
                    $shiftsWithDuration++;
                }
            }
        }

        // Process Patroller Jobs
        foreach ($patrollerSchedules->where('from_time', '<=', now()) as $shift) {
            $totalPastShifts++;
            if ($shift->pivot && $shift->pivot->actual_start_at) {
                $scheduledStart = $shift->from_time;
                $actualStart = \Carbon\Carbon::parse($shift->pivot->actual_start_at);
                if ($actualStart->lte($scheduledStart->copy()->addMinutes($lateTolerance))) {
                    $onTimeCount++;
                } else {
                    $lateCount++;
                }
                if ($shift->pivot->actual_end_at) {
                    $totalDurationHours += $actualStart->floatDiffInHours(\Carbon\Carbon::parse($shift->pivot->actual_end_at));
                    $shiftsWithDuration++;
                }
            }
        }

        $attendanceRate = $totalPastShifts > 0 ? round(($onTimeCount / $totalPastShifts) * 100) : 0;
        $avgShiftDuration = $shiftsWithDuration > 0 ? round($totalDurationHours / $shiftsWithDuration, 1) : 0;

        // --- Incidents (30 days) ---
        $incidentCount = \App\Models\Incident::where('employee_id', $employee->id)
            ->where('created_at', '>=', $rangeStart)
            ->count();

        $patrollerTickets = \App\Models\PatrollerIssueTicket::where('employee_id', $employee->id)
            ->where('created_at', '>=', $rangeStart)
            ->count();

        // --- 12-Month Trends ---
        $months = [];
        for ($i = 11; $i >= 0; $i--) {
            $months[] = now()->subMonths($i)->format('Y-m');
        }

        $incomeData = [];
        $trendIncidentData = [];
        foreach ($months as $month) {
            $startOfMonth = \Carbon\Carbon::parse($month . '-01')->startOfMonth();
            $endOfMonth = \Carbon\Carbon::parse($month . '-01')->endOfMonth();

            // Guard Income
            $guardIncome = $employee->schedules()
                ->where('schedules.status', 'completed')
                ->whereBetween('from_datetime', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($j) {
                    return $this->calculateGuardEarnings($j);
                });

            // Patroller Income
            $patrollerIncome = $employee->patrollerSchedules()
                ->whereIn('job_status', ['completed', 'auto_ended'])
                ->whereBetween('patroller_schedules.from_time', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($j) {
                    return $this->calculatePatrollerEarnings($j);
                });

            $incomeData[] = [
                'month' => $startOfMonth->format('M'),
                'total' => round($guardIncome + $patrollerIncome, 2),
            ];

            // Incidents (Guard)
            $gInc = \App\Models\Incident::where('employee_id', $employee->id)
                ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->count();

            // Tickets (Patroller)
            $pInc = \App\Models\PatrollerIssueTicket::where('employee_id', $employee->id)
                ->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->count();

            $trendIncidentData[] = [
                'month' => $startOfMonth->format('M'),
                'total' => $gInc + $pInc,
            ];
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'summary' => [
                    'attendance_rate' => $attendanceRate,
                    'avg_shift_duration' => $avgShiftDuration,
                    'incident_count' => $incidentCount,
                    'patroller_tickets' => $patrollerTickets,
                    'total_field_reports' => $incidentCount + $patrollerTickets,
                    'current_month_income' => count($incomeData) > 0 ? end($incomeData)['total'] : 0,
                ],
                'guard_operations' => [
                    'total_shifts' => $guardShiftCount,
                    'completed' => $guardCompletedCount,
                    'missed' => $guardMissedCount,
                    'upcoming' => $guardUpcomingCount,
                    'cancelled' => $guardCancelledCount,
                ],
                'patroller_operations' => [
                    'total_routes' => $patrollerShiftCount,
                    'completed' => $patrollerCompletedCount,
                    'missed' => $patrollerMissedCount,
                    'upcoming' => $patrollerUpcomingCount,
                ],
                'trends' => [
                    'income' => $incomeData,
                    'reporting' => $trendIncidentData,
                ],
            ],
        ]);
    }

    /**
     * /employee/requests
     *
     * Returns a paginated list of support/leave requests submitted by the employee.
     */
    public function listRequests(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $requests = \App\Models\EmployeeRequest::where('employee_id', $employee->id)
            ->with(['replies.user'])
            ->orderByDesc('created_at')
            ->paginate($request->query('per_page', 20));

        return response()->json([
            'status' => 'success',
            'data' => $requests,
        ]);
    }

    /**
     * /employee/requests/meta
     *
     * Returns a list of available request categories and their message templates.
     */
    /**
     * /employee/requests/meta
     *
     * Returns a list of available request categories and their message templates.
     */
    public function requestsMeta(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $requestTypes = \App\Models\RequestType::where('company_id', $employee->company_id)
            ->where('is_active', true)
            ->get();

        $templates = [];
        $types = [];

        foreach ($requestTypes as $rt) {
            $templates[$rt->key] = $rt->template ?? '';
            $types[] = ['value' => $rt->key, 'label' => $rt->name];
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'types' => $types,
                'templates' => $templates,
            ],
        ]);
    }

    /**
     * /employee/requests (Store)
     *
     * Submits a new support or leave request. Notifies relevant company users.
     */
    public function storeRequest(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $validated = $request->validate([
            'type' => [
                'required',
                \Illuminate\Validation\Rule::exists('request_types', 'key')
                    ->where('company_id', $employee->company_id)
                    ->where('is_active', true)
            ],
            'subject' => 'nullable|string|max:255',
            'message' => 'required|string',
        ]);

        $emplRequest = \App\Models\EmployeeRequest::create([
            'company_id' => $employee->company_id,
            'employee_id' => $employee->id,
            'type' => $validated['type'],
            'subject' => $validated['subject'] ?? null,
            'message' => $validated['message'],
            'status' => 'pending',
        ]);

        // Notify Company Users
        if ($employee->company && $employee->company->users) {
            foreach ($employee->company->users as $user) {
                \App\Models\Notification::create([
                    'company_id' => $employee->company_id,
                    'user_id' => $user->id,
                    'title' => 'New ' . ucfirst(str_replace('_', ' ', $validated['type'])) . ' Request',
                    'message' => "Employee {$employee->first_name} {$employee->last_name} has submitted a new " . str_replace('_', ' ', $validated['type']) . " request.",
                    'url' => route('employee-requests.show', $emplRequest->id),
                ]);
            }
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Request submitted successfully',
            'data' => $emplRequest,
        ]);
    }

    /**
     * /employee/requests/{id}
     *
     * Returns detailed information about a specific request, including all replies.
     */
    public function showRequest(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $emplRequest = \App\Models\EmployeeRequest::with(['replies.user'])->findOrFail($id);

        if ($emplRequest->employee_id != $employee->id) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized',
            ], 403);
        }

        return response()->json([
            'status' => 'success',
            'data' => $emplRequest,
        ]);
    }

    /**
     * /employee/requests/{id}/reply
     *
     * Submits a new reply to an existing request. Notifies relevant company users.
     */
    public function replyToRequest(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $emplRequest = \App\Models\EmployeeRequest::findOrFail($id);

        if ($emplRequest->employee_id != $employee->id) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized',
            ], 403);
        }

        $validated = $request->validate([
            'message' => 'required|string',
        ]);

        $reply = $emplRequest->replies()->create([
            'is_employee_reply' => true,
            'message' => $validated['message'],
        ]);

        $emplRequest->update(['status' => 'pending']);

        // Notify Company Users
        if ($employee->company && $employee->company->users) {
            foreach ($employee->company->users as $user) {
                \App\Models\Notification::create([
                    'company_id' => $employee->company_id,
                    'user_id' => $user->id,
                    'title' => 'New Reply to Request',
                    'message' => "Employee {$employee->first_name} {$employee->last_name} replied to a request.",
                    'url' => route('employee-requests.show', $emplRequest->id),
                ]);
            }
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Reply sent successfully',
            'data' => $reply->load('user'),
        ]);
    }

    /**
     * /employee/uniforms
     *
     * Returns available uniform inventory and uniforms issued to the employee.
     */
    public function uniforms(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }


        $myUniforms = $employee->issuedUniforms()->with('variant.uniform')->latest('issued_at')->get();

        return response()->json([
            'status' => 'success',
            'data' => $myUniforms,
        ]);
    }

    /**
     * /employee/notifications
     *
     * Returns a paginated list of notifications for the authenticated employee.
     */
    public function notifications(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $notifications = \App\Models\Notification::where('employee_id', $employee->id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'status' => 'success',
            'data' => $notifications,
        ]);
    }

    /**
     * /employee/notifications/{id}/read
     *
     * Marks a specific notification as read.
     */
    public function markNotificationAsRead(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $notification = \App\Models\Notification::where('employee_id', $employee->id)->findOrFail($id);

        $notification->update(['read_at' => now()]);

        return response()->json([
            'status' => 'success',
            'message' => 'Notification marked as read',
        ]);
    }

    /**
     * POST /employee/jobs/{id}/start
     * 
     * Start a job with all web panel validations
     */
    public function startJob(Request $request, $id)
    {
        $request->validate([
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $schedule = $employee->schedules()->findOrFail($id);

        // STRENGTHENED CHECK: Block Inactive, Cancelled, and Missed (archived) jobs
        if (in_array($schedule->status, ['inactive', 'cancelled', 'missed'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'Job has been marked as ' . strtoupper($schedule->status) . ' and cannot be initiated.',
            ], 422);
        }

        // Visibility Check
        $visibilityMinutes = $employee->getVisibilityMinutes();
        $visibilityDeadline = \Carbon\Carbon::now()->addMinutes($visibilityMinutes);

        if ($schedule->from_datetime > $visibilityDeadline) {
            return response()->json([
                'status' => 'error',
                'message' => 'Job is not yet cleared for initiation.',
                'details' => [
                    'scheduled' => $schedule->from_datetime->toIso8601String(),
                    'deadline' => $visibilityDeadline->toIso8601String(),
                    'window_minutes' => $visibilityMinutes,
                ]
            ], 422);
        }

        // Check for Concurrent Active Jobs (Single Active Job Constraint)
        $concurrentJob = $employee->schedules()
            ->wherePivotNotNull('actual_start_at')
            ->wherePivotNull('actual_end_at')
            ->first();

        if ($concurrentJob) {
            return response()->json([
                'status' => 'error',
                'message' => "You are already active on Duty #{$concurrentJob->duty_number}. Please end that duty before starting a new one.",
                'active_job' => [
                    'id' => $concurrentJob->id,
                    'duty_number' => $concurrentJob->duty_number,
                    'from_time' => $concurrentJob->from_datetime->format('H:i'),
                    'to_time' => $concurrentJob->to_datetime->format('H:i'),
                ]
            ], 422);
        }

        if ($schedule->to_datetime <= \Carbon\Carbon::now()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Job has expired and cannot be initiated.',
                'scheduled_end' => $schedule->to_datetime->toIso8601String(),
            ], 422);
        }

        if ($schedule->status === 'active' && $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->whereNotNull('actual_start_at')->exists()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Job is already active for your deployment.',
            ], 422);
        }

        $now = \Carbon\Carbon::now();
        $actualStartAt = $now->gt($schedule->from_datetime) ? $now : $schedule->from_datetime;

        $schedule->employees()->updateExistingPivot($employee->id, [
            'actual_start_at' => $actualStartAt,
            'start_lat' => $request->latitude,
            'start_lng' => $request->longitude,
        ]);

        $schedule->update(['status' => 'active']);

        // Log Start Location
        if ($request->latitude && $request->longitude) {
            \App\Models\JobLocationLog::create([
                'schedule_id' => $schedule->id,
                'employee_id' => $employee->id,
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'recorded_at' => \Carbon\Carbon::now(),
            ]);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Job initiated. Deployment is now ACTIVE. Please submit check-in evidence.',
            'data' => [
                'job_id' => $schedule->id,
                'duty_number' => $schedule->duty_number,
                'started_at' => $actualStartAt->toIso8601String(),
            ]
        ]);
    }

    /**
     * POST /employee/jobs/{id}/checkin
     * 
     * Upload check-in images
     */
    public function checkin(Request $request, $id)
    {
        $request->validate([
            'images' => 'required|array|min:1',
            'images.*' => 'image|max:10240',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $schedule = $employee->schedules()->findOrFail($id);

        $images = [];
        if ($request->hasFile('images')) {
            $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;
            $timestamp = $pivot->actual_start_at;
            $folderPath = \App\Services\ImageService::getFolderPath('check-in', $schedule->duty_number, $schedule->site->name, $timestamp);

            foreach ($request->file('images') as $file) {
                $path = \App\Services\ImageService::processAndStore($file, $folderPath);
                if ($path) {
                    $images[] = $path;
                }
            }
        }

        $schedule->employees()->updateExistingPivot($employee->id, [
            'checkin_images' => json_encode($images),
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Check-in evidence successfully archived. Job profile updated.',
            'data' => [
                'images_count' => count($images),
                'images' => $images,
            ]
        ]);
    }

    /**
     * POST /employee/jobs/{id}/incident
     * 
     * Report an incident during active job
     */
    public function reportIncident(Request $request, $id)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'description' => 'required|string',
            'images.*' => 'nullable|image|max:5120',
            'incident_parent_category_id' => 'required|exists:incident_parent_categories,id',
            'incident_category_id' => 'nullable|exists:incident_categories,id',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $schedule = $employee->schedules()->findOrFail($id);

        $images = [];
        if ($request->hasFile('images')) {
            $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;
            $timestamp = $pivot->actual_start_at;
            $folderPath = \App\Services\ImageService::getFolderPath('report-incident', $schedule->duty_number, $schedule->site->name, $timestamp);

            foreach ($request->file('images') as $file) {
                $path = \App\Services\ImageService::processAndStore($file, $folderPath);
                if ($path) {
                    $images[] = $path;
                }
            }
        }

        $incident = \App\Models\Incident::create([
            'employee_id' => $employee->id,
            'schedule_id' => $schedule->id,
            'subject' => $request->subject,
            'description' => $request->description,
            'images' => $images,
            'incident_parent_category_id' => $request->incident_parent_category_id,
            'incident_category_id' => $request->incident_category_id,
        ]);

        // Log Incident Location
        if ($request->latitude && $request->longitude) {
            \App\Models\JobLocationLog::create([
                'schedule_id' => $schedule->id,
                'employee_id' => $employee->id,
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'recorded_at' => now(),
            ]);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Incident report filed and prioritized for review.',
            'data' => [
                'incident_id' => $incident->id,
                'images_count' => count($images),
            ]
        ]);
    }

    /**
     * POST /employee/jobs/{id}/checkout-evidence
     * 
     * Upload checkout/extraction evidence
     */
    public function uploadCheckoutEvidence(Request $request, $id)
    {
        $request->validate([
            'images' => 'required|array|min:1',
            'images.*' => 'image|max:5120',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $schedule = $employee->schedules()->findOrFail($id);

        // Check for existing pivot data
        $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;
        $existingImages = $pivot->checkout_images ? json_decode($pivot->checkout_images, true) : [];

        $newImages = [];
        if ($request->hasFile('images')) {
            $timestamp = $pivot->actual_start_at;
            $folderPath = \App\Services\ImageService::getFolderPath('check-out', $schedule->duty_number, $schedule->site->name, $timestamp);

            foreach ($request->file('images') as $file) {
                $path = \App\Services\ImageService::processAndStore($file, $folderPath);
                if ($path) {
                    $newImages[] = $path;
                }
            }
        }

        $allImages = array_merge($existingImages, $newImages);

        $schedule->employees()->updateExistingPivot($employee->id, [
            'checkout_images' => json_encode($allImages),
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Checkout images uploaded successfully.',
            'data' => [
                'total_images' => count($allImages),
                'new_images' => count($newImages),
                'images' => $allImages,
            ]
        ]);
    }

    /**
     * POST /employee/jobs/{id}/end
     * 
     * End a job with validation for required evidence
     */
    public function endJob(Request $request, $id)
    {
        $request->validate([
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $schedule = $employee->schedules()->findOrFail($id);
        $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;

        $errors = [];
        if (empty($pivot->checkin_images)) {
            $errors[] = "Infiltration Evidence (Check-in Photos) missing.";
        }
        if (empty($pivot->checkout_images)) {
            $errors[] = "Extraction Evidence (Checkout Photos) missing.";
        }

        if (count($errors) > 0) {
            return response()->json([
                'status' => 'error',
                'message' => 'Job Clearance Denied',
                'errors' => $errors,
            ], 422);
        }

        $schedule->employees()->updateExistingPivot($employee->id, [
            'actual_end_at' => \Carbon\Carbon::now(),
            'end_lat' => $request->latitude,
            'end_lng' => $request->longitude,
        ]);

        // Log End Location
        if ($request->latitude && $request->longitude) {
            \App\Models\JobLocationLog::create([
                'schedule_id' => $schedule->id,
                'employee_id' => $employee->id,
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'recorded_at' => \Carbon\Carbon::now(),
            ]);
        }

        // Refresh the relation to check status of all employees
        $schedule->load('employees');

        // Check if ANY employee still has actual_end_at as null
        $hasActiveEmployees = $schedule->employees->contains(function ($emp) {
            return is_null($emp->pivot->actual_end_at);
        });

        if (!$hasActiveEmployees) {
            $schedule->update(['status' => 'completed']);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Job Terminated. Extraction complete.',
            'data' => [
                'job_id' => $schedule->id,
                'duty_number' => $schedule->duty_number,
                'ended_at' => \Carbon\Carbon::now()->toIso8601String(),
                'job_status' => $schedule->fresh()->status,
            ]
        ]);
    }

    /**
     * DELETE /employee/jobs/{id}/evidence/{type}/{index}
     * 
     * Delete a specific evidence image
     */
    public function deleteEvidence($id, $type, $index)
    {
        $employee = request()->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $schedule = $employee->schedules()->findOrFail($id);
        $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;

        if ($type === 'checkin') {
            $images = json_decode($pivot->checkin_images, true) ?? [];
            if (isset($images[$index])) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($images[$index]);
                unset($images[$index]);
                $schedule->employees()->updateExistingPivot($employee->id, [
                    'checkin_images' => json_encode(array_values($images)),
                ]);
            }
        } elseif ($type === 'checkout') {
            $images = json_decode($pivot->checkout_images, true) ?? [];
            if (isset($images[$index])) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($images[$index]);
                unset($images[$index]);
                $schedule->employees()->updateExistingPivot($employee->id, [
                    'checkout_images' => json_encode(array_values($images)),
                ]);
            }
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Evidence deleted successfully.',
        ]);
    }

    /**
     * GET /employee/incident-categories
     * 
     * Fetch all incident categories with sub-categories for reporting
     */
    public function incidentCategories()
    {
        $categories = \App\Models\IncidentParentCategory::with([
            'incident_categories' => function ($q) {
                $q->where('status', 1);
            }
        ])
            ->where('status', 1)
            ->get();

        return response()->json([
            'status' => 'success',
            'data' => $categories
        ]);
    }

    /**
     * POST /employee/jobs/{id}/location
     * 
     * Log periodic GPS location from mobile app
     */
    public function logLocation(Request $request, $id)
    {
        $request->validate([
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $schedule = $employee->schedules()->findOrFail($id);

        // Log the location
        $log = \App\Models\JobLocationLog::create([
            'schedule_id' => $schedule->id,
            'employee_id' => $employee->id,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'recorded_at' => now(),
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Location logged',
            'data' => $log
        ]);
    }

    /**
     * GET /employee/jobs
     * 
     * List jobs with filtering (Active, Today, Upcoming, etc.)
     */
    public function listJobs(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $type = $request->query('type', 'all'); // all, active, today, upcoming, completed, missed, pending, cancelled
        $search = $request->query('search');

        $now = \Carbon\Carbon::now();
        $today = \Carbon\Carbon::today();

        // Base query with relationships
        $query = $employee->schedules()->with(['site', 'employees', 'incidents']);

        // Search functionality
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('duty_number', 'like', "%{$search}%")
                    ->orWhereHas('site', function ($sq) use ($search) {
                        $sq->where('name', 'like', "%{$search}%");
                    });
            });
        }

        switch ($type) {
            case 'active':
                $query->where('schedules.status', 'active');
                break;

            case 'today':
                $query->whereDate('schedules.from_datetime', $today)
                    ->where('schedules.to_datetime', '>', $now)
                    ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled']);
                break;

            case 'upcoming':
                $query->whereDate('schedules.from_datetime', '>', $today)
                    ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled']);
                break;

            case 'completed':
                $query->where('schedules.status', 'completed');
                break;

            case 'missed':
                $query->where(function ($q) use ($now) {
                    $q->where('schedules.status', 'missed')
                        ->orWhere(function ($sq) use ($now) {
                            $sq->where('schedules.to_datetime', '<=', $now)
                                ->whereNotIn('schedules.status', ['active', 'completed', 'cancelled']);
                        });
                });
                break;

            case 'pending':
                $query->where('employee_schedule.cancellation_status', 'pending');
                break;

            case 'cancelled':
                $query->where('schedules.status', 'cancelled');
                break;
        }

        $jobs = $query->orderBy('schedules.from_datetime')->paginate(20);

        $jobs->getCollection()->transform(function ($job) use ($employee) {
            $pivot = $job->pivot;

            // Check if job should be considered missed (implied logic from web query)
            // Logic: if to_datetime <= now AND status is not active, completed, or cancelled => it is missed
            $isMissed = false;
            $now = \Carbon\Carbon::now();

            // Check for implied missed status regardless of current db status label (unless it's already final)
            if ($job->to_datetime <= $now && !in_array($job->status, ['active', 'completed', 'cancelled', 'missed'])) {
                $job->status = 'missed';
                $isMissed = true;
            }
            if ($job->status === 'missed') {
                $isMissed = true;
            }

            $statusColor = match ($job->status) {
                'active' => '#2563EB',
                'completed' => '#16A34A',
                'missed' => '#DC2626',
                'cancelled' => '#EF4444',
                default => '#D97706',
            };

            if (isset($pivot->cancellation_status)) {
                if ($pivot->cancellation_status === 'pending') {
                    $statusColor = '#F59E0B';
                } elseif ($pivot->cancellation_status === 'approved') {
                    $statusColor = '#EF4444';
                }
            }

            // Force inclusion of status_color
            $job->status_color = $statusColor;
            $job->cancellation_status = $pivot->cancellation_status ?? null;

            $job->can_start = (function () use ($job, $pivot, $employee, $now) {
                // 1. Status Check
                if (in_array($job->status, ['inactive', 'cancelled', 'missed', 'completed'])) {
                    return false;
                }
                if (($pivot->cancellation_status ?? null) === 'approved') {
                    return false;
                }
                // 2. Already Started Check
                if (!empty($pivot->actual_start_at)) {
                    return false;
                }
                // 3. Expiry Check
                if ($job->to_datetime <= $now) {
                    return false;
                }
                // 4. Timing/Visibility Check
                $visibilityMinutes = $employee->getVisibilityMinutes();
                $visibilityDeadline = $now->copy()->addMinutes($visibilityMinutes);

                if ($job->from_datetime > $visibilityDeadline) {
                    return false; // Too early
                }

                return true;
            })();

            return $job;
        });

        // Calculate counts for badges
        $counts = [
            'all' => $employee->schedules()->count(),
            'active' => $employee->schedules()->where('schedules.status', 'active')->count(),
            'today' => $employee->schedules()
                ->whereDate('schedules.from_datetime', $today)
                ->where('schedules.to_datetime', '>', $now)
                ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled'])
                ->count(),
            'upcoming' => $employee->schedules()
                ->whereDate('schedules.from_datetime', '>', $today)
                ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled'])
                ->count(),
            'completed' => $employee->schedules()->where('schedules.status', 'completed')->count(),
            'missed' => $employee->schedules()->where(function ($q) use ($now) {
                $q->where('schedules.status', 'missed')
                    ->orWhere(function ($sq) use ($now) {
                        $sq->where('schedules.to_datetime', '<=', $now)
                            ->whereNotIn('schedules.status', ['active', 'completed', 'cancelled']);
                    });
            })->count(),
            'cancelled' => $employee->schedules()->where('schedules.status', 'cancelled')->count(),
            'pending' => $employee->schedules()->where('employee_schedule.cancellation_status', 'pending')->count()
        ];

        return response()->json([
            'status' => 'success',
            'data' => [
                'jobs' => $jobs,
                'counts' => $counts
            ]
        ]);
    }

    /**
     * POST /employee/jobs/{id}/cancel
     * 
     * Request cancellation for a job
     */
    public function requestCancellation(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        if (!$employee) {
            return response()->json(['message' => 'Employee profile not found'], 404);
        }

        $schedule = $employee->schedules()->findOrFail($id);
        if ($schedule->from_datetime->isPast()) {
            return response()->json(['message' => 'Cancellation requests are only allowed for future schedules that have not started.'], 400);
        }

        $schedule->employees()->updateExistingPivot($employee->id, [
            'cancellation_requested_at' => \Carbon\Carbon::now(),
            'cancellation_reason' => $request->reason,
            'cancellation_status' => 'pending'
        ]);

        // Notify Company Users
        if ($employee->company) {
            foreach ($employee->company->users as $user) {
                \App\Models\Notification::create([
                    'company_id' => $employee->company_id,
                    'user_id' => $user->id,
                    'title' => 'New Cancellation Request',
                    'message' => "Employee {$employee->first_name} {$employee->last_name} has requested to cancel Duty #{$schedule->duty_number}.",
                    'url' => route('schedules.index', ['type' => 'all', 'search' => $schedule->duty_number]),
                ]);
            }
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Cancellation request submitted for review.'
        ]);
    }

    /**
     * GET /employee/jobs/{id}/reliever
     *
     * Get information about the reliever for a shift
     */
    public function getRelieverInfo(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = $employee->schedules()->findOrFail($id);

        $nextSchedule = \App\Models\Schedule::where('site_id', $schedule->site_id)
            ->where('from_datetime', '>=', $schedule->to_datetime->subMinutes(30))
            ->where('from_datetime', '<=', $schedule->to_datetime->addHours(4))
            ->where('id', '!=', $schedule->id)
            ->whereNotIn('status', ['cancelled', 'missed'])
            ->with(['employees'])
            ->orderBy('from_datetime', 'asc')
            ->first();

        if (!$nextSchedule) {
            return response()->json([
                'status' => 'success',
                'data' => null,
                'message' => 'No reliever found for this shift.'
            ]);
        }

        $reliever = $nextSchedule->employees->where('id', '!=', $employee->id)->first();

        if (!$reliever) {
            // Maybe same employee?
            $reliever = $nextSchedule->employees->first();
            if ($reliever && $reliever->id == $employee->id) {
                return response()->json([
                    'status' => 'success',
                    'data' => ['self_relief' => true],
                    'message' => 'You are scheduled for the next shift as well.'
                ]);
            }

            return response()->json([
                'status' => 'success',
                'data' => [
                    'site_name' => $schedule->site->name,
                    'start_time' => $nextSchedule->from_datetime->format('H:i'),
                    'employee' => null
                ]
            ]);
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'site_name' => $schedule->site->name,
                'start_time' => $nextSchedule->from_datetime->format('H:i'),
                'employee' => [
                    'name' => $reliever->first_name . ' ' . $reliever->last_name,
                    'photo' => $reliever->profile_picture_url,
                    'phone' => $reliever->phone_number,
                    'id' => $reliever->employee_id
                ]
            ]
        ]);
    }

    /**
     * GET /employee/jobs/{id}/report
     *
     * Get comprehensive job log report
     */
    public function getJobReport(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = $employee->schedules()->with([
            'site',
            'incidents' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            },
            'checkpointScans' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            }
        ])->findOrFail($id);
        $pivot = $schedule->pivot;

        // Dynamic status elevation for today
        $displayStatus = $schedule->status;
        if ($displayStatus === 'upcoming' && $schedule->from_datetime->isToday()) {
            $displayStatus = 'today';
        }

        // Determine Status Color logic (same as listJobs)
        $statusColor = match ($displayStatus) {
            'active' => '#2563EB',
            'completed' => '#16A34A',
            'missed' => '#DC2626',
            'cancelled' => '#EF4444',
            'today' => '#D97706',
            default => '#D97706',
        };

        if (isset($pivot->cancellation_status)) {
            if ($pivot->cancellation_status === 'pending') {
                $statusColor = '#F59E0B';
            } elseif ($pivot->cancellation_status === 'approved') {
                $statusColor = '#EF4444';
            }
        }

        // Format Timings
        $timings = [
            'scheduled_start' => $schedule->from_datetime->format('Y-m-d H:i'),
            'scheduled_end' => $schedule->to_datetime->format('Y-m-d H:i'),
            'actual_start' => $pivot->actual_start_at ? \Carbon\Carbon::parse($pivot->actual_start_at)->format('Y-m-d H:i') : 'N/A',
            'actual_end' => $pivot->actual_end_at ? \Carbon\Carbon::parse($pivot->actual_end_at)->format('Y-m-d H:i') : 'N/A',
            'start_coords' => [$pivot->start_lat ?? 'N/A', $pivot->start_lng ?? 'N/A'],
            'end_coords' => [$pivot->end_lat ?? 'N/A', $pivot->end_lng ?? 'N/A'],
        ];

        // Process Evidence Images
        $checkinImages = json_decode($pivot->checkin_images, true) ?? [];
        $checkoutImages = json_decode($pivot->checkout_images, true) ?? [];

        // Ensure images have full URLs and point to storage
        $checkinImages = array_map(function ($img) {
            return asset(str_starts_with($img, 'storage/') ? $img : 'storage/' . $img);
        }, $checkinImages);

        $checkoutImages = array_map(function ($img) {
            return asset(str_starts_with($img, 'storage/') ? $img : 'storage/' . $img);
        }, $checkoutImages);

        // Incidents with Images
        $incidents = $schedule->incidents->where('employee_id', $employee->id)->values()->map(function ($incident) {
            return [
                'id' => $incident->id,
                'subject' => $incident->subject,
                'description' => $incident->description,
                'images' => is_array($incident->images) ? array_map(function ($img) {
                    return asset(str_starts_with($img, 'storage/') ? $img : 'storage/' . $img);
                }, $incident->images) : [],
                'created_at' => $incident->created_at->format('Y-m-d H:i'),
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => [
                'job_details' => [
                    'id' => $schedule->id,
                    'duty_number' => $schedule->duty_number,
                    'site_name' => $schedule->site->name,
                    'status' => strtoupper($displayStatus),
                    'status_color' => $statusColor,
                    'customer_reference' => $schedule->customer_reference,
                    'gas' => $schedule->gas,
                    'comments' => $schedule->comments,
                ],
                'operative' => [
                    'name' => $employee->first_name . ' ' . $employee->last_name,
                    'id' => $employee->employee_id,
                ],
                'timings' => $timings,
                'remarks' => $pivot->additional_remarks ?? 'No remarks recorded.',
                'incidents' => $incidents,
                'gallery' => [
                    'checkin' => $checkinImages,
                    'checkout' => $checkoutImages,
                ]
            ]
        ]);
    }

    /**
     * GET /employee/jobs/{id}/notes
     *
     * Get notes for a job
     */
    public function getNotes(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $jobType = $request->query('job_type', 'guard');

        // Verify job ownership
        if ($jobType === 'guard') {
            $employee->schedules()->findOrFail($id);
        } else {
            $employee->patrollerSchedules()->findOrFail($id);
        }

        $notes = \App\Models\JobNote::where('job_id', $id)
            ->where('job_type', $jobType)
            ->where('employee_id', $employee->id)
            ->latest()
            ->get()
            ->map(function ($note) {
                return [
                    'id' => $note->id,
                    'note' => $note->note,
                    'image' => $note->image ? asset('storage/' . $note->image) : null,
                    'created_at' => $note->created_at->format('Y-m-d H:i'),
                    'start_time' => $note->created_at->format('H:i') // For UI compatibility if needed
                ];
            });

        return response()->json([
            'status' => 'success',
            'data' => $notes
        ]);
    }

    /**
     * POST /employee/jobs/{id}/notes
     *
     * Store a new note for a job
     */
    public function storeNote(Request $request, $id)
    {
        $request->validate([
            'note' => 'required|string',
            'job_type' => 'nullable|in:guard,patroller',
            'image' => 'nullable|image|max:5120',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $jobType = $request->input('job_type', 'guard');

        // Verify job
        $schedule = null;
        $dutyNumber = null;
        $siteName = null;

        if ($jobType === 'guard') {
            $schedule = $employee->schedules()->findOrFail($id);
            $dutyNumber = $schedule->duty_number;
            $siteName = $schedule->site->name;
        } else {
            $schedule = $employee->patrollerSchedules()->findOrFail($id);
            $dutyNumber = $schedule->duty_number;
            $siteName = 'Patrol Route';
        }

        $imagePath = null;
        if ($request->hasFile('image')) {
            $folderPath = \App\Services\ImageService::getFolderPath('job-notes', $dutyNumber, $siteName, now());
            $imagePath = \App\Services\ImageService::processAndStore($request->file('image'), $folderPath);
        }

        \App\Models\JobNote::create([
            'employee_id' => $employee->id,
            'job_type' => $jobType,
            'job_id' => $id,
            'note' => $request->note,
            'image' => $imagePath,
        ]);

        // Log Location (Only for Guard jobs for now as JobLocationLog links to schedules)
        if ($jobType === 'guard' && $request->latitude && $request->longitude) {
            \App\Models\JobLocationLog::create([
                'schedule_id' => $schedule->id,
                'employee_id' => $employee->id,
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'recorded_at' => now(),
            ]);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Note saved successfully.'
        ]);
    }

    /**
     * GET /employee/jobs/{id}/tour-progress
     *
     * Get tour progress, checkpoints, and scans for a job
     */
    public function getTourProgress(Request $request, $id)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = $employee->schedules()
            ->with([
                'tourRoutes.checkpoints',
                'checkpointScans' => function ($q) use ($employee) {
                    $q->where('employee_id', $employee->id);
                },
                'checkpointScans.checkpoint'
            ])
            ->findOrFail($id);

        $tourData = [];
        $now = now();

        foreach ($schedule->tourRoutes as $tour) {
            $checkpoints = $tour->checkpoints->map(function ($cp) use ($schedule, $now) {
                // Find all scans for this checkpoint in this schedule
                $scans = $schedule->checkpointScans->where('checkpoint_id', $cp->id)->sortByDesc('scanned_at');
                $lastScan = $scans->first();
                $isScanned = $scans->isNotEmpty();

                $nextScanAt = $lastScan ? $lastScan->scanned_at->addMinutes($cp->scan_interval_minutes) : null;
                $isOverdue = $nextScanAt ? $now->greaterThan($nextScanAt) : false;

                // Decode options matching web logic
                $opts = is_string($cp->extra_scan_options) ? json_decode($cp->extra_scan_options, true) : $cp->extra_scan_options;

                return [
                    'id' => $cp->id,
                    'name' => $cp->name,
                    'checkpoint_id_code' => $cp->checkpoint_id_code,
                    'latitude' => $cp->latitude,
                    'longitude' => $cp->longitude,
                    'is_scanned' => $isScanned,
                    'scanned_at' => $lastScan ? $lastScan->scanned_at->format('H:i:s') : null,
                    'scanned_at_human' => $lastScan ? $lastScan->scanned_at->diffForHumans() : 'Never',
                    'scan_image' => $lastScan && $lastScan->evidence_image ? asset('storage/' . $lastScan->evidence_image) : null,
                    'scan_interval_minutes' => $cp->scan_interval_minutes,
                    'grace_period_minutes' => $cp->grace_period_minutes,
                    'next_scan_at' => $nextScanAt ? $nextScanAt->format('H:i') : null,
                    'is_overdue' => $isOverdue,
                    'options' => $opts,
                    'status_text' => $isScanned ? 'Verified' : 'Pending Initial Scan',
                ];
            });

            $tourData[] = [
                'id' => $tour->id,
                'name' => $tour->name,
                'description' => $tour->description,
                'checkpoints' => $checkpoints->values()
            ];
        }

        // Recent History (Last 10 scans across all tours in this schedule)
        $history = $schedule->checkpointScans()
            ->where('employee_id', $employee->id)
            ->with('checkpoint:id,name')
            ->latest('scanned_at')
            ->limit(10)
            ->get()
            ->map(function ($scan) {
                return [
                    'checkpoint_name' => $scan->checkpoint->name ?? 'Unknown',
                    'scanned_at' => $scan->scanned_at->format('H:i:s'),
                    'scanned_at_human' => $scan->scanned_at->diffForHumans(),
                    'is_manual' => (bool) $scan->is_manual,
                ];
            });

        return response()->json([
            'status' => 'success',
            'data' => [
                'tours' => $tourData,
                'history' => $history,
                'stats' => [
                    'total_checkpoints' => $schedule->tourRoutes->flatMap(fn($t) => $t->checkpoints)->count(),
                    'scanned_checkpoints' => $schedule->checkpointScans->where('employee_id', $employee->id)->unique('checkpoint_id')->count(),
                ]
            ]
        ]);
    }

    /**
     * POST /employee/jobs/{id}/scan
     *
     * Scan a checkpoint (QR/NFC) code
     */
    public function scanCheckpoint(Request $request, $id)
    {
        $request->validate([
            'checkpoint_code' => 'required|string',
            'latitude' => 'nullable',
            'longitude' => 'nullable',
            'image' => 'nullable|image|max:5120',
            'employee_message' => 'nullable|string',
            'additional_photos.*' => 'nullable|image|max:5120',
        ]);

        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $schedule = $employee->schedules()->with('tourRoutes.checkpoints')->findOrFail($id);

        // Find the checkpoint by code within the assigned tour routes
        $checkpoint = null;
        $foundTourRoute = null;

        foreach ($schedule->tourRoutes as $tour) {
            $cp = $tour->checkpoints->where('checkpoint_id_code', $request->checkpoint_code)->first();
            if ($cp) {
                $checkpoint = $cp;
                $foundTourRoute = $tour;
                break;
            }
        }

        if (!$checkpoint) {
            return response()->json(['message' => 'Checkpoint ID #' . $request->checkpoint_code . ' does not match any checkpoints in your assigned tours.'], 404);
        }

        $imagePath = null;
        if ($request->hasFile('image')) {
            $folderPath = \App\Services\ImageService::getFolderPath('checkpoint-scans', $schedule->duty_number, $schedule->site->name, now());
            $imagePath = \App\Services\ImageService::processAndStore($request->file('image'), $folderPath);
        }

        $additionalPhotos = [];
        if ($request->hasFile('additional_photos')) {
            $folderPath = \App\Services\ImageService::getFolderPath('checkpoint-scans', $schedule->duty_number, $schedule->site->name, now());
            foreach ($request->file('additional_photos') as $photo) {
                $additionalPhotos[] = \App\Services\ImageService::processAndStore($photo, $folderPath);
            }
        }

        \App\Models\CheckpointScan::create([
            'schedule_id' => $schedule->id,
            'employee_id' => $employee->id,
            'checkpoint_id' => $checkpoint->id,
            'tour_route_id' => $foundTourRoute->id,
            'scanned_at' => now(),
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'is_manual' => true,
            'scan_code' => $request->checkpoint_code,
            'evidence_image' => $imagePath,
            'employee_message' => $request->employee_message,
            'additional_photos' => $additionalPhotos,
        ]);

        // Log Checkpoint Scan Location
        if ($request->latitude && $request->longitude) {
            \App\Models\JobLocationLog::create([
                'schedule_id' => $schedule->id,
                'employee_id' => $employee->id,
                'latitude' => $request->latitude,
                'longitude' => $request->longitude,
                'recorded_at' => now(),
            ]);
        }

        $nextScanTime = now()->addMinutes($checkpoint->scan_interval_minutes)->format('H:i');

        return response()->json([
            'status' => 'success',
            'message' => "Verified: [{$checkpoint->name}] has been scanned. Protocol updated. The next required scan is scheduled for {$nextScanTime}.",
            'data' => [
                'next_scan_time' => $nextScanTime,
                'checkpoint' => $checkpoint
            ]
        ]);
    }

    /**
     * GET /employee/operational-reports/filters
     */
    public function operationalReportFilters(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $sites = $employee->schedules()
            ->with('site')
            ->get()
            ->pluck('site')
            ->whereNotNull()
            ->unique('id')
            ->sortBy('name')
            ->values();

        return response()->json([
            'status' => 'success',
            'data' => [
                'sites' => $sites,
                'report_types' => [
                    ['id' => 'checkin', 'name' => 'Check-in Report'],
                    ['id' => 'checkout', 'name' => 'Checkout Report'],
                    ['id' => 'combined', 'name' => 'Check-IN/OUT Report'],
                    ['id' => 'scans', 'name' => 'Checkpoint Scans'],
                ]
            ]
        ]);
    }

    /**
     * GET /employee/operational-reports
     */
    public function operationalReports(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $reportType = $request->query('type', 'checkin');
        $isHistory = $reportType === 'scans_history' || ($reportType === 'scans' && $request->filled('schedule_id'));
        $perPage = $isHistory ? 40 : 20;

        $query = $this->getOperationalReportQuery($request, $employee);
        $reports = $query->paginate($perPage);

        // Transform logic depends on type
        if ($isHistory) {
            $reports->getCollection()->transform(function ($item) {
                $opts = is_string($item->checkpoint->extra_scan_options)
                    ? json_decode($item->checkpoint->extra_scan_options, true)
                    : $item->checkpoint->extra_scan_options;
                $type = $opts['type'] ?? 'log_only';

                return [
                    'id' => $item->id,
                    'scanned_at' => $item->scanned_at->format('Y-m-d H:i:s'),
                    'scanned_at_human' => $item->scanned_at->format('H:i:s, M d, Y'),
                    'site_name' => $item->schedule->site->name ?? 'N/A',
                    'duty_number' => $item->schedule->duty_number ?? 'N/A',
                    'checkpoint_name' => $item->checkpoint->name ?? 'Unknown Checkpoint',
                    'is_manual' => $item->is_manual,
                    'scan_method' => $item->is_manual ? 'MANUAL SCAN' : 'VERIFIED QR/NFC',
                    'protocol_type' => $type,
                    'employee_message' => $item->employee_message,
                    'evidence_image' => $item->evidence_image ? asset('storage/' . $item->evidence_image) : null,
                    'additional_photos' => collect($item->additional_photos ?? [])->map(fn($p) => asset('storage/' . $p)),
                ];
            });
        } else {
            $reports->getCollection()->transform(function ($job) {
                return [
                    'id' => $job->id,
                    'duty_number' => $job->duty_number,
                    'site_name' => $job->site->name ?? 'Unknown Site',
                    'scheduled_start' => $job->from_datetime->format('Y-m-d H:i:s'),
                    'scheduled_end' => $job->to_datetime->format('Y-m-d H:i:s'),
                    'actual_start' => $job->pivot->actual_start_at,
                    'actual_end' => $job->pivot->actual_end_at,
                    'total_scans' => $job->checkpointScans->count(),
                    'status' => $job->status,
                    'status_label' => ucfirst($job->status),
                ];
            });
        }

        return response()->json([
            'status' => 'success',
            'report_type' => $reportType,
            'export_urls' => [
                'pdf' => route('api.employee.reports.export.pdf', $request->all()), // Updated to API route name (will define below)
                'excel' => route('api.employee.reports.export.excel', $request->all())
            ],
            'data' => $reports
        ]);
    }

    /**
     * Helper to build report query
     */
    private function getOperationalReportQuery(Request $request, $employee)
    {
        $reportType = $request->query('type', 'checkin');
        $isHistory = $reportType === 'scans_history' || ($reportType === 'scans' && $request->filled('schedule_id'));

        if ($isHistory) {
            $query = $employee->checkpointScans()
                ->with(['checkpoint', 'schedule.site', 'tourRoute'])
                ->orderBy('scanned_at', 'desc');

            if ($request->filled('start_date'))
                $query->whereDate('scanned_at', '>=', $request->start_date);
            if ($request->filled('end_date'))
                $query->whereDate('scanned_at', '<=', $request->end_date);
            if ($request->filled('site_id')) {
                $query->whereHas('schedule', function ($q) use ($request) {
                    $q->where('site_id', $request->site_id);
                });
            }
            if ($request->filled('search')) {
                $search = $request->search;
                $query->whereHas('schedule', function ($q) use ($search) {
                    $q->where('duty_number', 'like', "%{$search}%");
                });
            }
            if ($request->filled('schedule_id')) {
                $query->where('schedule_id', $request->schedule_id);
            }
            return $query;
        } else {
            $query = $employee->schedules()
                ->with([
                    'site',
                    'checkpointScans' => function ($q) use ($employee) {
                        $q->where('employee_id', $employee->id);
                    }
                ])
                ->whereNotNull('employee_schedule.actual_start_at')
                ->orderBy('schedule_date', 'desc');

            if ($reportType === 'checkout')
                $query->whereNotNull('employee_schedule.actual_end_at');
            if ($reportType === 'scans')
                $query->whereHas('checkpointScans', function ($q) use ($employee) {
                    $q->where('employee_id', $employee->id);
                });

            if ($request->filled('start_date'))
                $query->whereDate('schedule_date', '>=', $request->start_date);
            if ($request->filled('end_date'))
                $query->whereDate('schedule_date', '<=', $request->end_date);
            if ($request->filled('site_id'))
                $query->where('site_id', $request->site_id);
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where('duty_number', 'like', "%{$search}%");
            }
            if ($request->filled('schedule_id'))
                $query->where('schedules.id', $request->schedule_id);

            return $query;
        }
    }

    /**
     * Export Report as PDF (API)
     */
    public function operationalReportExportPdf(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $reportType = $request->query('type', 'checkin');
        $isHistory = $reportType === 'scans_history' || ($reportType === 'scans' && $request->filled('schedule_id'));

        $query = $this->getOperationalReportQuery($request, $employee);
        $reports = $query->get();

        // Use the existing views
        $view = $isHistory ? 'employee.reports.scans_pdf' : 'employee.reports.pdf';

        $pdf = Pdf::loadView($view, compact('reports', 'reportType', 'employee'));
        return $pdf->download("Operational_Report_{$reportType}_" . now()->format('Ymd_Hi') . '.pdf');
    }

    /**
     * Export Report as Excel/CSV (API)
     */
    public function operationalReportExportExcel(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        $reportType = $request->query('type', 'checkin');
        $isHistory = $reportType === 'scans_history' || ($reportType === 'scans' && $request->filled('schedule_id'));

        $query = $this->getOperationalReportQuery($request, $employee);
        $reports = $query->get();

        $filename = "Operational_Report_{$reportType}_" . now()->format('Ymd_Hi') . ".csv";

        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];

        if ($isHistory) {
            $columns = ['Date/Time', 'Site Name', 'Duty #', 'Checkpoint', 'Type', 'Message', 'Manual Scan', 'Photos Count'];
        } else {
            $columns = $reportType === 'scans'
                ? ['Duty Number', 'Site Name', 'Scheduled Start', 'Scheduled End', 'Total Scans', 'Actual Start', 'Actual End', 'Status']
                : ['Duty Number', 'Site Name', 'Scheduled Start', 'Scheduled End', 'Actual Start', 'Actual End', 'Status'];
        }

        $callback = function () use ($reports, $columns, $isHistory, $reportType) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($reports as $item) {
                if ($isHistory) {
                    fputcsv($file, [
                        $item->scanned_at->format('Y-m-d H:i'),
                        $item->schedule->site->name ?? 'N/A',
                        $item->schedule->duty_number ?? 'N/A',
                        $item->checkpoint->name ?? 'N/A',
                        ucfirst(str_replace('_', ' ', $item->checkpoint->extra_scan_options['type'] ?? 'Log Only')),
                        $item->employee_message ?? 'N/A',
                        $item->is_manual ? 'Yes' : 'No',
                        count($item->additional_photos ?? []) + ($item->evidence_image ? 1 : 0)
                    ]);
                } elseif ($reportType === 'scans') {
                    fputcsv($file, [
                        $item->duty_number,
                        $item->site->name ?? 'Unknown',
                        $item->from_datetime->format('Y-m-d H:i'),
                        $item->to_datetime->format('Y-m-d H:i'),
                        $item->checkpointScans->count(),
                        $item->pivot->actual_start_at ? Carbon::parse($item->pivot->actual_start_at)->format('Y-m-d H:i') : 'N/A',
                        $item->pivot->actual_end_at ? Carbon::parse($item->pivot->actual_end_at)->format('Y-m-d H:i') : 'N/A',
                        ucfirst($item->status)
                    ]);
                } else {
                    fputcsv($file, [
                        $item->duty_number,
                        $item->site->name ?? 'Unknown',
                        $item->from_datetime->format('Y-m-d H:i'),
                        $item->to_datetime->format('Y-m-d H:i'),
                        $item->pivot->actual_start_at ? Carbon::parse($item->pivot->actual_start_at)->format('Y-m-d H:i') : 'N/A',
                        $item->pivot->actual_end_at ? Carbon::parse($item->pivot->actual_end_at)->format('Y-m-d H:i') : 'N/A',
                        ucfirst($item->status)
                    ]);
                }
            }
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * GET /employee/payouts
     */
    public function payouts(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        // --- Guard Schedules Base Query ---
        $guardQuery = $employee->schedules()
            ->with('site')
            ->whereNotNull('employee_schedule.actual_end_at')
            ->whereNotNull('employee_schedule.actual_start_at');

        // --- Patroller Schedules Base Query ---
        $patrollerQuery = $employee->patrollerSchedules()
            ->with('route')
            ->whereNotNull('employee_patroller_schedule.actual_end_at')
            ->whereNotNull('employee_patroller_schedule.actual_start_at');

        // Apply Filters
        if ($request->filled('start_date')) {
            $guardQuery->whereDate('schedule_date', '>=', $request->start_date);
            $patrollerQuery->whereDate('scheduled_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $guardQuery->whereDate('schedule_date', '<=', $request->end_date);
            $patrollerQuery->whereDate('scheduled_date', '<=', $request->end_date);
        }
        if ($request->filled('search')) {
            $search = $request->search;
            $guardQuery->where('duty_number', 'like', "%{$search}%");
            $patrollerQuery->where('duty_number', 'like', "%{$search}%");
        }

        // Get All matching schedules
        $guards = $guardQuery->orderBy('employee_schedule.actual_end_at', 'desc')->get();
        $patrollers = $patrollerQuery->orderBy('employee_patroller_schedule.actual_end_at', 'desc')->get();

        // Standardize both into one collection
        $unified = collect([]);

        foreach ($guards as $item) {
            $start = $item->pivot->actual_start_at ? Carbon::parse($item->pivot->actual_start_at) : null;
            $end = $item->pivot->actual_end_at ? Carbon::parse($item->pivot->actual_end_at) : null;
            $earnings = $this->calculateGuardEarnings($item);

            // Replicate hours display logic for guard (logic from original payouts method)
            $actualHours = ($start && $end) ? abs($end->diffInMinutes($start)) / 60 : 0;
            $dutyHours = abs(Carbon::parse($item->from_datetime)->diffInMinutes(Carbon::parse($item->to_datetime))) / 60;
            $rate = $item->pivot->wage_rate ?? 0;
            $displayedHours = ($rate > 0) ? $dutyHours : $actualHours;
            $avgRate = ($displayedHours > 0) ? $earnings / $displayedHours : $rate;
            $isAvgRate = ($rate <= 0);

            $unified->push([
                'id' => $item->id,
                'duty_number' => $item->duty_number,
                'site_name' => $item->site->name ?? 'Unknown Site',
                'date_formatted' => $item->schedule_date->format('M d, Y'),
                'actual_end_at' => $item->pivot->actual_end_at,
                'start_time' => $start ? $start->format('H:i') : '--',
                'end_time' => $end ? $end->format('H:i') : '--',
                'duration_label' => number_format($displayedHours, 2) . ' hrs',
                'rate_label' => '$' . number_format($avgRate, 2) . '/hr' . ($isAvgRate ? ' (Avg)' : ''),
                'amount' => number_format($earnings, 2),
                'currency_symbol' => '$',
                'type' => 'guard'
            ]);
        }

        foreach ($patrollers as $item) {
            $start = $item->pivot->actual_start_at ? Carbon::parse($item->pivot->actual_start_at) : null;
            $end = $item->pivot->actual_end_at ? Carbon::parse($item->pivot->actual_end_at) : null;
            $earnings = $this->calculatePatrollerEarnings($item);
            $displayedHours = ($start && $end) ? abs($end->diffInMinutes($start)) / 60 : 0;
            $rate = ($displayedHours > 0) ? $earnings / $displayedHours : ($item->pivot->wage_rate ?? 0);

            $unified->push([
                'id' => $item->id,
                'duty_number' => $item->duty_number,
                'site_name' => $item->route->name ?? 'Unknown Route',
                'date_formatted' => $item->scheduled_date->format('M d, Y'),
                'actual_end_at' => $item->pivot->actual_end_at,
                'start_time' => $start ? $start->format('H:i') : '--',
                'end_time' => $end ? $end->format('H:i') : '--',
                'duration_label' => number_format($displayedHours, 2) . ' hrs',
                'rate_label' => '$' . number_format($rate, 2) . '/hr',
                'amount' => number_format($earnings, 2),
                'currency_symbol' => '$',
                'type' => 'patroller'
            ]);
        }

        // Sort unified collection by actual_end_at desc
        $unified = $unified->sortByDesc('actual_end_at')->values();
        $totalEarnings = $unified->sum(function ($item) {
            return (float) str_replace(',', '', $item['amount']);
        });

        // Paginate manually
        $page = $request->input('page', 1);
        $perPage = 15;
        $paginated = new \Illuminate\Pagination\LengthAwarePaginator(
            $unified->forPage($page, $perPage),
            $unified->count(),
            $perPage,
            $page,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return response()->json([
            'status' => 'success',
            'total_earnings' => number_format($totalEarnings, 2, '.', ''),
            'export_url' => route('api.employee.payouts.export.pdf', $request->all()),
            'data' => $paginated
        ]);
    }

    /**
     * Export Payouts as PDF
     */
    public function payoutsExportPdf(Request $request)
    {
        $employee = $request->user();
        if (!$employee instanceof Employee) {
            $employee = Employee::where('email', $employee->email)->first();
        }

        // --- Guard Schedules ---
        $guardQuery = $employee->schedules()
            ->with(['site'])
            ->whereNotNull('employee_schedule.actual_end_at')
            ->whereNotNull('employee_schedule.actual_start_at');

        // --- Patroller Schedules ---
        $patrollerQuery = $employee->patrollerSchedules()
            ->with(['route'])
            ->whereNotNull('employee_patroller_schedule.actual_end_at')
            ->whereNotNull('employee_patroller_schedule.actual_start_at');

        if ($request->filled('start_date')) {
            $guardQuery->whereDate('schedule_date', '>=', $request->start_date);
            $patrollerQuery->whereDate('scheduled_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $guardQuery->whereDate('schedule_date', '<=', $request->end_date);
            $patrollerQuery->whereDate('scheduled_date', '<=', $request->end_date);
        }
        if ($request->filled('search')) {
            $search = $request->search;
            $guardQuery->where('duty_number', 'like', "%{$search}%");
            $patrollerQuery->where('duty_number', 'like', "%{$search}%");
        }

        $guards = $guardQuery->orderBy('employee_schedule.actual_end_at', 'desc')->get();
        $patrollers = $patrollerQuery->orderBy('employee_patroller_schedule.actual_end_at', 'desc')->get();

        // Standardize both into one collection for PDF
        $payouts = collect([]);

        foreach ($guards as $item) {
            $earnings = $this->calculateGuardEarnings($item);
            $payouts->push((object) [
                'duty_number' => $item->duty_number,
                'site_name' => $item->site->name ?? 'N/A',
                'date' => $item->schedule_date,
                'actual_start_at' => $item->pivot->actual_start_at,
                'actual_end_at' => $item->pivot->actual_end_at,
                'amount' => $earnings,
                'type' => 'Guard'
            ]);
        }

        foreach ($patrollers as $item) {
            $earnings = $this->calculatePatrollerEarnings($item);
            $payouts->push((object) [
                'duty_number' => $item->duty_number,
                'site_name' => $item->route->name ?? 'N/A',
                'date' => $item->scheduled_date,
                'actual_start_at' => $item->pivot->actual_start_at,
                'actual_end_at' => $item->pivot->actual_end_at,
                'amount' => $earnings,
                'type' => 'Patroller'
            ]);
        }

        $payouts = $payouts->sortByDesc('actual_end_at');
        $totalEarnings = $payouts->sum('amount');

        $pdf = Pdf::loadView('employee.reports.payouts_pdf', compact('payouts', 'totalEarnings', 'employee'));
        return $pdf->download('Payouts_Report_' . now()->format('Ymd_Hi') . '.pdf');
    }
}



