<?php

namespace App\Http\Controllers;

use App\Models\Schedule;
use App\Models\PatrollerSchedule;
use App\Models\Incident;
use App\Models\PatrollerIssueTicket;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class AnalyticsController extends Controller
{
    public function index()
    {
        $companyId = session('selected_company_id');

        // Get last 12 months
        $months = [];
        $monthLabels = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = Carbon::now()->subMonths($i);
            $months[] = $date->format('Y-m');
            $monthLabels[] = $date->format('M Y');
        }

        // Guard Jobs Data (Monthly)
        $guardJobsCompleted = [];
        $guardJobsUpcoming = [];
        $guardJobsMissed = [];

        foreach ($months as $month) {
            $startDate = Carbon::parse($month . '-01')->startOfMonth();
            $endDate = Carbon::parse($month . '-01')->endOfMonth();

            $guardJobsCompleted[] = Schedule::where('company_id', $companyId)
                ->where('status', 'completed')
                ->whereBetween('schedule_date', [$startDate, $endDate])
                ->count();

            $guardJobsUpcoming[] = Schedule::where('company_id', $companyId)
                ->whereIn('status', ['upcoming', 'today', 'scheduled', 'published', 'draft'])
                ->whereBetween('schedule_date', [$startDate, $endDate])
                ->count();

            $guardJobsMissed[] = Schedule::where('company_id', $companyId)
                ->where('status', 'missed')
                ->whereBetween('schedule_date', [$startDate, $endDate])
                ->count();
        }

        // Patroller Jobs Data (Monthly)
        $patrollerJobsCompleted = [];
        $patrollerJobsUpcoming = [];
        $patrollerJobsMissed = [];

        foreach ($months as $month) {
            $startDate = Carbon::parse($month . '-01')->startOfMonth();
            $endDate = Carbon::parse($month . '-01')->endOfMonth();

            $patrollerJobsCompleted[] = PatrollerSchedule::where('company_id', $companyId)
                ->whereIn('job_status', ['completed', 'auto_ended'])
                ->whereBetween('scheduled_date', [$startDate, $endDate])
                ->count();

            $patrollerJobsUpcoming[] = PatrollerSchedule::where('company_id', $companyId)
                ->where('job_status', 'pending')
                ->whereBetween('scheduled_date', [$startDate, $endDate])
                ->count();

            $patrollerJobsMissed[] = PatrollerSchedule::where('company_id', $companyId)
                ->where('job_status', 'missed')
                ->whereBetween('scheduled_date', [$startDate, $endDate])
                ->count();
        }

        // Incident Reports (Monthly)
        $incidentReports = [];
        foreach ($months as $month) {
            $startDate = Carbon::parse($month . '-01')->startOfMonth();
            $endDate = Carbon::parse($month . '-01')->endOfMonth();

            $incidentReports[] = Incident::whereHas('schedule', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();
        }

        // Issue Tickets (Monthly)
        $issueTickets = [];
        foreach ($months as $month) {
            $startDate = Carbon::parse($month . '-01')->startOfMonth();
            $endDate = Carbon::parse($month . '-01')->endOfMonth();

            $issueTickets[] = PatrollerIssueTicket::whereHas('patrollerSchedule', function ($q) use ($companyId) {
                $q->where('company_id', $companyId);
            })
                ->whereBetween('created_at', [$startDate, $endDate])
                ->count();
        }

        // Guard Check-ins Analysis
        $guardOnTimeCheckins = Schedule::where('schedules.company_id', $companyId)
            ->join('employee_schedule', 'schedules.id', '=', 'employee_schedule.schedule_id')
            ->whereNotNull('employee_schedule.actual_start_at')
            ->whereRaw('employee_schedule.actual_start_at <= schedules.from_datetime')
            ->count();

        $guardLateCheckins = Schedule::where('schedules.company_id', $companyId)
            ->join('employee_schedule', 'schedules.id', '=', 'employee_schedule.schedule_id')
            ->whereNotNull('employee_schedule.actual_start_at')
            ->whereRaw('employee_schedule.actual_start_at > schedules.from_datetime')
            ->count();

        // Patroller Check-ins Analysis
        $patrollerOnTimeCheckins = PatrollerSchedule::where('patroller_schedules.company_id', $companyId)
            ->join('employee_patroller_schedule', 'patroller_schedules.id', '=', 'employee_patroller_schedule.patroller_schedule_id')
            ->whereNotNull('employee_patroller_schedule.actual_start_at')
            ->whereRaw('employee_patroller_schedule.actual_start_at <= patroller_schedules.from_time')
            ->count();

        $patrollerLateCheckins = PatrollerSchedule::where('patroller_schedules.company_id', $companyId)
            ->join('employee_patroller_schedule', 'patroller_schedules.id', '=', 'employee_patroller_schedule.patroller_schedule_id')
            ->whereNotNull('employee_patroller_schedule.actual_start_at')
            ->whereRaw('employee_patroller_schedule.actual_start_at > patroller_schedules.from_time')
            ->count();

        return view('analytics.index', compact(
            'monthLabels',
            'guardJobsCompleted',
            'guardJobsUpcoming',
            'guardJobsMissed',
            'patrollerJobsCompleted',
            'patrollerJobsUpcoming',
            'patrollerJobsMissed',
            'incidentReports',
            'issueTickets',
            'guardOnTimeCheckins',
            'guardLateCheckins',
            'patrollerOnTimeCheckins',
            'patrollerLateCheckins'
        ));
    }
}
