<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AdminAuthController extends Controller
{
    public function showLogin()
    {
        if (Auth::check() && session('admin_login_verified')) {
            return redirect()->route('admin.dashboard');
        }
        return view('admin.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => ['required', 'email'],
            'password' => ['required'],
        ]);

        // Google reCAPTCHA Validation
        if (\App\Models\Setting::where('key', 'recaptcha_enabled')->value('value') === '1' && app()->environment('production')) {
            $request->validate([
                'g-recaptcha-response' => 'required',
            ], [
                'g-recaptcha-response.required' => 'Please complete the captcha verification.',
            ]);

            $response = \Illuminate\Support\Facades\Http::asForm()->post('https://www.google.com/recaptcha/api/siteverify', [
                'secret' => \App\Models\Setting::where('key', 'recaptcha_secret_key')->value('value'),
                'response' => $request->input('g-recaptcha-response'),
                'remoteip' => $request->ip(),
            ]);

            if (!$response->successful() || !$response->json('success')) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'g-recaptcha-response' => 'reCAPTCHA verification failed. Please try again.',
                ]);
            }
        }

        // Rate Limiting Logic
        $throttleKey = \Illuminate\Support\Str::transliterate(\Illuminate\Support\Str::lower($request->input('email')) . '|' . $request->ip());

        $maxAttempts = (int) \App\Models\Setting::where('key', 'login_max_attempts')->value('value') ?: 5;
        $decayMinutes = (int) \App\Models\Setting::where('key', 'login_decay_minutes')->value('value') ?: 1;

        if (\Illuminate\Support\Facades\RateLimiter::tooManyAttempts($throttleKey, $maxAttempts)) {
            $seconds = \Illuminate\Support\Facades\RateLimiter::availableIn($throttleKey);
            $minutes = ceil($seconds / 60);

            throw \Illuminate\Validation\ValidationException::withMessages([
                'email' => trans('auth.throttle', [
                    'seconds' => $seconds,
                    'minutes' => $minutes,
                ]),
            ]);
        }

        if (Auth::attempt($credentials)) {
            \Illuminate\Support\Facades\RateLimiter::clear($throttleKey);

            $user = Auth::user();
            // Check if user has Admin or Manager role (either via direct column or Spatie role)
            // Assuming 'role' column is used for primary check, or we should check permissions.
            // Safe bet: Check if role is 'admin' or 'manager' OR if they have the 'Admin' or 'Manager' role via Spatie.
            // But strict 'role' column check is what's blocking currently.

            $allowedRoles = ['admin', 'Admin'];

            if (!in_array($user->role, $allowedRoles) && !$user->hasAnyRole($allowedRoles)) {
                Auth::logout();
                return back()->withErrors([
                    'email' => 'You do not have administrative access.',
                ])->onlyInput('email');
            }

            // Set Admin Verified Flag
            session(['admin_login_verified' => true]);

            $request->session()->regenerate();
            return redirect()->intended(route('admin.dashboard'));
        }

        \Illuminate\Support\Facades\RateLimiter::hit($throttleKey, $decayMinutes * 60);

        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ])->onlyInput('email');
    }

    public function logout(Request $request)
    {
        // If we want to logout completely (both user and admin), use Auth::logout()
        // But maybe we only want to drop admin verification?
        // Usually logout means logging out of the system.
        Auth::logout();
        $request->session()->forget('admin_login_verified'); // Explicitly forget
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('admin.login');
    }
}
