<?php

namespace App\Http\Controllers;

use App\Models\Vehicle;
use App\Models\Country;
use App\Models\State;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class VehicleController extends Controller
{
    public function index(Request $request)
    {
        $company_id = session('selected_company_id');
        if (!$company_id) {
            return redirect()->route('select.company');
        }

        $query = Vehicle::where('company_id', $company_id);

        // Search by VIN
        if ($request->filled('search')) {
            $query->where('vin_number', 'LIKE', '%' . $request->search . '%');
        }

        // Filter by ownership
        if ($request->filled('ownership')) {
            $query->where('ownership', $request->ownership);
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $vehicles = $query->orderBy('created_at', 'desc')->paginate(100);

        return view('vehicles.index', compact('vehicles'));
    }

    public function create()
    {
        $company_id = session('selected_company_id');
        if (!$company_id) {
            return redirect()->route('select.company');
        }

        // Generate unique vehicle ID
        $generatedId = 'VEH-' . strtoupper(Str::random(8));
        while (Vehicle::where('vehicle_id', $generatedId)->exists()) {
            $generatedId = 'VEH-' . strtoupper(Str::random(8));
        }

        $countries = Country::all();

        return view('vehicles.create', compact('generatedId', 'countries'));
    }

    public function store(Request $request)
    {
        $company_id = session('selected_company_id');
        if (!$company_id) {
            return redirect()->route('select.company');
        }

        $request->validate([
            'ownership' => 'required|string',
            'status' => 'required|string',
            'vehicle_id' => 'required|string|unique:vehicles,vehicle_id',
            'make' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'year' => 'nullable|digits:4',
            'acquisition_date' => 'nullable|date',
            'license_plate' => 'nullable|string|max:255',
            'license_plate_country' => 'nullable|string',
            'license_plate_state' => 'nullable|string',
            'vin_number' => 'nullable|string|max:255|unique:vehicles,vin_number',
            'color' => 'nullable|string|max:255',
            'comments' => 'nullable|string',
            'images.*' => 'nullable|image|max:10240',
        ]);

        $data = $request->except('images');
        $data['company_id'] = $company_id;

        // Handle multiple image uploads
        $images = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('vehicles', 'public');
                $images[] = $path;
            }
        }
        $data['images'] = $images;

        Vehicle::create($data);

        return redirect()->route('vehicles.index')->with('success', 'Vehicle created successfully!');
    }

    public function show(Vehicle $vehicle)
    {
        $company_id = session('selected_company_id');
        if ($vehicle->company_id !== $company_id) {
            abort(403);
        }

        return view('vehicles.show', compact('vehicle'));
    }

    public function edit(Vehicle $vehicle)
    {
        $company_id = session('selected_company_id');
        if ($vehicle->company_id !== $company_id) {
            abort(403);
        }

        $countries = Country::all();
        $states = State::all();

        return view('vehicles.edit', compact('vehicle', 'countries', 'states'));
    }

    public function update(Request $request, Vehicle $vehicle)
    {
        $company_id = session('selected_company_id');
        if ($vehicle->company_id !== $company_id) {
            abort(403);
        }

        $request->validate([
            'ownership' => 'required|string',
            'status' => 'required|string',
            'vehicle_id' => 'required|string|unique:vehicles,vehicle_id,' . $vehicle->id,
            'make' => 'nullable|string|max:255',
            'model' => 'nullable|string|max:255',
            'year' => 'nullable|digits:4',
            'acquisition_date' => 'nullable|date',
            'license_plate' => 'nullable|string|max:255',
            'license_plate_country' => 'nullable|string',
            'license_plate_state' => 'nullable|string',
            'vin_number' => 'nullable|string|max:255|unique:vehicles,vin_number,' . $vehicle->id,
            'color' => 'nullable|string|max:255',
            'comments' => 'nullable|string',
            'images.*' => 'nullable|image|max:10240',
        ]);

        $data = $request->except('images');

        // Handle multiple image uploads
        $existingImages = $vehicle->images ?? [];

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('vehicles', 'public');
                $existingImages[] = $path;
            }
        }

        $data['images'] = $existingImages;

        $vehicle->update($data);

        return redirect()->route('vehicles.index')->with('success', 'Vehicle updated successfully!');
    }

    public function destroy(Vehicle $vehicle)
    {
        $company_id = session('selected_company_id');
        if ($vehicle->company_id !== $company_id) {
            abort(403);
        }

        // Delete images from storage
        if ($vehicle->images) {
            foreach ($vehicle->images as $image) {
                Storage::disk('public')->delete($image);
            }
        }

        $vehicle->delete();

        return redirect()->route('vehicles.index')->with('success', 'Vehicle deleted successfully!');
    }

    public function deleteImage(Vehicle $vehicle, $index)
    {
        $company_id = session('selected_company_id');
        if ($vehicle->company_id !== $company_id) {
            abort(403);
        }

        $images = $vehicle->images ?? [];

        if (isset($images[$index])) {
            Storage::disk('public')->delete($images[$index]);
            unset($images[$index]);
            $vehicle->update(['images' => array_values($images)]);
        }

        return back()->with('success', 'Image deleted successfully!');
    }
}
