<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\GuardCheckin;
use App\Models\Schedule;
use App\Models\PatrollerSchedule;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LocationResponseController extends Controller
{
    /**
     * Show the location request form
     */
    public function show(Request $request)
    {
        $employee = Auth::guard('employee')->user();

        $employeeId = $request->query('employee_id');
        $type = $request->query('type');
        $jobId = $request->query('job_id');

        // Verify this employee is the one being requested
        if ($employee->id != $employeeId) {
            abort(403, 'This request is not for you');
        }

        // Get job details
        if ($type === 'static') {
            $job = Schedule::with('site')->findOrFail($jobId);
            $jobName = $job->site->name;
            $jobRef = $job->duty_number;
        } else {
            $job = PatrollerSchedule::with('route')->findOrFail($jobId);
            $jobName = $job->route->name;
            $jobRef = $job->duty_number;
        }

        return view('employee.location-response', compact('employee', 'type', 'jobId', 'jobName', 'jobRef'));
    }

    /**
     * Store the location response
     */
    public function store(Request $request)
    {
        $employee = Auth::guard('employee')->user();

        $request->validate([
            'type' => 'required|in:static,patrol',
            'job_id' => 'required|integer',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'maps_link' => 'nullable|url',
            'location_reported' => 'nullable|string',
            'images.*' => 'nullable|image|max:10240',
            'remarks' => 'nullable|string',
        ]);

        // Validate at least one location method or images provided
        if (
            !$request->filled('latitude') &&
            !$request->filled('maps_link') &&
            !$request->filled('location_reported') &&
            !$request->hasFile('images')
        ) {
            return back()->withErrors(['error' => 'Please provide either location information or upload images.']);
        }

        $companyId = $employee->company_id;

        // Handle image uploads
        $imagePaths = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('location-responses/' . date('Y-m-d'), 'public');
                $imagePaths[] = $path;
            }
        }

        $data = [
            'company_id' => $companyId,
            'user_id' => null, // Employee submitted, not user
            'employee_id' => $employee->id,
            'checkin_type' => 'self_reported', // New type for employee self-reports
            'location_reported' => $request->location_reported,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'maps_link' => $request->maps_link,
            'is_verified' => false, // Not verified by customer care yet
            'notes' => $request->remarks,
            'images' => $imagePaths,
        ];

        if ($request->type === 'static') {
            $data['schedule_id'] = $request->job_id;
        } else {
            $data['patroller_schedule_id'] = $request->job_id;
        }

        GuardCheckin::create($data);

        return redirect()->route('employee.dashboard')->with('success', 'Location and evidence submitted successfully!');
    }
}
