<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use App\Services\ImageService;
use App\Models\CheckpointScan;
use App\Models\Schedule;

class EmployeeJobController extends Controller
{
    public function index(Request $request)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            Auth::guard('employee')->logout();
            return redirect()->route('employee.login')->withErrors(['email' => 'Session invalid.']);
        }
        $type = $request->query('type', 'all'); // all, active, today, upcoming, completed, missed

        $now = Carbon::now();
        $today = Carbon::today();

        // Base query with relationships
        $query = $employee->schedules()->with([
            'site',
            'employees',
            'incidents' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            },
            'tourRoutes.checkpoints',
            'checkpointScans'
        ]);

        switch ($type) {
            case 'active':
                $query->where('schedules.status', 'active');
                break;

            case 'today':
                $query->whereDate('schedules.from_datetime', $today)
                    ->where('schedules.to_datetime', '>', $now)
                    ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled']);
                break;

            case 'upcoming':
                $query->whereDate('schedules.from_datetime', '>', $today)
                    ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled']);
                break;

            case 'completed':
                $query->where('schedules.status', 'completed');
                break;

            case 'missed':
                $query->where(function ($q) use ($now) {
                    $q->where('schedules.status', 'missed')
                        ->orWhere(function ($sq) use ($now) {
                            $sq->where('schedules.to_datetime', '<=', $now)
                                ->whereNotIn('schedules.status', ['active', 'completed', 'cancelled']);
                        });
                });
                break;

            case 'pending':
                $query->where('employee_schedule.cancellation_status', 'pending');
                break;

            case 'cancelled':
                $query->where('schedules.status', 'cancelled');
                break;
        }

        $jobs = $query->orderBy('schedules.from_datetime')->paginate(100);

        // Counts for tabs - Re-instantiating relation to avoid any cloning side-effects
        // Updated counts to match the new strict status filtering
        $counts = [
            'all' => $employee->schedules()->count(),
            'active' => $employee->schedules()->where('schedules.status', 'active')->count(),
            'today' => $employee->schedules()
                ->whereDate('schedules.from_datetime', $today)
                ->where('schedules.to_datetime', '>', $now)
                ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled'])
                ->count(),
            'upcoming' => $employee->schedules()
                ->whereDate('schedules.from_datetime', '>', $today)
                ->whereNotIn('schedules.status', ['active', 'completed', 'missed', 'cancelled'])
                ->count(),
            'completed' => $employee->schedules()->where('schedules.status', 'completed')->count(),
            'missed' => $employee->schedules()->where(function ($q) use ($now) {
                $q->where('schedules.status', 'missed')
                    ->orWhere(function ($sq) use ($now) {
                        $sq->where('schedules.to_datetime', '<=', $now)
                            ->whereNotIn('schedules.status', ['active', 'completed', 'cancelled']);
                    });
            })->count(),
            'cancelled' => $employee->schedules()->where('schedules.status', 'cancelled')->count(),
            'pending' => $employee->schedules()->where('employee_schedule.cancellation_status', 'pending')->count()
        ];

        $categories = \App\Models\IncidentParentCategory::with([
            'incident_categories' => function ($q) {
                $q->where('status', true);
            }
        ])->where('status', true)->get();

        return view('employee.jobs.index', compact('jobs', 'type', 'counts', 'employee', 'categories'));
    }

    public function startJob(Request $request, $id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->findOrFail($id);

        // Visibility Check
        $visibilityMinutes = $employee->getVisibilityMinutes();
        $visibilityDeadline = Carbon::now()->addMinutes($visibilityMinutes);

        if ($schedule->from_datetime > $visibilityDeadline) {
            return back()->with('error', 'Job is not yet cleared for initiation. (Scheduled: ' . $schedule->from_datetime . ', Deadline: ' . $visibilityDeadline . ', Window: ' . $visibilityMinutes . ' mins)');
        }

        // Check for Concurrent Active Jobs (Single Active Job Constraint)
        $concurrentJob = $employee->schedules()
            ->wherePivotNotNull('actual_start_at')
            ->wherePivotNull('actual_end_at')
            ->first();

        if ($concurrentJob) {
            return back()->with('error', "You are already active on Duty #{$concurrentJob->duty_number} ({$concurrentJob->from_datetime->format('H:i')} - {$concurrentJob->to_datetime->format('H:i')}). Please end that duty before starting a new one.");
        }

        if ($schedule->to_datetime <= Carbon::now()) {
            return back()->with('error', 'Job has expired and cannot be initiated. (Scheduled End: ' . $schedule->to_datetime . ')');
        }

        if ($schedule->status === 'active' && $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->whereNotNull('actual_start_at')->exists()) {
            return back()->with('error', 'Job is already active for your deployment.');
        }

        $now = Carbon::now();
        $actualStartAt = $now->gt($schedule->from_datetime) ? $now : $schedule->from_datetime;

        $schedule->employees()->updateExistingPivot($employee->id, [
            'actual_start_at' => $actualStartAt,
            'start_lat' => $request->latitude,
            'start_lng' => $request->longitude,
        ]);

        $schedule->update([
            'status' => 'active'
        ]);

        return back()->with('success', 'Job initiated. Deployment is now ACTIVE. Please submit check-in evidence.');
    }

    public function requestCancellation(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->findOrFail($id);

        // Check if schedule is truly upcoming (StartDate > Today)
        // Correct fix: check if the START TIME is in the future, regardless of date
        if ($schedule->from_datetime->isPast()) {
            return back()->with('error', 'Cancellation requests are only allowed for future schedules that have not started.');
        }

        $schedule->employees()->updateExistingPivot($employee->id, [
            'cancellation_requested_at' => Carbon::now(),
            'cancellation_reason' => $request->reason,
            'cancellation_status' => 'pending'
        ]);

        // Notify Company Users
        foreach ($employee->company->users as $user) {
            \App\Models\Notification::create([
                'company_id' => $employee->company_id,
                'user_id' => $user->id,
                'title' => 'New Cancellation Request',
                'message' => "Employee {$employee->first_name} {$employee->last_name} has requested to cancel Duty #{$schedule->duty_number}.",
                'url' => route('schedules.index', ['type' => 'all', 'search' => $schedule->duty_number]),
            ]);
        }

        return back()->with('success', 'Cancellation request submitted for review.');
    }

    public function checkin(Request $request, $id)
    {
        $request->validate([
            'images' => 'required|array|min:1',
            'images.*' => 'image|max:10240',
        ]);

        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->findOrFail($id);

        $images = [];
        if ($request->hasFile('images')) {
            $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;
            $timestamp = $pivot->actual_start_at;
            $folderPath = ImageService::getFolderPath('check-in', $schedule->duty_number, $schedule->site->name, $timestamp);
            foreach ($request->file('images') as $file) {
                $path = ImageService::processAndStore($file, $folderPath);
                if ($path) {
                    $images[] = $path;
                }
            }
        }

        // Correctly update the pivot table using updateExistingPivot to avoid updating the employees table
        $schedule->employees()->updateExistingPivot($employee->id, [
            'checkin_images' => json_encode($images),
        ]);

        return back()->with('success', 'Check-in evidence successfully archived. Job profile updated.');
    }

    public function reportIncident(Request $request, $id)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'description' => 'required|string',
            'images.*' => 'nullable|image|max:5120',
            'incident_parent_category_id' => 'required|exists:incident_parent_categories,id',
            'incident_category_id' => 'required|exists:incident_categories,id',
        ]);

        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->findOrFail($id);

        $images = [];
        if ($request->hasFile('images')) {
            $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;
            $timestamp = $pivot->actual_start_at;
            $folderPath = ImageService::getFolderPath('report-incident', $schedule->duty_number, $schedule->site->name, $timestamp);
            foreach ($request->file('images') as $file) {
                $path = ImageService::processAndStore($file, $folderPath);
                if ($path) {
                    $images[] = $path;
                }
            }
        }

        \App\Models\Incident::create([
            'employee_id' => $employee->id,
            'schedule_id' => $schedule->id,
            'subject' => $request->subject,
            'description' => $request->description,
            'images' => $images,
            'incident_parent_category_id' => $request->incident_parent_category_id,
            'incident_category_id' => $request->incident_category_id,
        ]);

        return back()->with('success', 'Incident report filed and prioritized for review.');
    }

    public function uploadCheckoutEvidence(Request $request, $id)
    {
        $request->validate([
            'images.*' => 'required|image|max:5120',
        ]);

        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->findOrFail($id);

        // check for existing pivot data
        $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;
        $existingImages = $pivot->checkout_images ? json_decode($pivot->checkout_images, true) : [];

        $newImages = [];
        if ($request->hasFile('images')) {
            $timestamp = $pivot->actual_start_at;
            $folderPath = ImageService::getFolderPath('check-out', $schedule->duty_number, $schedule->site->name, $timestamp);
            foreach ($request->file('images') as $file) {
                $path = ImageService::processAndStore($file, $folderPath);
                if ($path) {
                    $newImages[] = $path;
                }
            }
        }

        $allImages = array_merge($existingImages, $newImages);

        $schedule->employees()->updateExistingPivot($employee->id, [
            'checkout_images' => json_encode($allImages),
        ]);

        return back()->with('success', 'check out images uploaded successfully');
    }

    public function deleteEvidence($id, $type, $index)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->findOrFail($id);
        $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;

        if ($type === 'checkin') {
            $images = json_decode($pivot->checkin_images, true) ?? [];
            if (isset($images[$index])) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($images[$index]);
                unset($images[$index]);
                $schedule->employees()->updateExistingPivot($employee->id, [
                    'checkin_images' => json_encode(array_values($images)),
                ]);
            }
        } elseif ($type === 'checkout') {
            $images = json_decode($pivot->checkout_images, true) ?? [];
            if (isset($images[$index])) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($images[$index]);
                unset($images[$index]);
                $schedule->employees()->updateExistingPivot($employee->id, [
                    'checkout_images' => json_encode(array_values($images)),
                ]);
            }
        }

        return back()->with('success', 'Evidence deleted successfully.');
    }

    public function endJob(Request $request, $id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->findOrFail($id);
        $pivot = $schedule->employees()->where('employee_schedule.employee_id', $employee->id)->first()->pivot;

        $errors = [];
        if (empty($pivot->checkin_images)) {
            $errors[] = "Infiltration Evidence (Check-in Photos) missing.";
        }
        if (empty($pivot->checkout_images)) {
            $errors[] = "Extraction Evidence (Checkout Photos) missing.";
        }

        if (count($errors) > 0) {
            return back()->with('error', 'Job Clearance Denied: ' . implode(' ', $errors));
        }

        // Correctly update the pivot table using updateExistingPivot
        $schedule->employees()->updateExistingPivot($employee->id, [
            'actual_end_at' => Carbon::now(),
            'end_lat' => $request->latitude,
            'end_lng' => $request->longitude,
        ]);

        // Refresh the relation to check status of all employees
        $schedule->load('employees');

        // Check if ANY employee still has actual_end_at as null
        // We use the pivot data from the reloaded relationship
        $hasActiveEmployees = $schedule->employees->contains(function ($emp) {
            return is_null($emp->pivot->actual_end_at);
        });

        if (!$hasActiveEmployees) {
            $schedule->update(['status' => 'completed']);
        }

        return back()->with('success', 'Job Terminated. Extraction complete.');
    }

    public function downloadReport($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->with([
            'site',
            'incidents' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            }
        ])->findOrFail($id);

        $pivot = $schedule->pivot;

        $isPdf = true;
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('employee.reports.complete', compact('schedule', 'employee', 'pivot', 'isPdf'));

        return $pdf->download("Job_Report_{$schedule->duty_number}.pdf");
    }

    public function viewReport($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->with([
            'site',
            'incidents' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            }
        ])->findOrFail($id);

        $pivot = $schedule->pivot;
        $isPdf = false;

        return view('employee.reports.complete', compact('schedule', 'employee', 'pivot', 'isPdf'));
    }

    public function downloadCheckinReport($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->with(['site'])->findOrFail($id);
        $pivot = $schedule->pivot;
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('employee.reports.checkin', compact('schedule', 'employee', 'pivot'));
        return $pdf->download("Checkin_Report_{$schedule->duty_number}.pdf");
    }

    public function downloadCheckoutReport($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->with(['site'])->findOrFail($id);
        $pivot = $schedule->pivot;
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('employee.reports.checkout', compact('schedule', 'employee', 'pivot'));
        return $pdf->download("Checkout_Report_{$schedule->duty_number}.pdf");
    }

    public function downloadPayoutReport($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->with(['site'])->findOrFail($id);
        $pivot = $schedule->pivot;

        $s_start = $schedule->from_datetime;
        $s_end = $schedule->to_datetime;
        $a_start = $pivot->actual_start_at ? Carbon::parse($pivot->actual_start_at) : null;
        $a_end = $pivot->actual_end_at ? Carbon::parse($pivot->actual_end_at) : null;

        $payoutData = [
            'hours' => 0,
            'payout' => 0,
            'payout_start' => null,
            'payout_end' => null,
            'wage_rate' => $pivot->wage_rate ?? 0,
            'gas_rate' => $pivot->gas_rate ?? 0,
        ];

        // If wage_rate is set, use duty hours (scheduled hours) for calculation
        if ($payoutData['wage_rate'] > 0) {
            // Calculate duty hours from scheduled times
            $dutyHours = $s_start->diffInMinutes($s_end) / 60;
            $payoutData['hours'] = round($dutyHours, 2);
            $payoutData['payout'] = ($payoutData['hours'] * $payoutData['wage_rate']) + $payoutData['gas_rate'];
            $payoutData['payout_start'] = $s_start;
            $payoutData['payout_end'] = $s_end;
        } else {
            // Use actual hours with payout rules when wage_rate is not set
            if ($a_start && $a_end) {
                // Rule 1: Early check-in -> scheduled start used.
                // Rule 2: Late check-in -> actual check-in used.
                $payout_start = $a_start->gt($s_start) ? $a_start : $s_start;

                // Rule 3: Early checkout -> actual checkout used.
                // Implied Rule: Late checkout -> scheduled end used (standard procedure).
                $payout_end = $a_end->lt($s_end) ? $a_end : $s_end;

                $durationMinutes = $payout_start->diffInMinutes($payout_end);
                if ($payout_end->lt($payout_start))
                    $durationMinutes = 0;

                $payoutData['hours'] = round($durationMinutes / 60, 2);
                $payoutData['payout'] = ($payoutData['hours'] * $payoutData['wage_rate']) + $payoutData['gas_rate'];
                $payoutData['payout_start'] = $payout_start;
                $payoutData['payout_end'] = $payout_end;
            }
        }

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('employee.reports.payout', compact('schedule', 'employee', 'pivot', 'payoutData'));
        return $pdf->download("Payout_Report_{$schedule->duty_number}.pdf");
    }

    public function downloadZip($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return redirect()->route('employee.login');

        $schedule = $employee->schedules()->with([
            'incidents' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            }
        ])->findOrFail($id);

        $dutyId = $schedule->duty_number ?? $schedule->id;
        $zipFileName = "Evidence_Duty_{$dutyId}.zip";
        $zipPath = storage_path("app/public/temp/{$zipFileName}");

        // Ensure temp directory exists
        if (!file_exists(dirname($zipPath))) {
            mkdir(dirname($zipPath), 0755, true);
        }

        $zip = new \ZipArchive;
        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === TRUE) {

            // We follow the requested structure:
            // {DutyID_Timestamp}/
            //   images/
            //     checkin_images/
            //     checkout_images/
            //     report_incidents/
            //       incident1/
            //       incident2/

            $pivot = $schedule->pivot;
            $timestamp = $pivot->actual_start_at ? Carbon::parse($pivot->actual_start_at)->format('Y-m-d_H-i-s') : now()->format('Y-m-d_H-i-s');
            $dutyFolder = "{$dutyId}_{$timestamp}";

            // Checkin Images
            $checkinImages = json_decode($pivot->checkin_images ?? '[]');
            if (is_array($checkinImages)) {
                foreach ($checkinImages as $index => $imagePath) {
                    $fullPath = storage_path('app/public/' . $imagePath);
                    if (file_exists($fullPath)) {
                        $extension = pathinfo($fullPath, PATHINFO_EXTENSION);
                        $zip->addFile($fullPath, "{$dutyFolder}/images/checkin_images/checkin_{$index}.{$extension}");
                    }
                }
            }

            // Checkout Images
            $checkoutImages = json_decode($pivot->checkout_images ?? '[]');
            if (is_array($checkoutImages)) {
                foreach ($checkoutImages as $index => $imagePath) {
                    $fullPath = storage_path('app/public/' . $imagePath);
                    if (file_exists($fullPath)) {
                        $extension = pathinfo($fullPath, PATHINFO_EXTENSION);
                        $zip->addFile($fullPath, "{$dutyFolder}/images/checkout_images/checkout_{$index}.{$extension}");
                    }
                }
            }

            // Incident Images
            foreach ($schedule->incidents as $incident) {
                $incidentImages = $incident->images; // Casted as array in model
                if (is_array($incidentImages)) {
                    $siteName = $schedule->site->name ?? 'Site';
                    $cleanSiteName = str_replace([' ', '/', '\\', ':', '*', '?', '"', '<', '>', '|'], '_', $siteName);
                    $incidentTime = $incident->created_at->format('H-i-s');
                    $incidentFolderName = "{$cleanSiteName}_{$incidentTime}";
                    foreach ($incidentImages as $index => $imagePath) {
                        $fullPath = storage_path('app/public/' . $imagePath);
                        if (file_exists($fullPath)) {
                            $extension = pathinfo($fullPath, PATHINFO_EXTENSION);
                            $zip->addFile($fullPath, "{$dutyFolder}/images/report_incident/{$incidentFolderName}/incident_{$index}.{$extension}");
                        }
                    }
                }
            }

            $zip->close();
        }

        return response()->download($zipPath)->deleteFileAfterSend(true);
    }

    public function logLocation(Request $request, \App\Models\Schedule $schedule)
    {
        $request->validate([
            'latitude' => 'required',
            'longitude' => 'required',
        ]);

        $employee = Auth::guard('employee')->user();
        if (!$employee) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        \App\Models\JobLocationLog::create([
            'schedule_id' => $schedule->id,
            'employee_id' => $employee->id,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'accuracy' => $request->accuracy,
            'recorded_at' => Carbon::now(),
        ]);

        return response()->json(['status' => 'logged']);
    }
    public function scanCheckpoint(Request $request, $id)
    {
        $request->validate([
            'checkpoint_code' => 'required|string',
            'latitude' => 'nullable',
            'longitude' => 'nullable',
            'image' => 'nullable|image|max:5120',
            'employee_message' => 'nullable|string',
            'additional_photos.*' => 'nullable|image|max:5120',
        ]);

        $employee = Auth::guard('employee')->user();
        $schedule = $employee->schedules()->with('tourRoutes.checkpoints')->findOrFail($id);

        // Find the checkpoint by code within the assigned tour routes
        $checkpoint = null;
        $foundTourRoute = null;

        foreach ($schedule->tourRoutes as $tour) {
            $cp = $tour->checkpoints->where('checkpoint_id_code', $request->checkpoint_code)->first();
            if ($cp) {
                $checkpoint = $cp;
                $foundTourRoute = $tour;
                break;
            }
        }

        if (!$checkpoint) {
            return back()->with('error', 'Checkpoint ID #' . $request->checkpoint_code . ' does not match any checkpoints in your assigned tours.');
        }

        $imagePath = null;
        if ($request->hasFile('image')) {
            $folderPath = ImageService::getFolderPath('checkpoint-scans', $schedule->duty_number, $schedule->site->name, now());
            $imagePath = ImageService::processAndStore($request->file('image'), $folderPath);
        }

        $additionalPhotos = [];
        if ($request->hasFile('additional_photos')) {
            $folderPath = ImageService::getFolderPath('checkpoint-scans', $schedule->duty_number, $schedule->site->name, now());
            foreach ($request->file('additional_photos') as $photo) {
                $additionalPhotos[] = ImageService::processAndStore($photo, $folderPath);
            }
        }

        \App\Models\CheckpointScan::create([
            'schedule_id' => $schedule->id,
            'employee_id' => $employee->id,
            'checkpoint_id' => $checkpoint->id,
            'tour_route_id' => $foundTourRoute->id,
            'scanned_at' => now(),
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'is_manual' => true,
            'scan_code' => $request->checkpoint_code,
            'evidence_image' => $imagePath,
            'employee_message' => $request->employee_message,
            'additional_photos' => $additionalPhotos,
        ]);

        $nextScanTime = now()->addMinutes($checkpoint->scan_interval_minutes)->format('H:i');

        return back()
            ->with('success', "Verified: [{$checkpoint->name}] has been scanned. Protocol updated. The next required scan is scheduled for {$nextScanTime}.")
            ->with('open_tour_modal', $id);
    }
    public function getRelieverInfo(\App\Models\Schedule $schedule)
    {
        $nextSchedule = \App\Models\Schedule::where('site_id', $schedule->site_id)
            ->where('from_datetime', '>=', $schedule->to_datetime->subMinutes(30))
            ->where('from_datetime', '<=', $schedule->to_datetime->addHours(4)) // optimized to avoid scanning too far
            ->where('id', '!=', $schedule->id)
            ->whereNotIn('status', ['cancelled', 'missed'])
            ->with(['employees'])
            ->orderBy('from_datetime', 'asc')
            ->first();

        if (!$nextSchedule) {
            return response()->json(null); // No reliever found
        }

        $currentEmployeeId = Auth::guard('employee')->id();
        $reliever = $nextSchedule->employees->where('id', '!=', $currentEmployeeId)->first();

        if (!$reliever) {
            // Maybe same employee?
            $reliever = $nextSchedule->employees->first();
            if ($reliever && $reliever->id == $currentEmployeeId) {
                return response()->json(['self_relief' => true]);
            }
            // Or empty schedule
            return response()->json([
                'site_name' => $schedule->site->name,
                'start_time' => $nextSchedule->from_datetime->format('H:i'),
                'employee' => null
            ]);
        }

        return response()->json([
            'site_name' => $schedule->site->name,
            'start_time' => $nextSchedule->from_datetime->format('H:i'),
            'employee' => [
                'name' => $reliever->first_name . ' ' . $reliever->last_name,
                'photo' => $reliever->profile_picture_url,
                'phone' => $reliever->phone_number,
                'id' => $reliever->employee_id
            ]
        ]);
    }

    public function storeNote(Request $request, $id)
    {
        $request->validate([
            'note' => 'required|string',
            'job_type' => 'required|in:guard,patroller',
            'image' => 'nullable|image|max:5120',
        ]);

        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return response()->json(['error' => 'Unauthorized'], 401);

        $imagePath = null;
        if ($request->hasFile('image')) {
            $schedule = \App\Models\Schedule::findOrFail($id);
            $folderPath = \App\Services\ImageService::getFolderPath('job-notes', $schedule->duty_number, $schedule->site->name, now());
            $imagePath = \App\Services\ImageService::processAndStore($request->file('image'), $folderPath);
        }

        \App\Models\JobNote::create([
            'employee_id' => $employee->id,
            'job_type' => $request->job_type,
            'job_id' => $id,
            'note' => $request->note,
            'image' => $imagePath,
        ]);

        return response()->json(['success' => true, 'message' => 'Note saved successfully.']);
    }

    public function getNotes($id)
    {
        $employee = Auth::guard('employee')->user();
        if (!$employee)
            return response()->json(['error' => 'Unauthorized'], 401);

        $notes = \App\Models\JobNote::where('job_id', $id)
            ->where('job_type', 'guard')
            ->where('employee_id', $employee->id)
            ->latest()
            ->get();

        return response()->json(['success' => true, 'notes' => $notes]);
    }
}
