<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;

use Spatie\Permission\Models\Role;

use Barryvdh\DomPDF\Facade\Pdf;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');
        $query = User::with(['companies', 'roles']);

        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Filter to show system users by excluding employees and guards
        // These are distinct entities managed elsewhere
        $query->where(function ($q) {
            $q->whereDoesntHave('roles', function ($rq) {
                $rq->whereIn('name', ['employee', 'guard']);
            });
        });

        // Also check the 'role' column if it's being used exclusively in legacy data
        if (\Schema::hasColumn('users', 'role')) {
            $query->where(function ($q) {
                $q->whereNotIn('role', ['employee', 'guard'])
                    ->orWhereNull('role');
            });
        }

        $users = $query->latest()->paginate(100)->withQueryString();
        $companies = Company::orderBy('name')->get();
        return view('admin.users.index', compact('users', 'companies', 'search'));
    }

    public function exportPDF()
    {
        $users = User::with('roles')->latest()->get();
        $data = [];
        foreach ($users as $user) {
            $data[] = [
                $user->name,
                $user->email,
                $user->roles->pluck('name')->implode(', '),
                $user->active ? 'Active' : 'Inactive',
                $user->created_at->format('M d, Y')
            ];
        }

        $pdf = Pdf::loadView('admin.exports.pdf_template', [
            'title' => 'Employees List',
            'headers' => ['Name', 'Email', 'Roles', 'Status', 'Joined At'],
            'data' => $data
        ]);

        return $pdf->download('employees-' . date('Y-m-d') . '.pdf');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $roles = Role::whereNotIn('name', ['employee', 'guard'])->get();
        return view('admin.users.create', compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'role' => ['required', 'exists:roles,name'],
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'active' => 1,
        ]);

        $user->assignRole($request->role);

        return redirect()->route('admin.users.index')
            ->with('success', 'User created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $roles = Role::whereNotIn('name', ['employee', 'guard'])->get();
        return view('admin.users.edit', compact('user', 'roles'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'password' => ['nullable', 'string', 'min:8', 'confirmed'],
            'active' => ['boolean'],
            'role' => ['required', 'exists:roles,name'],
        ]);

        $data = [
            'name' => $request->name,
            'email' => $request->email,
            'active' => $request->has('active') ? 1 : 0,
        ];

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $user->update($data);
        $user->syncRoles([$request->role]);

        return redirect()->route('admin.users.index')
            ->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        $user->delete();

        return redirect()->route('admin.users.index')
            ->with('success', 'User deleted successfully.');
    }

    /**
     * Update the companies assigned to a user.
     */
    public function syncCompanies(Request $request, User $user)
    {
        $request->validate([
            'companies' => 'array',
            'companies.*' => 'exists:companies,id',
        ]);

        $user->companies()->sync($request->companies ?? []);

        return redirect()->route('admin.users.index')
            ->with('success', 'Assigned companies updated successfully.');
    }

    public function toggleStatus(User $user)
    {
        $user->update(['active' => !$user->active]);
        return response()->json(['success' => true, 'active' => (bool) $user->active]);
    }

    public function changePassword(Request $request, User $user)
    {
        $request->validate([
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return redirect()->back()->with('success', 'Password updated successfully for ' . $user->name);
    }

    public function analytics(User $user)
    {
        return view('admin.users.analytics', compact('user'));
    }

    public function myAnalytics()
    {
        $user = auth()->user();
        return view('admin.users.analytics', compact('user'));
    }
}
