<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use Barryvdh\DomPDF\Facade\Pdf;

class StateController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');
        $sort = $request->input('sort', 'latest');

        $query = \App\Models\State::with('country');

        if ($search) {
            $query->where('name', 'like', "%{$search}%")
                  ->orWhereHas('country', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
        }

        switch ($sort) {
            case 'name_asc': $query->orderBy('name', 'asc'); break;
            case 'name_desc': $query->orderBy('name', 'desc'); break;
            case 'country_asc': $query->select('states.*')->join('countries', 'states.country_id', '=', 'countries.id')->orderBy('countries.name', 'asc'); break;
            case 'oldest': $query->orderBy('states.created_at', 'asc'); break;
            default: $query->latest('states.created_at'); break;
        }

        $states = $query->paginate(100)->withQueryString();
        return view('admin.states.index', compact('states', 'search'));
    }

    public function exportPDF()
    {
        $states = \App\Models\State::with('country')->latest()->get();
        $data = [];
        foreach ($states as $state) {
            $data[] = [
                $state->name,
                $state->country->name ?? 'N/A',
                $state->code ?? 'N/A',
                $state->active ? 'Active' : 'Inactive'
            ];
        }

        $pdf = Pdf::loadView('admin.exports.pdf_template', [
            'title' => 'States & Provinces List',
            'headers' => ['Name', 'Country', 'Code', 'Status'],
            'data' => $data
        ]);

        return $pdf->download('states-' . date('Y-m-d') . '.pdf');
    }

    public function create()
    {
        $countries = \App\Models\Country::orderBy('name')->get();
        return view('admin.states.create', compact('countries'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'country_id' => 'required|exists:countries,id',
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:10',
            'active' => 'boolean',
        ]);

        \App\Models\State::create($validated);

        return redirect()->route('admin.states.index')
            ->with('success', 'State created successfully.');
    }

    public function edit(\App\Models\State $state)
    {
        $countries = \App\Models\Country::orderBy('name')->get();
        return view('admin.states.edit', compact('state', 'countries'));
    }

    public function update(Request $request, \App\Models\State $state)
    {
        $validated = $request->validate([
            'country_id' => 'required|exists:countries,id',
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:10',
            'active' => 'boolean',
        ]);

        $state->update($validated);

        return redirect()->route('admin.states.index')
            ->with('success', 'State updated successfully.');
    }

    public function destroy(\App\Models\State $state)
    {
        $state->delete();
        return redirect()->route('admin.states.index')
            ->with('success', 'State deleted successfully.');
    }

    public function toggleStatus(\App\Models\State $state)
    {
        $state->update(['active' => !$state->active]);
        return response()->json(['success' => true, 'active' => (bool)$state->active]);
    }
}
