<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\RequestType;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class RequestTypeController extends Controller
{
    public function index()
    {
        $companyId = session('selected_company_id') ?? \Illuminate\Support\Facades\Auth::user()->last_selected_company_id;
        $requestTypes = RequestType::where(function ($query) use ($companyId) {
            if ($companyId) {
                $query->where('company_id', $companyId)
                    ->orWhereNull('company_id');
            } else {
                $query->whereNull('company_id');
            }
        })->get();
        return view('admin.request_types.index', compact('requestTypes'));
    }

    public function create()
    {
        return view('admin.request_types.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'template' => 'nullable|string',
        ]);

        $companyId = session('selected_company_id') ?? \Illuminate\Support\Facades\Auth::user()->last_selected_company_id;

        // Auto-generate key from name, ensuring uniqueness
        $key = Str::slug($request->name, '_');
        $originalKey = $key;
        $counter = 1;

        while (RequestType::where('company_id', $companyId)->where('key', $key)->exists()) {
            $key = $originalKey . '_' . $counter;
            $counter++;
        }

        RequestType::create([
            'company_id' => $companyId,
            'name' => $request->name,
            'key' => $key,
            'template' => $request->template,
            'is_active' => true,
        ]);

        return redirect()->route('admin.request-types.index')->with('success', 'Request Type created successfully.');
    }

    public function edit(RequestType $requestType)
    {
        $companyId = session('selected_company_id') ?? \Illuminate\Support\Facades\Auth::user()->last_selected_company_id;
        if ($requestType->company_id && $requestType->company_id != $companyId) {
            abort(403);
        }
        return view('admin.request_types.edit', compact('requestType'));
    }

    public function update(Request $request, RequestType $requestType)
    {
        $companyId = session('selected_company_id') ?? \Illuminate\Support\Facades\Auth::user()->last_selected_company_id;
        if ($requestType->company_id && $requestType->company_id != $companyId) {
            abort(403);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'template' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        // We generally don't update the 'key' to avoid breaking historical data link if using key.
        // But if name changes heavily, user might expect key change.
        // Safer to Keep key constant or allow expert edit.
        // For now, let's only update name/template/active. 

        $requestType->update([
            'name' => $request->name,
            'template' => $request->template,
            'is_active' => $request->has('is_active'),
        ]);

        return redirect()->route('admin.request-types.index')->with('success', 'Request Type updated successfully.');
    }

    public function destroy(RequestType $requestType)
    {
        $companyId = session('selected_company_id') ?? \Illuminate\Support\Facades\Auth::user()->last_selected_company_id;
        if ($requestType->company_id && $requestType->company_id != $companyId) {
            abort(403);
        }

        $requestType->delete();
        return redirect()->route('admin.request-types.index')->with('success', 'Request Type deleted successfully.');
    }

    public function toggleStatus(RequestType $requestType)
    {
        $companyId = session('selected_company_id') ?? \Illuminate\Support\Facades\Auth::user()->last_selected_company_id;
        if ($requestType->company_id && $requestType->company_id != $companyId) {
            abort(403);
        }

        $requestType->update(['is_active' => !$requestType->is_active]);
        return back()->with('success', 'Status updated successfully.');
    }
}
