<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\PatrollerSchedule;
use App\Models\PatrollerIssueTicket;
use App\Models\PatrollerJobSite;
use App\Models\Employee;
use App\Models\Route;
use App\Models\Site;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Log;

class PatrollerReportController extends Controller
{
    /**
     * Patroller In/Out Report (Pivot Data)
     */
    public function index(Request $request)
    {
        $company_id = session('selected_company_id');
        $reportType = $request->query('type', 'all'); // all, checkin, checkout

        $query = $this->buildInOutQuery($company_id, $request);
        $reports = $query->paginate(20);

        $routes = Route::where('company_id', $company_id)->orderBy('name')->get();
        $employees = Employee::where('company_id', $company_id)->orderBy('first_name')->get();

        return view('admin.reports.patroller.index', compact('reports', 'reportType', 'routes', 'employees'));
    }

    public function exportInOutExcel(Request $request)
    {
        // ... (Similar to OperationalReportController but for Patroller)
    }

    public function exportInOutPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->buildInOutQuery($company_id, $request);

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_patroller_schedule.id', $ids);
        }

        $reports = $query->get();
        $pdf = Pdf::loadView('admin.reports.patroller.in_out_pdf', compact('reports'))->setOptions([
            'tempDir' => public_path(),
            'chroot' => public_path(),
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => true,
            'defaultFont' => 'DejaVu Sans',
            'dpi' => 72,
            'defaultCompression' => true
        ]);
        return $pdf->download("Patroller_InOut_Report_" . now()->format('Ymd_His') . ".pdf");
    }

    /**
     * Tickets Reports (Today, Weekly, Monthly, Range)
     */
    public function tickets(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->buildTicketQuery($company_id, $request);

        $tickets = $query->paginate(100);

        $routes = Route::where('company_id', $company_id)->orderBy('name')->get();
        // Get sites that belong to routes of this company or just all sites
        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();
        $employees = Employee::where('company_id', $company_id)->orderBy('first_name')->get();

        return view('admin.reports.patroller.tickets', compact('tickets', 'routes', 'sites', 'employees'));
    }

    /**
     * Private Query Builder for In/Out
     */
    private function buildInOutQuery($company_id, Request $request)
    {
        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->select(
                'employee_patroller_schedule.id as record_id',
                'patroller_schedules.id as schedule_id',
                'patroller_schedules.duty_number',
                'patroller_schedules.scheduled_date',
                'patroller_schedules.from_time',
                'patroller_schedules.to_time',
                'patroller_schedules.job_status',
                'routes.name as route_name',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as actual_start_at"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as actual_end_at"),
                DB::raw("(SELECT COUNT(*) FROM patroller_issue_tickets WHERE patroller_issue_tickets.patroller_schedule_id = patroller_schedules.id AND patroller_issue_tickets.employee_id = employees.id) as tickets_count")
            )
            ->orderBy('patroller_schedules.scheduled_date', 'desc');

        // Apply filters (Date, Route, Employee, IDs)
        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_patroller_schedule.id', $ids);
        } else {
            if ($request->filled('start_date')) {
                $query->whereDate('patroller_schedules.scheduled_date', '>=', $request->start_date);
            }
            if ($request->filled('end_date')) {
                $query->whereDate('patroller_schedules.scheduled_date', '<=', $request->end_date);
            }
            if ($request->filled('route_id')) {
                $query->where('patroller_schedules.route_id', $request->route_id);
            }
            if ($request->filled('employee_id')) {
                $query->where('employee_patroller_schedule.employee_id', $request->employee_id);
            }
        }

        return $query; // Not filtering by started/ended strictly unless requested, unlike operational
    }

    /**
     * Private Query Builder for Tickets
     */
    public function exportTicketsExcel(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->buildTicketQuery($company_id, $request);
        $tickets = $query->get();

        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Headers
        $headers = ['Ticket #', 'Date', 'Type', 'Site', 'Route', 'Patroller', 'Description', 'Image Count'];
        $sheet->fromArray($headers, NULL, 'A1');

        $data = [];
        foreach ($tickets as $ticket) {
            $data[] = [
                $ticket->ticket_number,
                $ticket->created_at->format('Y-m-d H:i'),
                ucfirst($ticket->status),
                $ticket->site->name ?? '-',
                $ticket->patrollerSchedule->route->name ?? '-',
                ($ticket->employee->first_name ?? '') . ' ' . ($ticket->employee->last_name ?? ''),
                $ticket->description,
                is_array($ticket->images) ? count($ticket->images) : 0
            ];
        }

        if (count($data) > 0) {
            $sheet->fromArray($data, NULL, 'A2');
        }

        foreach (range('A', 'H') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        $filename = "Ticket_Report_" . now()->format('Ymd_His') . ".xlsx";

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename);
    }

    public function exportTicketsPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->buildTicketQuery($company_id, $request);
        $tickets = $query->get();

        $pdf = Pdf::loadView('admin.reports.patroller.tickets_pdf', compact('tickets'))->setOptions([
            'tempDir' => public_path(),
            'chroot' => public_path(),
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => true,
            'defaultFont' => 'DejaVu Sans',
            'dpi' => 72,
            'defaultCompression' => true
        ]);
        return $pdf->download("Ticket_Report_" . now()->format('Ymd_His') . ".pdf");
    }

    public function exportTicketsZip(Request $request)
    {
        $company_id = session('selected_company_id');
        $query = $this->buildTicketQuery($company_id, $request);
        $tickets = $query->get();

        $zipFileName = "Ticket_Images_" . now()->format('Ymd_His') . ".zip";
        $zipPath = storage_path('app/public/' . $zipFileName);

        $zip = new \ZipArchive;
        if ($zip->open($zipPath, \ZipArchive::CREATE) === TRUE) {
            $isUserRole = auth()->user()->role === 'user';
            $timestamp = now()->format('Ymd_His');

            foreach ($tickets as $ticket) {
                if (!empty($ticket->images) && is_array($ticket->images)) {
                    foreach ($ticket->images as $index => $image) {
                        $path = null;
                        $name = "img_{$index}.jpg";

                        if (is_string($image)) {
                            $path = $image;
                        } elseif (is_array($image) && isset($image['path'])) {
                            $path = $image['path'];
                            $name = $image['name'] ?? $name;
                        } elseif (is_object($image) && isset($image->path)) {
                            $path = $image->path;
                            $name = $image->name ?? $name;
                        }

                        if ($path) {
                            $fullPath = storage_path('app/public/' . $path);
                            if (file_exists($fullPath)) {
                                $folder = $ticket->ticket_number . "_" . $ticket->status;

                                if ($isUserRole) {
                                    $routeName = $ticket->patrollerSchedule->route->name ?? 'Unknown_Route';
                                    // Sanitize folder name
                                    $cleanRouteName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $routeName);
                                    $routeFolder = $cleanRouteName . "_" . $timestamp;
                                    $zip->addFile($fullPath, $routeFolder . '/' . $folder . '/' . $name);
                                } else {
                                    $zip->addFile($fullPath, $folder . '/' . $name);
                                }
                            }
                        }
                    }
                }
            }
            $zip->close();
        }

        return response()->download($zipPath)->deleteFileAfterSend(true);
    }

    /**
     * Private Query Builder for Tickets
     */
    private function buildTicketQuery($company_id, Request $request)
    {
        $query = PatrollerIssueTicket::with(['patrollerSchedule.route', 'employee', 'site'])
            ->whereHas('employee', function ($q) use ($company_id) {
                $q->where('company_id', $company_id);
            });

        // Filter by IDs if provided
        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('id', $ids);
        } else {
            // Apply other filters
            if ($request->filled('period')) {
                $period = $request->period;
                if ($period === 'today') {
                    $query->whereDate('created_at', now()->today());
                } elseif ($period === 'week') {
                    $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
                } elseif ($period === 'month') {
                    $query->whereMonth('created_at', now()->month)->whereYear('created_at', now()->year);
                }
            }

            if ($request->filled('start_date')) {
                $query->whereDate('created_at', '>=', $request->start_date);
            }
            if ($request->filled('end_date')) {
                $query->whereDate('created_at', '<=', $request->end_date);
            }
            if ($request->filled('route_id')) {
                $query->whereHas('patrollerSchedule', function ($q) use ($request) {
                    $q->where('route_id', $request->route_id);
                });
            }
            if ($request->filled('site_id')) {
                $query->where('site_id', $request->site_id);
            }
            if ($request->filled('duty_number')) {
                $query->whereHas('patrollerSchedule', function ($q) use ($request) {
                    $q->where('duty_number', 'like', '%' . $request->duty_number . '%');
                });
            }
            if ($request->filled('ticket_number')) {
                $query->where('ticket_number', 'like', '%' . $request->ticket_number . '%');
            }
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }
            if ($request->filled('employee_id')) {
                $query->where('employee_id', $request->employee_id);
            }
        }

        // Sorting
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if ($sort === 'type') {
            $query->orderBy('status', $direction);
        } elseif ($sort === 'site') {
            $query->join('sites', 'patroller_issue_tickets.site_id', '=', 'sites.id')
                ->orderBy('sites.name', $direction)
                ->select('patroller_issue_tickets.*');
        } elseif ($sort === 'patroller') {
            $query->join('employees', 'patroller_issue_tickets.employee_id', '=', 'employees.id')
                ->orderBy('employees.first_name', $direction)
                ->select('patroller_issue_tickets.*');
        } else {
            $query->orderBy('created_at', 'desc');
        }

        return $query;
    }
    /**
     * Monthly Ticket Chart Report
     */
    public function monthlyTickets(Request $request)
    {
        $company_id = session('selected_company_id');
        $month = $request->input('month', now()->month);
        $year = $request->input('year', now()->year);
        $siteId = $request->input('site_id');

        // Get all sites for filter
        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();

        // Calculate days in month
        $totalDaysInMonth = \Carbon\Carbon::create($year, $month)->daysInMonth;

        // If current month/year, only show up to today
        if ($year == now()->year && $month == now()->month) {
            $daysInMonth = range(1, min($totalDaysInMonth, now()->day));
        } else {
            $daysInMonth = range(1, $totalDaysInMonth);
        }

        // Build query
        $query = PatrollerIssueTicket::query()
            ->whereHas('patrollerSchedule', function ($q) use ($company_id) {
                $q->where('company_id', $company_id);
            })
            ->whereYear('created_at', $year)
            ->whereMonth('created_at', $month);

        if ($siteId) {
            $query->where('site_id', $siteId);
        }

        // Get tickets grouped by site and day
        $tickets = $query->with('site')
            ->selectRaw('site_id, DAY(created_at) as day, COUNT(*) as count')
            ->groupBy('site_id', 'day')
            ->get();

        // Get unique sites from results
        $siteIds = $tickets->pluck('site_id')->unique();
        $sitesData = Site::whereIn('id', $siteIds)->get()->keyBy('id');

        // Build report data structure
        $reportData = [];
        foreach ($siteIds as $sid) {
            $siteName = $sitesData[$sid]->name ?? 'Unknown Site';
            $reportData[$sid] = [
                'name' => $siteName,
                'days' => array_fill_keys($daysInMonth, 0),
                'total' => 0
            ];
        }

        // Fill in ticket counts
        foreach ($tickets as $ticket) {
            $reportData[$ticket->site_id]['days'][$ticket->day] = $ticket->count;
            $reportData[$ticket->site_id]['total'] += $ticket->count;
        }

        return view('admin.reports.patroller.monthly_tickets', compact(
            'reportData',
            'daysInMonth',
            'sites',
            'month',
            'year'
        ));
    }

    /**
     * Export Monthly Ticket Chart to Excel
     */
    public function exportMonthlyTickets(Request $request)
    {
        $company_id = session('selected_company_id');
        $month = $request->input('month', now()->month);
        $year = $request->input('year', now()->year);
        $siteId = $request->input('site_id');

        $totalDaysInMonth = \Carbon\Carbon::create($year, $month)->daysInMonth;

        // If current month/year, only show up to today
        if ($year == now()->year && $month == now()->month) {
            $daysInMonth = range(1, min($totalDaysInMonth, now()->day));
        } else {
            $daysInMonth = range(1, $totalDaysInMonth);
        }

        // Build query (same as above)
        $query = PatrollerIssueTicket::query()
            ->whereHas('patrollerSchedule', function ($q) use ($company_id) {
                $q->where('company_id', $company_id);
            })
            ->whereYear('created_at', $year)
            ->whereMonth('created_at', $month);

        if ($siteId) {
            $query->where('site_id', $siteId);
        }

        $tickets = $query->with('site')
            ->selectRaw('site_id, DAY(created_at) as day, COUNT(*) as count')
            ->groupBy('site_id', 'day')
            ->get();

        $siteIds = $tickets->pluck('site_id')->unique();
        $sitesData = Site::whereIn('id', $siteIds)->get()->keyBy('id');

        $reportData = [];
        foreach ($siteIds as $sid) {
            $siteName = $sitesData[$sid]->name ?? 'Unknown Site';
            $reportData[$sid] = [
                'name' => $siteName,
                'days' => array_fill_keys($daysInMonth, 0),
                'total' => 0
            ];
        }

        foreach ($tickets as $ticket) {
            $reportData[$ticket->site_id]['days'][$ticket->day] = $ticket->count;
            $reportData[$ticket->site_id]['total'] += $ticket->count;
        }

        // Create Excel
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Headers
        $headers = ['Site'];
        foreach ($daysInMonth as $day) {
            $headers[] = $day;
        }
        $headers[] = 'Total';
        $sheet->fromArray($headers, NULL, 'A1');

        // Data
        $row = 2;
        foreach ($reportData as $siteData) {
            $rowData = [$siteData['name']];
            foreach ($daysInMonth as $day) {
                $rowData[] = $siteData['days'][$day];
            }
            $rowData[] = $siteData['total'];
            $sheet->fromArray($rowData, NULL, 'A' . $row);
            $row++;
        }

        // Auto-size columns
        $highestColumn = $sheet->getHighestColumn();
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);
        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $sheet->getColumnDimensionByColumn($col)->setAutoSize(true);
        }

        $monthName = \Carbon\Carbon::create()->month($month)->format('F');
        $filename = "Ticket_Monthly_Report_{$monthName}_{$year}_" . now()->format('Ymd_His') . ".xlsx";

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename);
    }

    /**
     * Patroller Payout Report
     */
    public function payout(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $employeeId = $request->input('employee_id');
        $reportType = $request->input('report_type', 'actual'); // 'actual' or 'scheduled'

        // Get all employees for filter
        $employees = Employee::where('company_id', $company_id)->orderBy('first_name')->get();

        // Build query
        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate)
            ->select(
                'employee_patroller_schedule.id as record_id',
                'employees.id as employee_id',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'employees.email as employee_email',
                'routes.name as route_name',
                'patroller_schedules.duty_number',
                'patroller_schedules.scheduled_date',
                'patroller_schedules.from_time as scheduled_start',
                'patroller_schedules.to_time as scheduled_end',
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as actual_start"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as actual_end"),
                'employee_patroller_schedule.wage_rate',
                'employee_patroller_schedule.gas_rate',
                'employee_patroller_schedule.allowance',
                'employee_patroller_schedule.other_expense',
                'employee_patroller_schedule.wage_types'
            );

        // Filter based on report type
        if ($reportType === 'actual') {
            $query->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at)'))
                ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at)'));
        }

        if ($employeeId) {
            $query->where('employees.id', $employeeId);
        }

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_patroller_schedule.id', $ids);
        }

        $records = $query->orderBy('employees.first_name')
            ->orderBy('patroller_schedules.scheduled_date')
            ->get();




        // Group by employee
        $payoutData = [];
        foreach ($records as $record) {
            if (!isset($payoutData[$record->employee_id])) {
                $payoutData[$record->employee_id] = [
                    'employee_name' => $record->employee_name,
                    'employee_email' => $record->employee_email,
                    'routes' => [],
                    'total_hours' => 0,
                    'total_amount' => 0,
                    'total_gas' => 0,
                    'total_allowance' => 0,
                    'grand_total' => 0
                ];
            }

            // Calculate hours based on report type
            if ($reportType === 'scheduled') {
                $start = \Carbon\Carbon::parse($record->scheduled_start);
                $end = \Carbon\Carbon::parse($record->scheduled_end);
            } else {
                // Skip if no actual times
                if (!$record->actual_start || !$record->actual_end) {
                    continue;
                }
                $start = \Carbon\Carbon::parse($record->actual_start);
                $end = \Carbon\Carbon::parse($record->actual_end);
            }

            $hours = $start->diffInMinutes($end) / 60;

            // Parse wage_types JSON and calculate amounts with waterfall logic
            $wageTypes = json_decode($record->wage_types, true);
            $sinRate = 0;
            $otherRate = 0;
            $totalAmount = 0;
            $paymentTypeDisplay = 'N/A';

            if (is_array($wageTypes) && count($wageTypes) > 0) {
                // Sort wage types: SIN (id=1) first, then others
                usort($wageTypes, function ($a, $b) {
                    $aIsSin = ($a['id'] ?? 0) == 1 ? 0 : 1;
                    $bIsSin = ($b['id'] ?? 0) == 1 ? 0 : 1;
                    return $aIsSin - $bIsSin;
                });

                $paymentTypes = [];
                $remainingHours = $hours;

                foreach ($wageTypes as $wageType) {
                    if ($remainingHours <= 0) {
                        break;
                    }

                    $rate = $wageType['rate'] ?? 0;
                    $allocatedHours = $wageType['allocated_hours'] ?? 0;
                    $wageId = $wageType['id'] ?? null;
                    $wageName = $wageType['name'] ?? "Wage {$wageId}";

                    // Waterfall: allocate hours up to the allocated amount
                    $payHours = min($remainingHours, $allocatedHours);
                    $amount = $rate * $payHours;
                    $totalAmount += $amount;
                    $remainingHours -= $payHours;

                    // Track rates (assuming id 1 is SIN, others are other)
                    if ($wageId == 1) {
                        $sinRate += $rate;
                    } else {
                        $otherRate += $rate;
                    }

                    $paymentTypes[] = "{$wageName}: " . number_format($payHours, 2) . "h @ $" . number_format($rate, 2) . "/hr = $" . number_format($amount, 2);
                }


                $paymentTypeDisplay = implode(' | ', $paymentTypes);

            } else {
                $wageRate = $record->wage_rate ?? 0;
                $totalAmount = $wageRate * $hours;
                $sinRate = $wageRate;
                $paymentTypeDisplay = '$' . number_format($wageRate, 2) . '/hr';
            }



            $gas = $record->gas_rate ?? 0;
            $allowance = $record->allowance ?? 0;
            $grandTotal = $totalAmount + $gas + $allowance;

            $payoutData[$record->employee_id]['routes'][] = [
                'record_id' => $record->record_id,
                'route_name' => $record->route_name,
                'duty_number' => $record->duty_number,
                'start_time' => $start->format('Y-m-d H:i A'),
                'end_time' => $end->format('Y-m-d H:i A'),
                'actual_hours' => number_format($hours, 2),
                'payment_type' => $paymentTypeDisplay,
                'sin_rate' => $sinRate,
                'other_rate' => $otherRate,
                'total_amount' => $totalAmount,
                'gas' => $gas,
                'allowance' => $allowance,
                'grand_total' => $grandTotal
            ];

            $payoutData[$record->employee_id]['total_hours'] += $hours;
            $payoutData[$record->employee_id]['total_amount'] += $totalAmount;
            $payoutData[$record->employee_id]['total_gas'] += $gas;
            $payoutData[$record->employee_id]['total_allowance'] += $allowance;
            $payoutData[$record->employee_id]['grand_total'] += $grandTotal;
        }

        // Format total hours
        foreach ($payoutData as &$data) {
            $data['total_hours'] = number_format($data['total_hours'], 2);
        }



        return view('admin.reports.patroller.payout', compact('payoutData', 'employees'));
    }

    /**
     * Export Patroller Payout to Excel
     */
    public function payoutExport(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $employeeId = $request->input('employee_id');

        // Same query as above
        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate)
            ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at)'))
            ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at)'))
            ->select(
                'employees.id as employee_id',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'employees.email as employee_email',
                'routes.name as route_name',
                'patroller_schedules.duty_number',
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as start_time"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as end_time"),
                'employee_patroller_schedule.wage_rate',
                'employee_patroller_schedule.gas_rate',
                'employee_patroller_schedule.allowance',
                'employee_patroller_schedule.other_expense'
            );

        if ($employeeId) {
            $query->where('employees.id', $employeeId);
        }

        $records = $query->orderBy('employees.first_name')->get();

        // Create Excel
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Headers
        $headers = ['Employee', 'Email', 'Route', 'Duty Number', 'Start Time', 'End Time', 'Hours', 'Wage Rate', 'Total Amount', 'Gas', 'Allowance', 'Grand Total'];
        $sheet->fromArray($headers, NULL, 'A1');

        // Data
        $row = 2;
        foreach ($records as $record) {
            $start = \Carbon\Carbon::parse($record->start_time);
            $end = \Carbon\Carbon::parse($record->end_time);
            $hours = $start->diffInMinutes($end) / 60;

            // Parse wage_types JSON and calculate amounts
            $wageTypes = json_decode($record->wage_types, true);
            $totalAmount = 0;

            if (is_array($wageTypes) && count($wageTypes) > 0) {
                if (count($wageTypes) === 1) {
                    $rate = $wageTypes[0]['rate'] ?? 0;
                    $totalAmount = $rate * $hours;
                } else {
                    $remainingHours = $hours;
                    foreach ($wageTypes as $wageType) {
                        if ($remainingHours <= 0)
                            break;

                        $rate = $wageType['rate'] ?? 0;
                        $allocatedHours = $wageType['allocated_hours'] ?? 0;

                        $payHours = min($remainingHours, $allocatedHours);
                        $amount = $rate * $payHours;
                        $totalAmount += $amount;
                        $remainingHours -= $payHours;
                    }
                }
            } else {
                $wageRate = $record->wage_rate ?? 0;
                $totalAmount = $wageRate * $hours;
            }

            $gas = $record->gas_rate ?? 0;
            $allowance = $record->allowance ?? 0;
            $grandTotal = $totalAmount + $gas + $allowance;

            $rowData = [
                $record->employee_name,
                $record->employee_email,
                $record->route_name,
                $record->duty_number,
                $start->format('Y-m-d H:i'),
                $end->format('Y-m-d H:i'),
                number_format($hours, 2),
                $totalAmount > 0 ? number_format($totalAmount / $hours, 2) : '0.00',
                number_format($totalAmount, 2),
                number_format($gas, 2),
                number_format($allowance, 2),
                number_format($grandTotal, 2)
            ];
            $sheet->fromArray($rowData, NULL, 'A' . $row);
            $row++;
        }

        // Auto-size columns
        $highestColumn = $sheet->getHighestColumn();
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);
        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $sheet->getColumnDimensionByColumn($col)->setAutoSize(true);
        }

        $filename = "Patroller_Payout_Report_" . now()->format('Ymd_His') . ".xlsx";

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename);
    }

    public function payoutExportPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $employeeId = $request->input('employee_id');
        $reportType = $request->input('report_type', 'actual');

        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate)
            ->select(
                'employees.id as employee_id',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'employees.email as employee_email',
                'routes.name as route_name',
                'patroller_schedules.duty_number',
                'patroller_schedules.from_time as scheduled_start',
                'patroller_schedules.to_time as scheduled_end',
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as actual_start"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as actual_end"),
                'employee_patroller_schedule.wage_types',
                'employee_patroller_schedule.wage_rate',
                'employee_patroller_schedule.gas_rate',
                'employee_patroller_schedule.allowance'
            );

        if ($reportType === 'actual') {
            $query->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at)'))
                ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at)'));
        }

        if ($employeeId) {
            $query->where('employees.id', $employeeId);
        }

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_patroller_schedule.id', $ids);
        }

        $records = $query->orderBy('employees.first_name')->get();
        $payoutData = [];
        foreach ($records as $record) {
            if (!isset($payoutData[$record->employee_id])) {
                $payoutData[$record->employee_id] = [
                    'employee_name' => $record->employee_name,
                    'employee_email' => $record->employee_email,
                    'routes' => [],
                    'total_hours' => 0,
                    'total_amount' => 0,
                    'total_gas' => 0,
                    'total_allowance' => 0,
                    'grand_total' => 0
                ];
            }
            $start = \Carbon\Carbon::parse($reportType === 'scheduled' ? $record->scheduled_start : $record->actual_start);
            $end = \Carbon\Carbon::parse($reportType === 'scheduled' ? $record->scheduled_end : $record->actual_end);
            $hours = $start->diffInMinutes($end) / 60;

            $totalAmount = 0;
            $wageTypes = json_decode($record->wage_types, true);
            if (is_array($wageTypes) && count($wageTypes) > 0) {
                usort($wageTypes, function ($a, $b) {
                    return (($a['id'] ?? 0) == 1 ? 0 : 1) - (($b['id'] ?? 0) == 1 ? 0 : 1);
                });
                $rem = $hours;
                foreach ($wageTypes as $wt) {
                    if ($rem <= 0)
                        break;
                    $ph = min($rem, $wt['allocated_hours'] ?? 0);
                    $totalAmount += ($wt['rate'] ?? 0) * $ph;
                    $rem -= $ph;
                }
            } else {
                $totalAmount = ($record->wage_rate ?? 0) * $hours;
            }
            $subTotal = $totalAmount + ($record->gas_rate ?? 0) + ($record->allowance ?? 0);
            $payoutData[$record->employee_id]['routes'][] = [
                'route_name' => $record->route_name,
                'duty_number' => $record->duty_number,
                'start_time' => $start->format('Y-m-d H:i'),
                'end_time' => $end->format('Y-m-d H:i'),
                'hours' => number_format($hours, 2),
                'total' => $subTotal
            ];

            $payoutData[$record->employee_id]['total_hours'] += $hours;
            $payoutData[$record->employee_id]['total_amount'] += $totalAmount;
            $payoutData[$record->employee_id]['total_gas'] += ($record->gas_rate ?? 0);
            $payoutData[$record->employee_id]['total_allowance'] += ($record->allowance ?? 0);
            $payoutData[$record->employee_id]['grand_total'] += $subTotal;
        }

        $pdf = Pdf::loadView('admin.reports.patroller.payout_pdf', compact('payoutData', 'startDate', 'endDate'))->setOptions([
            'tempDir' => public_path(),
            'chroot' => public_path(),
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => true,
            'defaultFont' => 'DejaVu Sans',
            'dpi' => 72,
            'defaultCompression' => true
        ]);
        return $pdf->download("Patroller_Payout_Report_" . now()->format('Ymd_His') . ".pdf");
    }


    /**
     * Route Payout Report
     */
    public function routePayout(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $routeId = $request->input('route_id');
        $reportType = $request->input('report_type', 'actual'); // 'actual' or 'scheduled'

        // Get all routes for filter
        $routes = Route::where('company_id', $company_id)->orderBy('name')->get();

        // Build query
        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate)
            ->select(
                'employee_patroller_schedule.id as record_id',
                'routes.id as route_id',
                'routes.name as route_name',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'patroller_schedules.duty_number',
                'patroller_schedules.from_time as scheduled_start',
                'patroller_schedules.to_time as scheduled_end',
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as actual_start"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as actual_end"),
                'employee_patroller_schedule.wage_rate',
                'employee_patroller_schedule.wage_types'
            );

        // Filter based on report type
        if ($reportType === 'actual') {
            $query->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at)'))
                ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at)'));
        }

        if ($routeId) {
            $query->where('routes.id', $routeId);
        }

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_patroller_schedule.id', $ids);
        }

        $records = $query->orderBy('routes.name')
            ->orderBy('patroller_schedules.scheduled_date')
            ->get();

        // Group by route
        $payoutData = [];
        foreach ($records as $record) {
            if (!isset($payoutData[$record->route_id])) {
                $payoutData[$record->route_id] = [
                    'route_name' => $record->route_name,
                    'route_code' => 'Route ' . $record->route_id,
                    'employees' => [],
                    'total_hours' => 0,
                    'total_amount' => 0
                ];
            }

            // Calculate hours based on report type
            if ($reportType === 'scheduled') {
                $start = \Carbon\Carbon::parse($record->scheduled_start);
                $end = \Carbon\Carbon::parse($record->scheduled_end);
            } else {
                // Skip if no actual times
                if (!$record->actual_start || !$record->actual_end) {
                    continue;
                }
                $start = \Carbon\Carbon::parse($record->actual_start);
                $end = \Carbon\Carbon::parse($record->actual_end);
            }

            $hours = $start->diffInMinutes($end) / 60;

            // Calculate amount with waterfall logic
            $wageTypes = json_decode($record->wage_types, true);
            $totalAmount = 0;
            $avgRate = 0;

            if (is_array($wageTypes) && count($wageTypes) > 0) {
                // Sort wage types: SIN (id=1) first, then others
                usort($wageTypes, function ($a, $b) {
                    $aIsSin = ($a['id'] ?? 0) == 1 ? 0 : 1;
                    $bIsSin = ($b['id'] ?? 0) == 1 ? 0 : 1;
                    return $aIsSin - $bIsSin;
                });

                $remainingHours = $hours;
                foreach ($wageTypes as $wageType) {
                    if ($remainingHours <= 0) {
                        break;
                    }

                    $rate = $wageType['rate'] ?? 0;
                    $allocatedHours = $wageType['allocated_hours'] ?? 0;

                    $payHours = min($remainingHours, $allocatedHours);
                    $totalAmount += $rate * $payHours;
                    $remainingHours -= $payHours;
                }

                $avgRate = $hours > 0 ? $totalAmount / $hours : 0;
            } else {
                $wageRate = $record->wage_rate ?? 0;
                $totalAmount = $wageRate * $hours;
                $avgRate = $wageRate;
            }

            $payoutData[$record->route_id]['employees'][] = [
                'record_id' => $record->record_id,
                'employee_name' => $record->employee_name,
                'duty_number' => $record->duty_number,
                'in_time' => $start->format('Y-m-d H:i A'),
                'out_time' => $end->format('Y-m-d H:i A'),
                'pay_rate' => $hours > 0 ? $totalAmount / $hours : 0,
                'total_hours' => number_format($hours, 2),
                'total_amount' => $totalAmount
            ];

            $payoutData[$record->route_id]['total_hours'] += $hours;
            $payoutData[$record->route_id]['total_amount'] += $totalAmount;
        }

        // Format total hours
        foreach ($payoutData as &$data) {
            $data['total_hours'] = number_format($data['total_hours'], 2);
        }

        return view('admin.reports.patroller.route_payout', compact('payoutData', 'routes'));
    }

    /**
     * Export Route Payout to Excel
     */
    public function routePayoutExport(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $routeId = $request->input('route_id');

        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate)
            ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at)'))
            ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at)'))
            ->select(
                'routes.name as route_name',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'patroller_schedules.duty_number',
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as start_time"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as end_time"),
                'employee_patroller_schedule.wage_rate',
                'employee_patroller_schedule.wage_types'
            );

        if ($routeId) {
            $query->where('routes.id', $routeId);
        }

        $records = $query->orderBy('routes.name')->get();

        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = ['Route', 'Employee', 'Duty Number', 'In Time', 'Out Time', 'Pay Rate', 'Total Hours', 'Total Amount'];
        $sheet->fromArray($headers, NULL, 'A1');

        $row = 2;
        foreach ($records as $record) {
            $start = \Carbon\Carbon::parse($record->start_time);
            $end = \Carbon\Carbon::parse($record->end_time);
            $hours = $start->diffInMinutes($end) / 60;

            $wageTypes = json_decode($record->wage_types, true);
            $totalAmount = 0;
            if (is_array($wageTypes) && count($wageTypes) > 0) {
                foreach ($wageTypes as $wageType) {
                    $rate = $wageType['rate'] ?? 0;
                    $allocatedHours = $wageType['allocated_hours'] ?? 0;
                    $totalAmount += $rate * min($hours, $allocatedHours);
                }
            } else {
                $wageRate = $record->wage_rate ?? 0;
                $totalAmount = $wageRate * $hours;
            }

            $avgRate = $hours > 0 ? $totalAmount / $hours : 0;

            $rowData = [
                $record->route_name,
                $record->employee_name,
                $record->duty_number,
                $start->format('Y-m-d H:i'),
                $end->format('Y-m-d H:i'),
                number_format($avgRate, 2),
                number_format($hours, 2),
                number_format($totalAmount, 2)
            ];
            $sheet->fromArray($rowData, NULL, 'A' . $row);
            $row++;
        }

        $highestColumn = $sheet->getHighestColumn();
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);
        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $sheet->getColumnDimensionByColumn($col)->setAutoSize(true);
        }

        $filename = "Route_Payout_Report_" . now()->format('Ymd_His') . ".xlsx";

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename);
    }

    public function routePayoutExportPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $routeId = $request->input('route_id');
        $reportType = $request->input('report_type', 'actual');

        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate)
            ->select(
                'routes.id as route_id',
                'routes.name as route_name',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'patroller_schedules.duty_number',
                'patroller_schedules.from_time as scheduled_start',
                'patroller_schedules.to_time as scheduled_end',
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as actual_start"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as actual_end"),
                'employee_patroller_schedule.wage_types',
                'employee_patroller_schedule.wage_rate'
            );

        if ($reportType === 'actual') {
            $query->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at)'))
                ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at)'));
        }

        if ($routeId) {
            $query->where('routes.id', $routeId);
        }

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_patroller_schedule.id', $ids);
        }

        $records = $query->orderBy('routes.name')->orderBy('patroller_schedules.scheduled_date')->get();
        $payoutData = [];
        foreach ($records as $record) {
            if (!isset($payoutData[$record->route_id])) {
                $payoutData[$record->route_id] = [
                    'route_name' => $record->route_name,
                    'employees' => [],
                    'total_hours' => 0,
                    'total_amount' => 0
                ];
            }
            $start = \Carbon\Carbon::parse($reportType === 'scheduled' ? $record->scheduled_start : $record->actual_start);
            $end = \Carbon\Carbon::parse($reportType === 'scheduled' ? $record->scheduled_end : $record->actual_end);
            $hours = $start->diffInMinutes($end) / 60;

            $totalAmount = 0;
            $wageTypes = json_decode($record->wage_types, true);
            if (is_array($wageTypes) && count($wageTypes) > 0) {
                usort($wageTypes, function ($a, $b) {
                    return (($a['id'] ?? 0) == 1 ? 0 : 1) - (($b['id'] ?? 0) == 1 ? 0 : 1);
                });
                $rem = $hours;
                foreach ($wageTypes as $wt) {
                    if ($rem <= 0)
                        break;
                    $ph = min($rem, $wt['allocated_hours'] ?? 0);
                    $totalAmount += ($wt['rate'] ?? 0) * $ph;
                    $rem -= $ph;
                }
            } else {
                $totalAmount = ($record->wage_rate ?? 0) * $hours;
            }

            $payoutData[$record->route_id]['employees'][] = [
                'employee_name' => $record->employee_name,
                'duty_number' => $record->duty_number,
                'in_time' => $start->format('Y-m-d H:i'),
                'out_time' => $end->format('Y-m-d H:i'),
                'hours' => number_format($hours, 2),
                'total' => $totalAmount
            ];
            $payoutData[$record->route_id]['total_hours'] += $hours;
            $payoutData[$record->route_id]['total_amount'] += $totalAmount;
        }

        $pdf = Pdf::loadView('admin.reports.patroller.route_payout_pdf', compact('payoutData', 'startDate', 'endDate'));
        return $pdf->download("Route_Payout_Report_" . now()->format('Ymd_His') . ".pdf");
    }


    /**
     * Route Margin Report
     */
    public function routeMargin(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $routeId = $request->input('route_id');
        $reportType = $request->input('report_type', 'actual'); // 'actual' or 'scheduled'

        $routes = Route::where('company_id', $company_id)->orderBy('name')->get();

        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate)
            ->select(
                'employee_patroller_schedule.id as record_id',
                'routes.id as route_id',
                'routes.name as route_name',
                'patroller_schedules.route_rate',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'patroller_schedules.duty_number',
                'patroller_schedules.from_time as scheduled_start',
                'patroller_schedules.to_time as scheduled_end',
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as actual_start"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as actual_end"),
                'employee_patroller_schedule.wage_rate',
                'employee_patroller_schedule.gas_rate',
                'employee_patroller_schedule.allowance',
                'employee_patroller_schedule.wage_types'
            );

        // Filter based on report type
        if ($reportType === 'actual') {
            $query->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at)'))
                ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at)'));
        }

        if ($routeId) {
            $query->where('routes.id', $routeId);
        }

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_patroller_schedule.id', $ids);
        }

        $records = $query->orderBy('routes.name')->get();

        $marginData = [];
        foreach ($records as $record) {
            if (!isset($marginData[$record->route_id])) {
                $marginData[$record->route_id] = [
                    'route_name' => $record->route_name,
                    'route_code' => 'Route ' . $record->route_id,
                    'employees' => [],
                    'total_hours' => 0,
                    'route_total' => 0,
                    'total_cost' => 0,
                    'total_margin' => 0
                ];
            }

            // Calculate hours based on report type
            if ($reportType === 'scheduled') {
                $start = \Carbon\Carbon::parse($record->scheduled_start);
                $end = \Carbon\Carbon::parse($record->scheduled_end);
            } else {
                // Skip if no actual times
                if (!$record->actual_start || !$record->actual_end) {
                    continue;
                }
                $start = \Carbon\Carbon::parse($record->actual_start);
                $end = \Carbon\Carbon::parse($record->actual_end);
            }

            $hours = $start->diffInMinutes($end) / 60;

            // Route revenue
            $routeRate = $record->route_rate ?? 0;
            $routeTotal = $routeRate * $hours;

            // Employee cost with waterfall logic
            $wageTypes = json_decode($record->wage_types, true);
            $sinRate = 0;
            $otherRate = 0;
            $totalAmount = 0;
            $paymentTypeDisplay = 'N/A';

            if (is_array($wageTypes) && count($wageTypes) > 0) {
                // Sort wage types: SIN (id=1) first, then others
                usort($wageTypes, function ($a, $b) {
                    $aIsSin = ($a['id'] ?? 0) == 1 ? 0 : 1;
                    $bIsSin = ($b['id'] ?? 0) == 1 ? 0 : 1;
                    return $aIsSin - $bIsSin;
                });

                $paymentTypes = [];
                $remainingHours = $hours;

                foreach ($wageTypes as $wageType) {
                    if ($remainingHours <= 0) {
                        break;
                    }

                    $rate = $wageType['rate'] ?? 0;
                    $allocatedHours = $wageType['allocated_hours'] ?? 0;
                    $wageId = $wageType['id'] ?? null;

                    // Waterfall: allocate hours up to the allocated amount
                    $payHours = min($remainingHours, $allocatedHours);
                    $totalAmount += $rate * $payHours;
                    $remainingHours -= $payHours;

                    if ($wageId == 1) {
                        $sinRate += $rate;
                        $paymentTypes[] = 'sin';
                    } else {
                        $otherRate += $rate;
                        $paymentTypes[] = 'other';
                    }
                }
                $paymentTypeDisplay = implode('+', array_unique($paymentTypes));
            } else {
                $wageRate = $record->wage_rate ?? 0;
                $totalAmount = $wageRate * $hours;
                $sinRate = $wageRate;
                $paymentTypeDisplay = 'sin';
            }

            $gas = $record->gas_rate ?? 0;
            $allowance = $record->allowance ?? 0;
            $totalCost = $totalAmount + $gas + $allowance;
            $margin = $routeTotal - $totalCost;

            $marginData[$record->route_id]['employees'][] = [
                'record_id' => $record->record_id,
                'employee_name' => $record->employee_name,
                'duty_number' => $record->duty_number,
                'in_time' => $start->format('Y-m-d H:i A'),
                'out_time' => $end->format('Y-m-d H:i A'),
                'total_hours' => number_format($hours, 2),
                'route_rate' => $routeRate,
                'route_total' => $routeTotal,
                'payment_type' => $paymentTypeDisplay,
                'sin_rate' => $sinRate,
                'other_rate' => $otherRate,
                'gas' => $gas,
                'allowance' => $allowance,
                'total_cost' => $totalCost,
                'margin' => $margin
            ];

            $marginData[$record->route_id]['total_hours'] += $hours;
            $marginData[$record->route_id]['route_total'] += $routeTotal;
            $marginData[$record->route_id]['total_cost'] += $totalCost;
            $marginData[$record->route_id]['total_margin'] += $margin;
        }

        foreach ($marginData as &$data) {
            $data['total_hours'] = number_format($data['total_hours'], 2);
        }

        return view('admin.reports.patroller.route_margin', compact('marginData', 'routes'));
    }

    /**
     * Export Route Margin to Excel
     */
    public function routeMarginExport(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $routeId = $request->input('route_id');

        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate)
            ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at)'))
            ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at)'))
            ->select(
                'routes.name as route_name',
                'patroller_schedules.route_rate',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'patroller_schedules.duty_number',
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as start_time"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as end_time"),
                'employee_patroller_schedule.wage_rate',
                'employee_patroller_schedule.gas_rate',
                'employee_patroller_schedule.allowance',
                'employee_patroller_schedule.wage_types'
            );

        if ($routeId) {
            $query->where('routes.id', $routeId);
        }

        $records = $query->orderBy('routes.name')->get();

        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = ['Route', 'Employee', 'Duty #', 'In Time', 'Out Time', 'Hours', 'Route Rate', 'Route Total', 'Employee Cost', 'Gas', 'Allowance', 'Total Cost', 'Margin'];
        $sheet->fromArray($headers, NULL, 'A1');

        $row = 2;
        foreach ($records as $record) {
            $start = \Carbon\Carbon::parse($record->start_time);
            $end = \Carbon\Carbon::parse($record->end_time);
            $hours = $start->diffInMinutes($end) / 60;

            $routeRate = $record->route_rate ?? 0;
            $routeTotal = $routeRate * $hours;

            $wageTypes = json_decode($record->wage_types, true);
            $totalAmount = 0;
            if (is_array($wageTypes) && count($wageTypes) > 0) {
                foreach ($wageTypes as $wageType) {
                    $rate = $wageType['rate'] ?? 0;
                    $allocatedHours = $wageType['allocated_hours'] ?? 0;
                    $totalAmount += $rate * min($hours, $allocatedHours);
                }
            } else {
                $wageRate = $record->wage_rate ?? 0;
                $totalAmount = $wageRate * $hours;
            }

            $gas = $record->gas_rate ?? 0;
            $allowance = $record->allowance ?? 0;
            $totalCost = $totalAmount + $gas + $allowance;
            $margin = $routeTotal - $totalCost;

            $rowData = [
                $record->route_name,
                $record->employee_name,
                $record->duty_number,
                $start->format('Y-m-d H:i'),
                $end->format('Y-m-d H:i'),
                number_format($hours, 2),
                number_format($routeRate, 2),
                number_format($routeTotal, 2),
                number_format($totalAmount, 2),
                number_format($gas, 2),
                number_format($allowance, 2),
                number_format($totalCost, 2),
                number_format($margin, 2)
            ];
            $sheet->fromArray($rowData, NULL, 'A' . $row);
            $row++;
        }

        $highestColumn = $sheet->getHighestColumn();
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);
        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $sheet->getColumnDimensionByColumn($col)->setAutoSize(true);
        }

        $filename = "Route_Margin_Report_" . now()->format('Ymd_His') . ".xlsx";

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename);
    }

    public function routeMarginExportPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $startDate = $request->input('start_date', now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        $routeId = $request->input('route_id');
        $reportType = $request->input('report_type', 'actual');

        $query = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->join('routes', 'patroller_schedules.route_id', '=', 'routes.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereDate('patroller_schedules.scheduled_date', '>=', $startDate)
            ->whereDate('patroller_schedules.scheduled_date', '<=', $endDate)
            ->select(
                'routes.id as route_id',
                'routes.name as route_name',
                'patroller_schedules.route_rate',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'patroller_schedules.duty_number',
                'patroller_schedules.from_time as scheduled_start',
                'patroller_schedules.to_time as scheduled_end',
                DB::raw("COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at) as actual_start"),
                DB::raw("COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at) as actual_end"),
                'employee_patroller_schedule.wage_types',
                'employee_patroller_schedule.wage_rate',
                'employee_patroller_schedule.gas_rate',
                'employee_patroller_schedule.allowance'
            );

        if ($reportType === 'actual') {
            $query->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_start_at, patroller_schedules.job_started_at)'))
                ->whereNotNull(DB::raw('COALESCE(employee_patroller_schedule.actual_end_at, patroller_schedules.job_ended_at)'));
        }

        if ($routeId) {
            $query->where('routes.id', $routeId);
        }

        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $query->whereIn('employee_patroller_schedule.id', $ids);
        }

        $records = $query->orderBy('routes.name')->orderBy('patroller_schedules.scheduled_date')->get();
        $marginData = [];
        foreach ($records as $record) {
            if (!isset($marginData[$record->route_id])) {
                $marginData[$record->route_id] = [
                    'route_name' => $record->route_name,
                    'employees' => [],
                    'total_hours' => 0,
                    'route_total' => 0,
                    'total_cost' => 0,
                    'total_margin' => 0
                ];
            }
            $start = \Carbon\Carbon::parse($reportType === 'scheduled' ? $record->scheduled_start : $record->actual_start);
            $end = \Carbon\Carbon::parse($reportType === 'scheduled' ? $record->scheduled_end : $record->actual_end);
            $hours = $start->diffInMinutes($end) / 60;

            $routeRate = $record->route_rate ?? 0;
            $routeTotal = $routeRate * $hours;

            $totalAmount = 0;
            $wageTypes = json_decode($record->wage_types, true);
            if (is_array($wageTypes) && count($wageTypes) > 0) {
                usort($wageTypes, function ($a, $b) {
                    return (($a['id'] ?? 0) == 1 ? 0 : 1) - (($b['id'] ?? 0) == 1 ? 0 : 1);
                });
                $rem = $hours;
                foreach ($wageTypes as $wt) {
                    if ($rem <= 0)
                        break;
                    $ph = min($rem, $wt['allocated_hours'] ?? 0);
                    $totalAmount += ($wt['rate'] ?? 0) * $ph;
                    $rem -= $ph;
                }
            } else {
                $totalAmount = ($record->wage_rate ?? 0) * $hours;
            }

            $totalCost = $totalAmount + ($record->gas_rate ?? 0) + ($record->allowance ?? 0);
            $margin = $routeTotal - $totalCost;

            $marginData[$record->route_id]['employees'][] = [
                'employee_name' => $record->employee_name,
                'duty_number' => $record->duty_number,
                'hours' => number_format($hours, 2),
                'route_total' => $routeTotal,
                'total_cost' => $totalCost,
                'margin' => $margin
            ];
            $marginData[$record->route_id]['total_hours'] += $hours;
            $marginData[$record->route_id]['route_total'] += $routeTotal;
            $marginData[$record->route_id]['total_cost'] += $totalCost;
            $marginData[$record->route_id]['total_margin'] += $margin;
        }

        $pdf = Pdf::loadView('admin.reports.patroller.route_margin_pdf', compact('marginData', 'startDate', 'endDate'))->setOptions([
            'tempDir' => public_path(),
            'chroot' => public_path(),
            'isHtml5ParserEnabled' => true,
            'isRemoteEnabled' => true,
            'defaultFont' => 'DejaVu Sans',
            'dpi' => 72,
            'defaultCompression' => true
        ]);
        return $pdf->download("Route_Margin_Report_" . now()->format('Ymd_His') . ".pdf");
    }

}
