<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use App\Models\Company;
use Illuminate\Http\Request;

use Barryvdh\DomPDF\Facade\Pdf;

class EmployeeController extends Controller
{
    public function exportPDF(Request $request)
    {
        $query = Employee::query()->with(['company', 'jobRole', 'department', 'wageTypes', 'permanentCountry', 'permanentState']);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%")
                    ->orWhere('employee_id', 'like', "%{$search}%")
                    ->orWhere('phone_number', 'like', "%{$search}%");
            });
        }

        // Filter by Company
        if ($request->filled('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        $employees = $query->latest()->get();

        $headers = [
            'ID',
            'Company',
            'First Name',
            'Middle Name',
            'Last Name',
            'Employee ID',
            'Gender',
            'Email',
            'Phone',
            'Department',
            'Job Role',
            'Status',
            'Zone',
            'DOB',
            'Marital Status',
            'Perm. Address 1',
            'Perm. City',
            'Perm. Country',
            'License Number',
            'License Expiry',
            'SIN',
            'Hired Date',
            'Terminated Date',
            'Reporting Manager',
            'Created At'
        ];

        $data = $employees->map(function ($employee) {
            return [
                $employee->id,
                $employee->company->name ?? 'N/A',
                $employee->first_name,
                $employee->middle_name,
                $employee->last_name,
                $employee->employee_id,
                $employee->gender,
                $employee->email,
                $employee->phone_number,
                $employee->department->name ?? 'N/A',
                $employee->jobRole->name ?? 'N/A',
                $employee->active ? 'Active' : 'Inactive',
                $employee->zone->name ?? 'N/A',
                $employee->dob,
                $employee->marital_status,
                $employee->permanent_address_line_1,
                $employee->permanent_city,
                $employee->permanentCountry->name ?? '',
                $employee->license_number,
                $employee->license_expiry,
                $employee->sin_number,
                $employee->employment_date,
                $employee->terminated_date,
                $employee->reporting_manager,
                $employee->created_at->format('Y-m-d'),
            ];
        });

        $pdf = Pdf::loadView('pdf.sites_all', [
            'title' => 'Admin - Employees List',
            'headers' => $headers,
            'data' => $data
        ])->setPaper('a4', 'landscape');

        return $pdf->download('admin-employees-export-' . now()->format('Y-m-d') . '.pdf');
    }
    public function index(Request $request)
    {
        $query = Employee::query()->with(['company', 'jobRole', 'department']);

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                    ->orWhere('phone_number', 'like', "%{$search}%")
                    ->orWhere('employee_id', 'like', "%{$search}%");
            });
        }

        // Filter by Company
        if ($request->filled('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        // Sorting
        $sortField = $request->get('sort', 'created_at');
        $sortOrder = $request->get('direction', 'desc');

        $allowedSorts = ['first_name', 'email', 'employment_date', 'created_at'];
        if (in_array($sortField, $allowedSorts)) {
            $query->orderBy($sortField, $sortOrder);
        }

        $employees = $query->paginate(100)->withQueryString();
        $companies = Company::withCount(['employees', 'sites'])->where('active', true)->orderBy('name')->get();

        if ($request->ajax()) {
            return view('admin.employees._table', compact('employees'))->render();
        }

        return view('admin.employees.index', compact('employees', 'companies'));
    }

    public function show(Employee $employee)
    {
        $employee->load([
            'company',
            'jobRole',
            'department',
            'skills',
            'identifications',
            'permanentCountry',
            'permanentState',
            'correspondingCountry',
            'correspondingState',
            'licenceIssuingProvinceState',
            'bannedSites'
        ]);
        return view('admin.employees.show', compact('employee'));
    }

    public function destroy(Employee $employee)
    {
        $employee->delete();
        return back()->with('success', 'Employee deleted successfully.');
    }

    public function toggleStatus(Employee $employee)
    {
        $employee->active = !$employee->active;
        $employee->save();

        return response()->json([
            'success' => true,
            'active' => $employee->active,
            'message' => 'Status updated successfully.'
        ]);
    }
}
