<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\EmailCampaign;
use App\Models\EmailTemplate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class EmailCampaignController extends Controller
{
    public function index()
    {
        $company_id = session('selected_company_id');
        $campaigns = EmailCampaign::where('company_id', $company_id)
            ->with('template')
            ->orderByDesc('created_at')
            ->paginate(20);
        return view('admin.email-campaigns.index', compact('campaigns'));
    }

    public function create()
    {
        $company_id = session('selected_company_id');
        $templates = EmailTemplate::where('company_id', $company_id)->get();
        return view('admin.email-campaigns.create', compact('templates'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required_unless:recipient_type,particular|nullable|string|max:255',
            'recipient_type' => 'required|in:clients,employees,particular,custom',
            'email_template_id' => 'required|exists:email_templates,id',
            'custom_file' => 'required_if:recipient_type,custom|file|mimes:xlsx,csv,xls',
            'specific_email' => 'required_if:recipient_type,particular|nullable|email',
            'scheduled_at' => 'nullable|date|after:now',
        ]);

        $company_id = session('selected_company_id');
        $filePath = null;

        if ($request->hasFile('custom_file')) {
            $filePath = $request->file('custom_file')->store('campaigns', 'local');
        }

        $name = $request->name;
        if (!$name && $request->recipient_type === 'particular') {
            $name = 'Single Send: ' . $request->specific_email;
        }

        EmailCampaign::create([
            'company_id' => $company_id,
            'name' => $name,
            'recipient_type' => $request->recipient_type,
            'email_template_id' => $request->email_template_id,
            'custom_file_path' => $filePath,
            'specific_email' => $request->specific_email,
            'status' => $request->filled('scheduled_at') ? 'scheduled' : 'draft',
            'scheduled_at' => $request->scheduled_at,
        ]);

        return redirect()->route('email-campaigns.index')->with('success', 'Campaign created successfully.');
    }

    public function send(EmailCampaign $emailCampaign)
    {
        // We dispatch the job to handle the actual sending via Mailgun
        \App\Jobs\SendEmailCampaignJob::dispatchSync($emailCampaign);

        $emailCampaign->update([
            'status' => 'sent',
            'sent_at' => now(),
        ]);

        return back()->with('success', 'Campaign queued for sending/resending.');
    }

    public function showLogs(EmailCampaign $email_campaign)
    {
        $logs = $email_campaign->logs()->orderByDesc('created_at')->paginate(50);
        return view('admin.email-campaigns.logs', compact('email_campaign', 'logs'));
    }

    public function destroy(EmailCampaign $emailCampaign)
    {
        if ($emailCampaign->custom_file_path) {
            Storage::disk('local')->delete($emailCampaign->custom_file_path);
        }
        $emailCampaign->delete();
        return back()->with('success', 'Campaign deleted.');
    }
}
