<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Company;
use App\Models\TimeZone;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

use Barryvdh\DomPDF\Facade\Pdf;

class CompanyController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search = $request->input('search');
        $query = Company::with('timeZone');

        if ($search) {
            $query->where('name', 'like', "%{$search}%")
                ->orWhere('email', 'like', "%{$search}%")
                ->orWhere('phone', 'like', "%{$search}%");
        }

        $companies = $query->latest()->paginate(100)->withQueryString();
        return view('admin.companies.index', compact('companies', 'search'));
    }

    public function exportPDF()
    {
        $companies = Company::with(['timeZone', 'defaultWageType', 'branches'])->latest()->get();
        $data = [];
        $headers = [
            'ID',
            'Name',
            'Type',
            'Contact Person',
            'CEO',
            'Mobile',
            'Email',
            'Operational Areas',
            'Branches',
            'Service Tax No',
            'HST',
            'Time Zone',
            'Status',
            'Created At'
        ];

        foreach ($companies as $company) {
            $branchesInfo = $company->branches->map(function ($b) {
                return $b->name . ($b->branch_code ? " ({$b->branch_code})" : "");
            })->implode(', ');

            $data[] = [
                $company->id,
                $company->name,
                $company->company_type ?? '-',
                $company->contact_person,
                $company->ceo_name ?? '-',
                $company->mobile_number,
                $company->email,
                $company->operational_areas ?? '-',
                $branchesInfo ?: '-',
                $company->service_tax_number ?? '-',
                $company->hst ?? '-',
                $company->timeZone->name ?? 'N/A',
                $company->active ? 'Active' : 'Inactive',
                $company->created_at->format('Y-m-d')
            ];
        }

        $pdf = Pdf::loadView('admin.exports.companies_pdf_template', [
            'title' => 'Companies List',
            'headers' => $headers,
            'data' => $data
        ])->setPaper('a4', 'landscape');

        return $pdf->download('companies-' . date('Y-m-d') . '.pdf');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $timeZones = TimeZone::where('active', 1)->get();
        return view('admin.companies.create', compact('timeZones'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'time_zone_id' => 'required|exists:time_zones,id',
            'name' => 'required|string|max:255',
            'company_type' => 'nullable|string|max:255',
            'tagline' => 'nullable|string|max:255',
            'contact_person' => 'required|string|max:255',
            'ceo_name' => 'nullable|string|max:255',
            'ceo_email' => 'nullable|email|max:255',
            'ceo_phone' => 'nullable|string|max:20',
            'mobile_number' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'website' => 'nullable|url|max:255',
            'operational_areas' => 'nullable|string',
            'service_tax_number' => 'nullable|string|max:255',
            'hst' => 'nullable|string|max:255',
            'cst_no' => 'nullable|string|max:255',
            'cin_no' => 'nullable|string|max:255',
            'logo' => 'required|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'address_1' => 'required|string',
            'address_2' => 'nullable|string',
            'branches' => 'nullable|array',
            'branches.*.name' => 'required_with:branches|string|max:255',
            'branches.*.branch_code' => 'nullable|string|max:50',
            'branches.*.contact_person' => 'nullable|string|max:255',
            'branches.*.contact_phone' => 'nullable|string|max:20',
            'branches.*.contact_email' => 'nullable|email|max:255',
            'branches.*.address' => 'nullable|string',
        ]);

        if ($request->hasFile('logo')) {
            $path = $request->file('logo')->store('logos', 'public');
            $validated['logo'] = $path;
        }

        $company = Company::create($validated);

        if ($request->has('branches')) {
            foreach ($request->branches as $branchData) {
                if (!empty($branchData['name'])) {
                    $company->branches()->create($branchData);
                }
            }
        }

        return redirect()->route('admin.companies.index')
            ->with('success', 'Company created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Company $company)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Company $company)
    {
        $company->load('branches');
        $timeZones = TimeZone::where('active', 1)->get();
        return view('admin.companies.edit', compact('company', 'timeZones'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Company $company)
    {
        $validated = $request->validate([
            'time_zone_id' => 'required|exists:time_zones,id',
            'name' => 'required|string|max:255',
            'company_type' => 'nullable|string|max:255',
            'tagline' => 'nullable|string|max:255',
            'contact_person' => 'required|string|max:255',
            'ceo_name' => 'nullable|string|max:255',
            'ceo_email' => 'nullable|email|max:255',
            'ceo_phone' => 'nullable|string|max:20',
            'mobile_number' => 'required|string|max:20',
            'email' => 'required|email|max:255',
            'website' => 'nullable|url|max:255',
            'operational_areas' => 'nullable|string',
            'service_tax_number' => 'nullable|string|max:255',
            'hst' => 'nullable|string|max:255',
            'cst_no' => 'nullable|string|max:255',
            'cin_no' => 'nullable|string|max:255',
            'logo' => 'nullable|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'address_1' => 'required|string',
            'address_2' => 'nullable|string',
            'branches' => 'nullable|array',
            'branches.*.name' => 'required_with:branches|string|max:255',
            'branches.*.branch_code' => 'nullable|string|max:50',
            'branches.*.contact_person' => 'nullable|string|max:255',
            'branches.*.contact_phone' => 'nullable|string|max:20',
            'branches.*.contact_email' => 'nullable|email|max:255',
            'branches.*.address' => 'nullable|string',
        ]);

        if ($request->hasFile('logo')) {
            // Delete old logo
            if ($company->logo) {
                Storage::disk('public')->delete($company->logo);
            }
            $path = $request->file('logo')->store('logos', 'public');
            $validated['logo'] = $path;
        }

        $company->update($validated);

        if ($request->has('branches')) {
            $company->branches()->delete();
            foreach ($request->branches as $branchData) {
                if (!empty($branchData['name'])) {
                    $company->branches()->create($branchData);
                }
            }
        }

        return redirect()->route('admin.companies.index')
            ->with('success', 'Company updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Company $company)
    {
        if ($company->logo) {
            Storage::disk('public')->delete($company->logo);
        }
        $company->delete();

        return redirect()->route('admin.companies.index')
            ->with('success', 'Company deleted successfully.');
    }

    public function toggleStatus(Company $company)
    {
        $company->update(['active' => !$company->active]);
        return response()->json(['success' => true, 'active' => (bool) $company->active]);
    }
}
