<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AttendanceController extends Controller
{
    /**
     * Display employee attendance chart
     */
    public function chart(Request $request)
    {
        $company_id = session('selected_company_id');
        $month = $request->input('month', now()->month);
        $year = $request->input('year', now()->year);
        $employeeId = $request->input('employee_id');

        // Get all employees for filter
        $employees = Employee::where('company_id', $company_id)
            ->orderBy('first_name')
            ->get();

        // Calculate days in month
        $daysInMonth = Carbon::create($year, $month, 1)->daysInMonth;
        $daysArray = range(1, $daysInMonth);

        // Build query for guard schedules
        $guardQuery = DB::table('employee_schedule')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->where('schedules.company_id', $company_id)
            ->whereYear('schedules.schedule_date', $year)
            ->whereMonth('schedules.schedule_date', $month)
            ->select(
                'employees.id as employee_id',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                DB::raw("DAY(schedules.schedule_date) as day"),
                'employee_schedule.actual_start_at',
                'employee_schedule.actual_end_at',
                'schedules.status'
            );

        // Build query for patroller schedules
        $patrollerQuery = DB::table('employee_patroller_schedule')
            ->join('patroller_schedules', 'employee_patroller_schedule.patroller_schedule_id', '=', 'patroller_schedules.id')
            ->join('employees', 'employee_patroller_schedule.employee_id', '=', 'employees.id')
            ->where('patroller_schedules.company_id', $company_id)
            ->whereYear('patroller_schedules.from_time', $year)
            ->whereMonth('patroller_schedules.from_time', $month)
            ->select(
                'employees.id as employee_id',
                DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                DB::raw("DAY(patroller_schedules.from_time) as day"),
                'employee_patroller_schedule.actual_start_at',
                'employee_patroller_schedule.actual_end_at',
                'patroller_schedules.job_status as status'
            );

        // Apply employee filter if specified
        if ($employeeId) {
            $guardQuery->where('employees.id', $employeeId);
            $patrollerQuery->where('employees.id', $employeeId);
        }

        // Get both guard and patroller records
        $guardRecords = $guardQuery->get();
        $patrollerRecords = $patrollerQuery->get();

        // Merge both records
        $allRecords = $guardRecords->concat($patrollerRecords);

        // Process attendance data
        $attendanceData = [];

        foreach ($allRecords as $record) {
            if (!isset($attendanceData[$record->employee_id])) {
                $attendanceData[$record->employee_id] = [
                    'name' => $record->employee_name,
                    'days' => [],
                    'present_count' => 0,
                    'absent_count' => 0,
                    'total_jobs' => 0
                ];
            }

            $day = $record->day;

            // Determine attendance status
            // Present: has actual_start_at OR status is 'completed' or 'active'
            // Absent: has a job but no actual_start_at AND status is not 'completed' or 'active'

            $isPresent = false;
            $isAbsent = false;

            // Create Carbon object for the record's date
            $recordDate = Carbon::createFromDate($year, $month, $day)->startOfDay();
            $now = Carbon::today();

            if ($record->actual_start_at) {
                $isPresent = true;
            } elseif (in_array($record->status, ['completed', 'active', 'in_progress', 'missed'])) {
                // If the job is active or finished but this individual has no start time, they are absent
                $isAbsent = true;
            } elseif ($recordDate->lt($now)) {
                // If the schedule date is in the past and they haven't started, they are absent
                // This covers cases where the status status cron hasn't run yet (still 'upcoming' or 'today')
                $isAbsent = true;
            }

            // Only count if not already marked for this day
            if (!isset($attendanceData[$record->employee_id]['days'][$day])) {
                $attendanceData[$record->employee_id]['total_jobs']++;

                if ($isPresent) {
                    $attendanceData[$record->employee_id]['days'][$day] = 'present';
                    $attendanceData[$record->employee_id]['present_count']++;
                } elseif ($isAbsent) {
                    $attendanceData[$record->employee_id]['days'][$day] = 'absent';
                    $attendanceData[$record->employee_id]['absent_count']++;
                }
            } else {
                // If already marked as absent but this record shows present, update it
                if ($isPresent && $attendanceData[$record->employee_id]['days'][$day] === 'absent') {
                    $attendanceData[$record->employee_id]['days'][$day] = 'present';
                    $attendanceData[$record->employee_id]['present_count']++;
                    $attendanceData[$record->employee_id]['absent_count']--;
                }
            }
        }

        return view('admin.attendance.chart', compact('attendanceData', 'employees', 'month', 'year'))->with('daysInMonth', $daysArray);
    }
}
