<?php

namespace App\Http\Controllers\Webhook;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class MailgunWebhookController extends Controller
{
    public function handle(Request $request)
    {
        $data = $request->all();

        // 1. Verify Signature (Mailgun security)
        if (!$this->verifySignature($request)) {
            return response()->json(['message' => 'Invalid signature'], 401);
        }

        $eventData = $data['event-data'] ?? [];
        $event = $eventData['event'] ?? '';
        $messageId = $eventData['message']['headers']['message-id'] ?? null;

        if (!$messageId) {
            return response()->json(['message' => 'No message-id found'], 200);
        }

        // Search for the log entry
        $log = \App\Models\EmailCampaignLog::where('message_id', '<', '%' . $messageId . '%')
            ->orWhere('message_id', $messageId)
            ->first();

        // Fallback: Mailgun sometimes wraps IDs in brackets
        if (!$log) {
            $strippedId = trim($messageId, '<>');
            $log = \App\Models\EmailCampaignLog::where('message_id', 'like', '%' . $strippedId . '%')->first();
        }

        if ($log) {
            if ($event === 'delivered') {
                $log->update(['status' => 'delivered']);
            } elseif ($event === 'permanent_fail' || $event === 'failed') {
                $status = ($eventData['severity'] ?? '') === 'permanent' ? 'bounced' : 'failed';
                $reason = $eventData['delivery-status']['description'] ?? ($eventData['delivery-status']['message'] ?? 'Unknown failure');
                $log->update([
                    'status' => $status,
                    'error' => $reason
                ]);
            }
        }

        return response()->json(['message' => 'Processed'], 200);
    }

    private function verifySignature(Request $request)
    {
        $apiKey = config('services.mailgun.secret');
        $timestamp = $request->input('signature.timestamp');
        $token = $request->input('signature.token');
        $signature = $request->input('signature.signature');

        if (!$timestamp || !$token || !$signature)
            return false;

        $hash = hash_hmac('sha256', $timestamp . $token, $apiKey);
        return hash_equals($hash, $signature);
    }
}
