<?php

namespace App\Http\Controllers;

use App\Models\Site;
use App\Models\Country;
use App\Models\State;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel; // If using imported package, but code used IOFactory
use PhpOffice\PhpSpreadsheet\IOFactory;

class SiteImportController extends Controller
{
    public function showImportSettings()
    {
        return view('sites.import-settings');
    }

    public function downloadTemplate()
    {
        $headers = [
            'Site ID',
            'Name',
            'Address Line 1',
            'Address Line 2',
            'City',
            'State',
            'Country',
            'Zip Code',
            'Latitude',
            'Longitude'
        ];

        $callback = function () use ($headers) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $headers);
            fclose($file);
        };

        return response()->stream($callback, 200, [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=sites_import_template.csv",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ]);
    }

    public function previewCsv(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'csv_file' => 'required|file|mimes:csv,txt,xlsx,xls|max:10240',
        ]);

        if ($validator->fails()) {
            return redirect()->route('sites.import.settings')
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $file = $request->file('csv_file');
            $extension = $file->getClientOriginalExtension();
            $tempPath = $file->store('temp_imports');

            $previewData = [];
            $csvHeaders = [];

            if (in_array(strtolower($extension), ['xlsx', 'xls'])) {
                $spreadsheet = IOFactory::load(storage_path('app/' . $tempPath));
                $worksheet = $spreadsheet->getActiveSheet();
                $rows = $worksheet->toArray();

                if (count($rows) > 0) {
                    $csvHeaders = array_map('trim', $rows[0]);
                    $previewData = array_slice($rows, 0, 6); // Header + 5 rows
                }
            } else {
                $handle = fopen(storage_path('app/' . $tempPath), 'r');
                if ($handle !== false) {
                    // Read header
                    $headerRow = fgetcsv($handle);
                    if ($headerRow !== false) {
                        // Handle BOM
                        if (count($headerRow) > 0) {
                            $headerRow[0] = preg_replace('/[\x00-\x1F\x80-\xFF]/', '', $headerRow[0]);
                        }
                        $csvHeaders = array_map('trim', $headerRow);

                        $previewData[] = $csvHeaders;

                        // Read 5 preview rows
                        for ($i = 0; $i < 5; $i++) {
                            $row = fgetcsv($handle);
                            if ($row === false)
                                break;
                            $previewData[] = $row;
                        }
                    }
                    fclose($handle);
                }
            }

            $siteColumns = $this->getSiteColumns();

            return view('sites.import-mapping', compact(
                'csvHeaders',
                'previewData',
                'siteColumns',
                'tempPath'
            ));

        } catch (\Exception $e) {
            Log::error('Site Import Preview Error: ' . $e->getMessage());
            return redirect()->route('sites.import.settings')
                ->with('error', 'Error reading file: ' . $e->getMessage());
        }
    }

    public function processImport(Request $request)
    {
        $request->validate([
            'temp_file_path' => 'required|string',
            'mapping' => 'required|array'
        ]);

        $company_id = session('company_id') ?? Company::first()->id; // Default or session
        $filePath = storage_path('app/' . $request->temp_file_path);

        if (!file_exists($filePath)) {
            return redirect()->route('sites.import.settings')->with('error', 'Temporary file expired. Please upload again.');
        }

        $mapping = $request->mapping;
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $importedCount = 0;
        $errors = [];

        try {
            DB::beginTransaction();

            $rows = [];
            if (in_array(strtolower($extension), ['xlsx', 'xls'])) {
                $spreadsheet = IOFactory::load($filePath);
                $worksheet = $spreadsheet->getActiveSheet();
                $rows = $worksheet->toArray();
                array_shift($rows); // Remove header
            } else {
                $rows = array_map('str_getcsv', file($filePath));
                array_shift($rows); // Remove header
            }

            foreach ($rows as $index => $row) {
                // Skip empty rows
                if (empty(array_filter($row)))
                    continue;

                try {
                    $siteData = [
                        'company_id' => $company_id,
                        'active' => 1
                    ];

                    foreach ($mapping as $dbField => $csvIndex) {
                        if ($csvIndex !== null && $csvIndex !== '' && isset($row[$csvIndex])) {
                            $val = trim($row[$csvIndex]);

                            // Transform fields
                            if ($dbField === 'country' || $dbField === 'country_id') {
                                // Resolve Country
                                $country = Country::where('name', 'LIKE', $val)
                                    ->orWhere('iso2', 'LIKE', $val)
                                    ->orWhere('iso3', 'LIKE', $val)
                                    ->first();
                                $siteData['country_id'] = $country ? $country->id : null;
                            } elseif ($dbField === 'state' || $dbField === 'state_id') {
                                // Store state name temp, resolve later if possible
                                // Resolving state usually requires Country ID.
                                // We'll try to find by name global or filtered if country known
                                $stateQuery = State::where('name', 'LIKE', $val)->orWhere('code', 'LIKE', $val);
                                if (isset($siteData['country_id'])) {
                                    $stateQuery->where('country_id', $siteData['country_id']);
                                }
                                $state = $stateQuery->first();
                                $siteData['state_id'] = $state ? $state->id : null;
                            } else {
                                $siteData[$dbField] = $val;
                            }
                        }
                    }

                    // Validation: Name is required
                    if (empty($siteData['name'])) {
                        throw new \Exception("Site Name is required");
                    }
                    if (empty($siteData['site_id'])) {
                        // Generate site_id if missing? Or error?
                        // Let's assume user provides it, or we generate slug
                        $siteData['site_id'] = \Illuminate\Support\Str::slug($siteData['name']) . '-' . rand(1000, 9999);
                    }

                    Site::updateOrCreate(
                        ['site_id' => $siteData['site_id'], 'company_id' => $company_id],
                        $siteData
                    );

                    $importedCount++;

                } catch (\Exception $e) {
                    $errors[] = "Row " . ($index + 2) . ": " . $e->getMessage();
                }
            }

            DB::commit();
            @unlink($filePath);

            return redirect()->route('sites.index')->with('success', "Imported $importedCount sites successfully." . (count($errors) > 0 ? " " . count($errors) . " errors occurred." : ""));

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('sites.import.settings')->with('error', 'Import Process Failed: ' . $e->getMessage());
        }
    }

    private function getSiteColumns()
    {
        return [
            [
                'name' => 'site_id',
                'label' => 'Site ID / Code',
                'required' => true,
                'category' => 'Identification'
            ],
            [
                'name' => 'name',
                'label' => 'Site Name',
                'required' => true,
                'category' => 'Identification'
            ],
            [
                'name' => 'address_line_1',
                'label' => 'Address Line 1',
                'required' => false,
                'category' => 'Location'
            ],
            [
                'name' => 'address_line_2',
                'label' => 'Address Line 2',
                'required' => false,
                'category' => 'Location'
            ],
            [
                'name' => 'city',
                'label' => 'City',
                'required' => false,
                'category' => 'Location'
            ],
            [
                'name' => 'state', // Mapped to state_id
                'label' => 'State / Province',
                'required' => false,
                'category' => 'Location'
            ],
            [
                'name' => 'country', // Mapped to country_id
                'label' => 'Country',
                'required' => false,
                'category' => 'Location'
            ],
            [
                'name' => 'zip_code',
                'label' => 'Zip / Postal Code',
                'required' => false,
                'category' => 'Location'
            ],
            [
                'name' => 'latitude',
                'label' => 'Latitude',
                'required' => false,
                'category' => 'Coordinates'
            ],
            [
                'name' => 'longitude',
                'label' => 'Longitude',
                'required' => false,
                'category' => 'Coordinates'
            ],
        ];
    }
}
