<?php

namespace App\Http\Controllers;

use App\Models\GuardCheckin;
use App\Models\Schedule;
use App\Models\PatrollerSchedule;
use App\Models\Employee;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class GuardCheckinController extends Controller
{
    /**
     * Display a list of currently active guards (Static & Patroller).
     */
    public function activeGuards()
    {
        $companyId = session('selected_company_id');

        // 1. Static Schedules (Active)
        $staticSchedules = Schedule::where('company_id', $companyId)
            ->where('status', 'active')
            ->with(['site', 'checkins', 'employees'])
            ->get();

        // 2. Patroller Schedules (In Progress)
        $patrollerSchedules = PatrollerSchedule::where('company_id', $companyId)
            ->where('job_status', 'in_progress')
            ->with(['route', 'checkins', 'employees'])
            ->get();

        // 3. Transform into a unified collection
        $activeGuards = collect();
        $now = Carbon::now();

        foreach ($staticSchedules as $schedule) {
            foreach ($schedule->employees as $employee) {
                // Calculate Shift Metrics
                $startTime = Carbon::parse($schedule->from_datetime);
                $endTime = Carbon::parse($schedule->to_datetime);
                $totalMinutes = $startTime->diffInMinutes($endTime);
                $totalHours = round($totalMinutes / 60, 1);

                // Interval Logic
                // 3 Equal check-ins implies dividing the shift into 3 parts
                // E.g. 6 hours: checkpoints at +2h, +4h, +6h
                // Or maybe mid-points. Let's stick to simple distribution: Start + (Duration/3)*i

                $intervalMinutes = $totalMinutes / 3;
                $checkpoints = collect();
                for ($i = 1; $i <= 3; $i++) {
                    $checkpoints->push($startTime->copy()->addMinutes($intervalMinutes * $i));
                }

                // Get Last Check-in for THIS employee on THIS schedule
                $lastCheckin = $schedule->checkins
                    ->where('employee_id', $employee->id)
                    ->sortByDesc('created_at')
                    ->first();

                // Calculate Next Call Time
                // Find method: Find the first checkpoint that is in the future relative to NOW
                // If all are past, show "Shift End" or similar?
                // Alternatively, find first checkpoint > last_checkin_time (if exists) OR > startTime (if no checkin)

                $nextCall = null;
                $referenceTime = $lastCheckin ? $lastCheckin->created_at : $startTime;

                // If we want "Next Upcoming Call based on Time":
                // $nextCall = $checkpoints->first(function($point) use ($now) {
                //     return $point->gt($now);
                // });

                // If we want "Next Scheduled Call in Sequence":
                // Filter points that are greater than reference time
                $nextCall = $checkpoints->first(function ($point) use ($referenceTime) {
                    return $point->gt($referenceTime); // strict inequality to advance
                });

                // If no next call implies shift is effectively fully checked or over
                if (!$nextCall && $endTime->gt($now)) {
                    $nextCall = $endTime; // Fallback to end of shift
                }

                $activeGuards->push((object) [
                    'type' => 'static',
                    'job_id' => $schedule->id,
                    'reference_number' => $schedule->duty_number,
                    'site_name' => $schedule->site->name,
                    'schedule_time' => $startTime->format('H:i') . ' - ' . $endTime->format('H:i'),
                    'total_hours' => $totalHours,
                    'employee' => $employee,
                    'phone' => $employee->phone_number,
                    'checkin_time' => $employee->pivot->actual_start_at ? Carbon::parse($employee->pivot->actual_start_at)->format('H:i') : 'N/A',
                    'last_call' => $lastCheckin ? $lastCheckin->created_at->format('H:i') : 'Never',
                    'last_call_relative' => $lastCheckin ? $lastCheckin->created_at->diffForHumans() : null,
                    'next_call' => $nextCall ? $nextCall->format('H:i') : 'Shift End',
                    'next_call_is_overdue' => $nextCall && $nextCall->lt($now),
                ]);
            }
        }

        foreach ($patrollerSchedules as $patrol) {
            foreach ($patrol->employees as $employee) {
                $startTime = Carbon::parse($patrol->from_time);
                $endTime = Carbon::parse($patrol->to_time);
                $totalMinutes = $startTime->diffInMinutes($endTime);
                $totalHours = round($totalMinutes / 60, 1);

                $intervalMinutes = $totalMinutes / 3;
                $checkpoints = collect();
                for ($i = 1; $i <= 3; $i++) {
                    $checkpoints->push($startTime->copy()->addMinutes($intervalMinutes * $i));
                }

                $lastCheckin = $patrol->checkins
                    ->where('employee_id', $employee->id)
                    ->sortByDesc('created_at')
                    ->first();

                $referenceTime = $lastCheckin ? $lastCheckin->created_at : $startTime;
                $nextCall = $checkpoints->first(function ($point) use ($referenceTime) {
                    return $point->gt($referenceTime);
                });

                if (!$nextCall && $endTime->gt($now)) {
                    $nextCall = $endTime;
                }


                $activeGuards->push((object) [
                    'type' => 'patrol',
                    'job_id' => $patrol->id,
                    'reference_number' => 'PAT-' . $patrol->id,
                    'site_name' => $patrol->route->name,
                    'schedule_time' => $startTime->format('H:i') . ' - ' . $endTime->format('H:i'),
                    'total_hours' => $totalHours,
                    'employee' => $employee,
                    'phone' => $employee->phone_number,
                    'checkin_time' => $employee->pivot->actual_start_at ? Carbon::parse($employee->pivot->actual_start_at)->format('H:i') : 'N/A',
                    'last_call' => $lastCheckin ? $lastCheckin->created_at->format('H:i') : 'Never',
                    'last_call_relative' => $lastCheckin ? $lastCheckin->created_at->diffForHumans() : null,
                    'next_call' => $nextCall ? $nextCall->format('H:i') : 'Shift End',
                    'next_call_is_overdue' => $nextCall && $nextCall->lt($now),
                ]);
            }
        }

        return view('care.active-guards', compact('activeGuards'));
    }

    /**
     * Store a verified check-in call log.
     */
    public function store(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'type' => 'required|in:static,patrol',
            'job_id' => 'required|integer',
            'location_reported' => 'nullable|string',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'maps_link' => 'nullable|url',
            'notes' => 'nullable|string',
            'images.*' => 'nullable|image|max:10240', // 10MB max
            'checkin_type' => 'required|in:call,video,visit',
        ]);

        $companyId = session('selected_company_id');
        $userId = auth()->id();

        $imagePaths = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('checkins/' . date('Y-m-d'), 'public');
                $imagePaths[] = $path;
            }
        }

        $data = [
            'company_id' => $companyId,
            'user_id' => $userId,
            'employee_id' => $request->employee_id,
            'checkin_type' => $request->checkin_type,
            'location_reported' => $request->location_reported,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'maps_link' => $request->maps_link,
            'is_verified' => $request->has('is_verified'),
            'notes' => $request->notes,
            'images' => $imagePaths,
        ];

        if ($request->type === 'static') {
            $data['schedule_id'] = $request->job_id;
        } else {
            $data['patroller_schedule_id'] = $request->job_id;
        }

        GuardCheckin::create($data);

        return back()->with('success', 'Check-in logged successfully.');
    }

    /**
     * Display a history of check-ins.
     */
    public function index(Request $request)
    {
        $companyId = session('selected_company_id');

        $query = GuardCheckin::where('company_id', $companyId)
            ->with(['user', 'employee', 'schedule.site', 'patrollerSchedule.route'])
            ->latest();

        // Filters
        if ($request->filled('date')) {
            $query->whereDate('created_at', $request->date);
        }

        if ($request->filled('employee_id')) {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->filled('status')) {
            if ($request->status === 'verified') {
                $query->where('is_verified', true);
            } elseif ($request->status === 'unverified') {
                $query->where('is_verified', false);
            }
        }

        $checkins = $query->paginate(20);
        $employees = Employee::where('company_id', $companyId)->orderBy('first_name')->get();

        return view('care.history', compact('checkins', 'employees'));
    }

    /**
     * Export active guards to PDF.
     */
    public function exportPdf()
    {
        $companyId = session('selected_company_id');
        $company = \App\Models\Company::find($companyId);

        // Reuse the same logic from activeGuards()
        $staticSchedules = Schedule::where('company_id', $companyId)
            ->where('status', 'active')
            ->with(['site', 'checkins', 'employees'])
            ->get();

        $patrollerSchedules = PatrollerSchedule::where('company_id', $companyId)
            ->where('job_status', 'in_progress')
            ->with(['route', 'checkins', 'employees'])
            ->get();

        $activeGuards = collect();
        $now = Carbon::now();

        foreach ($staticSchedules as $schedule) {
            foreach ($schedule->employees as $employee) {
                $startTime = Carbon::parse($schedule->from_datetime);
                $endTime = Carbon::parse($schedule->to_datetime);
                $totalMinutes = $startTime->diffInMinutes($endTime);
                $totalHours = round($totalMinutes / 60, 1);

                $intervalMinutes = $totalMinutes / 3;
                $checkpoints = collect();
                for ($i = 1; $i <= 3; $i++) {
                    $checkpoints->push($startTime->copy()->addMinutes($intervalMinutes * $i));
                }

                $lastCheckin = $schedule->checkins
                    ->where('employee_id', $employee->id)
                    ->sortByDesc('created_at')
                    ->first();

                $referenceTime = $lastCheckin ? $lastCheckin->created_at : $startTime;
                $nextCall = $checkpoints->first(function ($point) use ($referenceTime) {
                    return $point->gt($referenceTime);
                });

                if (!$nextCall && $endTime->gt($now)) {
                    $nextCall = $endTime;
                }

                $activeGuards->push((object) [
                    'type' => 'static',
                    'job_id' => $schedule->id,
                    'reference_number' => $schedule->duty_number,
                    'site_name' => $schedule->site->name,
                    'schedule_time' => $startTime->format('H:i') . ' - ' . $endTime->format('H:i'),
                    'total_hours' => $totalHours,
                    'employee' => $employee,
                    'phone' => $employee->phone_number,
                    'checkin_time' => $employee->pivot->actual_start_at ? Carbon::parse($employee->pivot->actual_start_at)->format('H:i') : 'N/A',
                    'last_call' => $lastCheckin ? $lastCheckin->created_at->format('H:i') : 'Never',
                    'last_call_relative' => $lastCheckin ? $lastCheckin->created_at->diffForHumans() : null,
                    'next_call' => $nextCall ? $nextCall->format('H:i') : 'Shift End',
                    'next_call_is_overdue' => $nextCall && $nextCall->lt($now),
                ]);
            }
        }

        foreach ($patrollerSchedules as $patrol) {
            foreach ($patrol->employees as $employee) {
                $startTime = Carbon::parse($patrol->from_time);
                $endTime = Carbon::parse($patrol->to_time);
                $totalMinutes = $startTime->diffInMinutes($endTime);
                $totalHours = round($totalMinutes / 60, 1);

                $intervalMinutes = $totalMinutes / 3;
                $checkpoints = collect();
                for ($i = 1; $i <= 3; $i++) {
                    $checkpoints->push($startTime->copy()->addMinutes($intervalMinutes * $i));
                }

                $lastCheckin = $patrol->checkins
                    ->where('employee_id', $employee->id)
                    ->sortByDesc('created_at')
                    ->first();

                $referenceTime = $lastCheckin ? $lastCheckin->created_at : $startTime;
                $nextCall = $checkpoints->first(function ($point) use ($referenceTime) {
                    return $point->gt($referenceTime);
                });

                if (!$nextCall && $endTime->gt($now)) {
                    $nextCall = $endTime;
                }

                $activeGuards->push((object) [
                    'type' => 'patrol',
                    'job_id' => $patrol->id,
                    'reference_number' => 'PAT-' . $patrol->id,
                    'site_name' => $patrol->route->name,
                    'schedule_time' => $startTime->format('H:i') . ' - ' . $endTime->format('H:i'),
                    'total_hours' => $totalHours,
                    'employee' => $employee,
                    'phone' => $employee->phone_number,
                    'checkin_time' => $employee->pivot->actual_start_at ? Carbon::parse($employee->pivot->actual_start_at)->format('H:i') : 'N/A',
                    'last_call' => $lastCheckin ? $lastCheckin->created_at->format('H:i') : 'Never',
                    'last_call_relative' => $lastCheckin ? $lastCheckin->created_at->diffForHumans() : null,
                    'next_call' => $nextCall ? $nextCall->format('H:i') : 'Shift End',
                    'next_call_is_overdue' => $nextCall && $nextCall->lt($now),
                ]);
            }
        }

        $pdf = \PDF::loadView('care.active-guards-pdf', compact('activeGuards', 'company', 'now'));
        return $pdf->download('active-guards-' . now()->format('Y-m-d-His') . '.pdf');
    }

    /**
     * Export check-in history to PDF.
     */
    public function exportHistoryPdf(Request $request)
    {
        $companyId = session('selected_company_id');
        $company = \App\Models\Company::find($companyId);

        $query = GuardCheckin::where('company_id', $companyId)
            ->with(['user', 'employee', 'schedule.site', 'patrollerSchedule.route'])
            ->latest();

        // Apply same filters as index
        if ($request->filled('date')) {
            $query->whereDate('created_at', $request->date);
        }

        if ($request->filled('employee_id')) {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->filled('status')) {
            if ($request->status === 'verified') {
                $query->where('is_verified', true);
            } elseif ($request->status === 'unverified') {
                $query->where('is_verified', false);
            }
        }

        $checkins = $query->get();
        $filters = $request->only(['date', 'employee_id', 'status']);

        $pdf = \PDF::loadView('care.history-pdf', compact('checkins', 'company', 'filters'));
        return $pdf->download('checkin-history-' . now()->format('Y-m-d-His') . '.pdf');
    }

    /**
     * Verify a check-in.
     */
    public function verify(GuardCheckin $checkin)
    {
        $checkin->update([
            'is_verified' => true
        ]);

        return back()->with('success', 'Check-in verified successfully.');
    }
}
