<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\Zone;
use App\Models\JobRole;
use App\Models\Department;
use App\Models\Country;
use App\Models\State;
use App\Models\WageType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use PhpOffice\PhpSpreadsheet\IOFactory;

class EmployeeImportController extends Controller
{
    /**
     * Show the import settings page with column mapping
     */
    public function showImportSettings()
    {
        $company_id = session('selected_company_id');

        // Get all employee table columns
        $employeeColumns = $this->getEmployeeColumns();

        // Get reference data for dropdowns
        $zones = Zone::all();
        $jobRoles = JobRole::where('active', true)->get();
        $departments = Department::where('active', true)->get();
        $countries = Country::where('active', true)->get();
        $states = State::where('active', true)->get();
        $wageTypes = WageType::where('active', true)->get();

        return view('employees.import-settings', compact(
            'employeeColumns',
            'zones',
            'jobRoles',
            'departments',
            'countries',
            'states',
            'wageTypes'
        ));
    }

    /**
     * Get all employee table columns with their metadata
     */
    private function getEmployeeColumns()
    {
        return [
            // Basic Information
            [
                'name' => 'company_id',
                'label' => 'Company ID',
                'type' => 'foreign_key',
                'reference' => 'companies',
                'required' => false,
                'category' => 'Basic Information',
                'note' => 'Defaults to current session company if not provided'
            ],
            [
                'name' => 'employee_id',
                'label' => 'Employee ID',
                'type' => 'string',
                'required' => true,
                'unique' => true,
                'category' => 'Basic Information'
            ],
            [
                'name' => 'first_name',
                'label' => 'First Name',
                'type' => 'string',
                'required' => true,
                'category' => 'Basic Information'
            ],
            [
                'name' => 'middle_name',
                'label' => 'Middle Name',
                'type' => 'string',
                'required' => false,
                'category' => 'Basic Information'
            ],
            [
                'name' => 'last_name',
                'label' => 'Last Name',
                'type' => 'string',
                'required' => false,
                'category' => 'Basic Information'
            ],
            [
                'name' => 'gender',
                'label' => 'Gender',
                'type' => 'select',
                'options' => ['Male', 'Female', 'Other'],
                'required' => true,
                'category' => 'Basic Information'
            ],
            [
                'name' => 'email',
                'label' => 'Email',
                'type' => 'email',
                'required' => true,
                'unique' => true,
                'category' => 'Basic Information'
            ],
            [
                'name' => 'dob',
                'label' => 'Date of Birth',
                'type' => 'date',
                'required' => false,
                'category' => 'Basic Information'
            ],
            [
                'name' => 'marital_status',
                'label' => 'Marital Status',
                'type' => 'string',
                'required' => false,
                'category' => 'Basic Information'
            ],

            // Job & Location
            [
                'name' => 'zone_id',
                'label' => 'Zone',
                'type' => 'foreign_key',
                'reference' => 'zones',
                'required' => false,
                'category' => 'Job & Location'
            ],
            [
                'name' => 'job_role_id',
                'label' => 'Job Role',
                'type' => 'foreign_key',
                'reference' => 'job_roles',
                'required' => false,
                'category' => 'Job & Location'
            ],
            [
                'name' => 'department_id',
                'label' => 'Department',
                'type' => 'foreign_key',
                'reference' => 'departments',
                'required' => false,
                'category' => 'Job & Location'
            ],

            // Contact Information
            [
                'name' => 'phone_number',
                'label' => 'Phone Number',
                'type' => 'string',
                'required' => true,
                'category' => 'Contact Information'
            ],
            [
                'name' => 'cell_number',
                'label' => 'Cell Number',
                'type' => 'string',
                'required' => false,
                'category' => 'Contact Information'
            ],
            [
                'name' => 'emergency_number',
                'label' => 'Emergency Number',
                'type' => 'string',
                'required' => false,
                'category' => 'Contact Information'
            ],

            // Permanent Address
            [
                'name' => 'permanent_address_line_1',
                'label' => 'Permanent Address Line 1',
                'type' => 'string',
                'required' => false,
                'category' => 'Permanent Address'
            ],
            [
                'name' => 'permanent_address_line_2',
                'label' => 'Permanent Address Line 2',
                'type' => 'string',
                'required' => false,
                'category' => 'Permanent Address'
            ],
            [
                'name' => 'permanent_city',
                'label' => 'Permanent City',
                'type' => 'string',
                'required' => false,
                'category' => 'Permanent Address'
            ],
            [
                'name' => 'permanent_state_id',
                'label' => 'Permanent State',
                'type' => 'foreign_key',
                'reference' => 'states',
                'required' => false,
                'category' => 'Permanent Address'
            ],
            [
                'name' => 'permanent_country_id',
                'label' => 'Permanent Country',
                'type' => 'foreign_key',
                'reference' => 'countries',
                'required' => false,
                'category' => 'Permanent Address'
            ],
            [
                'name' => 'permanent_zip_code',
                'label' => 'Permanent Zip Code',
                'type' => 'string',
                'required' => false,
                'category' => 'Permanent Address'
            ],

            // Corresponding Address
            [
                'name' => 'corresponding_address_line_1',
                'label' => 'Corresponding Address Line 1',
                'type' => 'string',
                'required' => false,
                'category' => 'Corresponding Address'
            ],
            [
                'name' => 'corresponding_address_line_2',
                'label' => 'Corresponding Address Line 2',
                'type' => 'string',
                'required' => false,
                'category' => 'Corresponding Address'
            ],
            [
                'name' => 'corresponding_city',
                'label' => 'Corresponding City',
                'type' => 'string',
                'required' => false,
                'category' => 'Corresponding Address'
            ],
            [
                'name' => 'corresponding_state_id',
                'label' => 'Corresponding State',
                'type' => 'foreign_key',
                'reference' => 'states',
                'required' => false,
                'category' => 'Corresponding Address'
            ],
            [
                'name' => 'corresponding_country_id',
                'label' => 'Corresponding Country',
                'type' => 'foreign_key',
                'reference' => 'countries',
                'required' => false,
                'category' => 'Corresponding Address'
            ],
            [
                'name' => 'corresponding_zip_code',
                'label' => 'Corresponding Zip Code',
                'type' => 'string',
                'required' => false,
                'category' => 'Corresponding Address'
            ],

            // License Information
            [
                'name' => 'licence_issuing_province_id',
                'label' => 'License Issuing Province',
                'type' => 'foreign_key',
                'reference' => 'states',
                'required' => false,
                'category' => 'License Information'
            ],
            [
                'name' => 'license_number',
                'label' => 'License Number',
                'type' => 'string',
                'required' => false,
                'category' => 'License Information'
            ],
            [
                'name' => 'license_expiry',
                'label' => 'License Expiry Date',
                'type' => 'date',
                'required' => false,
                'category' => 'License Information'
            ],
            [
                'name' => 'licence_province_text',
                'label' => 'License Province Text',
                'type' => 'string',
                'required' => false,
                'category' => 'License Information'
            ],

            // Other Information
            [
                'name' => 'reporting_manager',
                'label' => 'Reporting Manager',
                'type' => 'string',
                'required' => false,
                'category' => 'Other Information'
            ],
            [
                'name' => 'sin_number',
                'label' => 'SIN Number',
                'type' => 'string',
                'required' => false,
                'category' => 'Other Information'
            ],
            [
                'name' => 'additional_remarks',
                'label' => 'Additional Remarks',
                'type' => 'text',
                'required' => false,
                'category' => 'Other Information'
            ],
            [
                'name' => 'tags',
                'label' => 'Tags',
                'type' => 'string',
                'required' => false,
                'category' => 'Other Information'
            ],
            [
                'name' => 'fax',
                'label' => 'Fax',
                'type' => 'string',
                'required' => false,
                'category' => 'Other Information'
            ],
            [
                'name' => 'employment_date',
                'label' => 'Employment Date',
                'type' => 'date',
                'required' => false,
                'category' => 'Other Information'
            ],
            [
                'name' => 'birthday',
                'label' => 'Birthday',
                'type' => 'date',
                'required' => false,
                'category' => 'Other Information'
            ],
            [
                'name' => 'terminated_date',
                'label' => 'Terminated Date',
                'type' => 'date',
                'required' => false,
                'category' => 'Other Information'
            ],
            [
                'name' => 'has_car',
                'label' => 'Has Car',
                'type' => 'boolean',
                'required' => false,
                'category' => 'Other Information'
            ],
            [
                'name' => 'password',
                'label' => 'Password',
                'type' => 'string',
                'required' => false,
                'category' => 'Other Information',
                'note' => 'Will be hashed automatically'
            ],
            [
                'name' => 'active',
                'label' => 'Active Status',
                'type' => 'boolean',
                'required' => false,
                'default' => true,
                'category' => 'Other Information'
            ],
        ];
    }

    /**
     * Redirect to import settings if preview is accessed directly
     */
    public function redirectToSettings()
    {
        return redirect()->route('employees.import.settings')
            ->with('error', 'Please upload a CSV file first to preview and map columns.');
    }

    /**
     * Preview CSV file and show column mapping
     */
    public function previewCsv(Request $request)
    {
        \Log::info('Preview CSV called', [
            'method' => $request->method(),
            'has_file' => $request->hasFile('csv_file'),
            'file_details' => $request->hasFile('csv_file') ? [
                'original_name' => $request->file('csv_file')->getClientOriginalName(),
                'mime_type' => $request->file('csv_file')->getMimeType(),
                'extension' => $request->file('csv_file')->getClientOriginalExtension(),
                'size' => $request->file('csv_file')->getSize(),
            ] : null
        ]);

        // Validate the file upload - accept various CSV and Excel MIME types
        $validator = Validator::make($request->all(), [
            'csv_file' => 'required|file|mimes:csv,txt,xlsx,xls|mimetypes:text/plain,text/csv,text/x-csv,application/csv,application/x-csv,text/comma-separated-values,text/x-comma-separated-values,application/vnd.ms-excel,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet|max:10240'
        ]);

        if ($validator->fails()) {
            \Log::error('CSV validation failed', [
                'errors' => $validator->errors()->toArray()
            ]);
            return redirect()->route('employees.import.settings')
                ->withErrors($validator)
                ->withInput();
        }

        $file = $request->file('csv_file');

        if (!$file) {
            return redirect()->route('employees.import.settings')
                ->with('error', 'No file was uploaded. Please select a file and try again.');
        }

        $extension = $file->getClientOriginalExtension();
        \Log::info('File extension detected', ['extension' => $extension]);

        try {
            \Log::info('Starting file read process');

            if (in_array($extension, ['xlsx', 'xls'])) {
                \Log::info('Processing Excel file');
                $spreadsheet = IOFactory::load($file->getRealPath());
                $worksheet = $spreadsheet->getActiveSheet();
                $csvHeaders = $worksheet->toArray()[0] ?? [];
                $previewData = array_slice($worksheet->toArray(), 0, 6); // Header + 5 rows
            } else {
                \Log::info('Processing CSV file');
                $handle = fopen($file->getRealPath(), 'r');
                $csvHeaders = fgetcsv($handle);
                \Log::info('CSV headers read', ['headers' => $csvHeaders]);
                $previewData = [$csvHeaders];

                for ($i = 0; $i < 5; $i++) {
                    $row = fgetcsv($handle);
                    if ($row === false)
                        break;
                    $previewData[] = $row;
                }
                fclose($handle);
                \Log::info('CSV preview data collected', ['rows' => count($previewData)]);
            }

            // Store file temporarily
            \Log::info('Storing file temporarily');
            $tempPath = $file->store('temp_imports');
            \Log::info('File stored', ['path' => $tempPath]);

            \Log::info('Getting employee columns');
            $employeeColumns = $this->getEmployeeColumns();

            \Log::info('About to return import-mapping view', [
                'csv_headers_count' => count($csvHeaders),
                'preview_rows' => count($previewData),
                'temp_path' => $tempPath
            ]);

            return view('employees.import-mapping', compact(
                'csvHeaders',
                'previewData',
                'employeeColumns',
                'tempPath'
            ));
        } catch (\Exception $e) {
            \Log::error('Error reading CSV file', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            return redirect()->route('employees.import.settings')
                ->with('error', 'Error reading file: ' . $e->getMessage());
        }
    }

    /**
     * Process the import with column mapping
     */
    public function processImport(Request $request)
    {
        $request->validate([
            'temp_file_path' => 'required|string',
            'column_mapping' => 'required|array'
        ]);

        $company_id = session('selected_company_id');
        $filePath = storage_path('app/' . $request->temp_file_path);

        if (!file_exists($filePath)) {
            return back()->with('error', 'Temporary file not found. Please upload again.');
        }

        $columnMapping = $request->column_mapping;
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);

        try {
            DB::beginTransaction();

            $importedCount = 0;
            $errors = [];

            if (in_array($extension, ['xlsx', 'xls'])) {
                $spreadsheet = IOFactory::load($filePath);
                $worksheet = $spreadsheet->getActiveSheet();
                $rows = $worksheet->toArray();
                array_shift($rows); // Remove header row

                foreach ($rows as $index => $row) {
                    $result = $this->importRow($row, $columnMapping, $company_id, $index + 2);
                    if ($result['success']) {
                        $importedCount++;
                    } else {
                        $errors[] = $result['error'];
                    }
                }
            } else {
                $handle = fopen($filePath, 'r');
                fgetcsv($handle); // Skip header row
                $rowIndex = 2;

                while (($row = fgetcsv($handle)) !== false) {
                    $result = $this->importRow($row, $columnMapping, $company_id, $rowIndex);
                    if ($result['success']) {
                        $importedCount++;
                    } else {
                        $errors[] = $result['error'];
                    }
                    $rowIndex++;
                }
                fclose($handle);
            }

            DB::commit();

            // Delete temporary file
            @unlink($filePath);

            $message = "Successfully imported {$importedCount} employees.";
            if (count($errors) > 0) {
                $message .= " " . count($errors) . " rows had errors.";
            }

            return redirect()->route('employees.index')
                ->with('success', $message)
                ->with('import_errors', $errors);

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Import failed: ' . $e->getMessage());
        }
    }

    /**
     * Import a single row
     */
    private function importRow($row, $columnMapping, $company_id, $rowIndex)
    {
        try {
            $data = ['company_id' => $company_id];
            $hasPassword = false;

            foreach ($columnMapping as $csvColumn => $dbColumn) {
                if ($dbColumn === 'skip' || empty($dbColumn)) {
                    continue;
                }

                $value = $row[$csvColumn] ?? null;

                // Skip empty values
                if ($value === null || $value === '') {
                    continue;
                }

                // Handle special fields
                if ($dbColumn === 'password') {
                    $data[$dbColumn] = bcrypt($value);
                    $hasPassword = true;
                } elseif (in_array($dbColumn, ['active', 'has_car'])) {
                    $data[$dbColumn] = filter_var($value, FILTER_VALIDATE_BOOLEAN);
                } elseif (in_array($dbColumn, ['dob', 'license_expiry', 'employment_date', 'birthday', 'terminated_date'])) {
                    $data[$dbColumn] = date('Y-m-d', strtotime($value));
                } else {
                    $data[$dbColumn] = $value;
                }
            }

            // Set default password if not provided
            if (!$hasPassword) {
                $data['password'] = bcrypt('12345678');
            }

            // Validate required fields
            $validator = Validator::make($data, [
                'employee_id' => 'required|unique:employees,employee_id',
                'first_name' => 'required|string|max:255',
                'email' => 'required|email|unique:employees,email',
                'phone_number' => 'required|string',
                'gender' => 'required|in:Male,Female,Other',
            ]);

            if ($validator->fails()) {
                return [
                    'success' => false,
                    'error' => "Row {$rowIndex}: " . implode(', ', $validator->errors()->all())
                ];
            }

            Employee::create($data);

            return ['success' => true];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => "Row {$rowIndex}: " . $e->getMessage()
            ];
        }
    }

    /**
     * Download sample CSV template
     */
    public function downloadTemplate()
    {
        $columns = $this->getEmployeeColumns();
        $headers = array_column($columns, 'label');

        $filename = 'employee_import_template.csv';
        $handle = fopen('php://output', 'w');

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        fputcsv($handle, $headers);

        // Add sample row
        $sampleRow = [
            'EMP001', // Employee ID
            'John', // First Name
            'M', // Middle Name
            'Doe', // Last Name
            'Male', // Gender
            'john.doe@example.com', // Email
            '1990-01-15', // DOB
            'Single', // Marital Status
            '', // Zone ID
            '', // Job Role ID
            '', // Department ID
            '+1234567890', // Phone Number
            '+1234567890', // Cell Number
            '+0987654321', // Emergency Number
            '123 Main St', // Permanent Address Line 1
            'Apt 4B', // Permanent Address Line 2
            'New York', // Permanent City
            '', // Permanent State ID
            '', // Permanent Country ID
            '10001', // Permanent Zip Code
            // ... add more sample data as needed
        ];

        fputcsv($handle, $sampleRow);
        fclose($handle);
        exit;
    }
}
