<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\EmployeeRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

use App\Models\RequestType;

class EmployeeRequestController extends Controller
{
    public function index()
    {
        $employee = auth('employee')->user();
        $requests = EmployeeRequest::where('employee_id', $employee->id)
            ->orderByDesc('created_at')
            ->paginate(10);
        return view('employee.requests.index', compact('requests'));
    }

    public function create()
    {
        $employee = auth('employee')->user();
        $requestTypes = RequestType::where(function ($query) use ($employee) {
            $query->where('company_id', $employee->company_id)
                ->orWhereNull('company_id');
        })
            ->where('is_active', true)
            ->get();

        return view('employee.requests.create', compact('requestTypes'));
    }

    public function store(Request $request)
    {
        $employee = auth('employee')->user();

        $request->validate([
            'type' => [
                'required',
                \Illuminate\Validation\Rule::exists('request_types', 'key')
                    ->where(function ($query) use ($employee) {
                        $query->where('company_id', $employee->company_id)
                            ->orWhereNull('company_id');
                    })
                    ->where('is_active', true)
            ],
            'subject' => 'nullable|string|max:255',
            'message' => 'required|string',
        ]);

        $emplRequest = EmployeeRequest::create([
            'company_id' => $employee->company_id,
            'employee_id' => $employee->id,
            'type' => $request->type,
            'subject' => $request->subject,
            'message' => $request->message,
            'status' => 'pending',
        ]);

        // Notify Company Users
        foreach ($employee->company->users as $user) {
            \App\Models\Notification::create([
                'company_id' => $employee->company_id,
                'user_id' => $user->id,
                'title' => 'New ' . ucfirst($request->type) . ' Request',
                'message' => "Employee {$employee->first_name} {$employee->last_name} has submitted a new {$request->type} request.",
                'url' => route('employee-requests.show', $emplRequest->id),
            ]);
        }

        return redirect()->route('employee.requests.index')->with('success', 'Request submitted successfully.');
    }

    public function show(EmployeeRequest $request)
    {
        if ($request->employee_id != auth('employee')->id()) {
            abort(403);
        }

        $request->load('replies.user');

        return view('employee.requests.show', compact('request'));
    }

    public function reply(Request $httpRequest, EmployeeRequest $request)
    {
        if ($request->employee_id != auth('employee')->id()) {
            abort(403);
        }

        $httpRequest->validate(['message' => 'required|string']);

        $request->replies()->create([
            'is_employee_reply' => true,
            'message' => $httpRequest->message
        ]);

        $request->update(['status' => 'pending']); // Mark unread/pending for admin

        $employee = auth('employee')->user();
        // Notify Company Users
        foreach ($employee->company->users as $user) {
            \App\Models\Notification::create([
                'company_id' => $employee->company_id,
                'user_id' => $user->id,
                'title' => 'New Reply to Request',
                'message' => "Employee {$employee->first_name} {$employee->last_name} replied to a request.",
                'url' => route('employee-requests.show', $request->id),
            ]);
        }

        return back()->with('success', 'Reply sent.');
    }
}
