<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class EmployeeAuthController extends Controller
{
    public function showLogin()
    {
        if (Auth::guard('employee')->check()) {
            return redirect()->route('employee.dashboard');
        }
        return view('employee.auth.login');
    }

    public function login(Request $request)
    {
        $request->validate([
            'login_id' => ['required', 'string'],
            'password' => ['required'],
        ]);

        $login = $request->input('login_id');

        // Determine if login is email or employee_id
        $fieldType = filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'employee_id';

        $credentials = [
            $fieldType => $login,
            'password' => $request->input('password'),
        ];


        // Google reCAPTCHA Validation
        if (\App\Models\Setting::where('key', 'recaptcha_enabled')->value('value') === '1' && app()->environment('production')) {
            $request->validate([
                'g-recaptcha-response' => 'required',
            ], [
                'g-recaptcha-response.required' => 'Please complete the captcha verification.',
            ]);

            $response = \Illuminate\Support\Facades\Http::asForm()->post('https://www.google.com/recaptcha/api/siteverify', [
                'secret' => \App\Models\Setting::where('key', 'recaptcha_secret_key')->value('value'),
                'response' => $request->input('g-recaptcha-response'),
                'remoteip' => $request->ip(),
            ]);

            if (!$response->successful() || !$response->json('success')) {
                throw \Illuminate\Validation\ValidationException::withMessages([
                    'g-recaptcha-response' => 'reCAPTCHA verification failed. Please try again.',
                ]);
            }
        }

        if (Auth::guard('employee')->attempt($credentials)) {
            $user = Auth::guard('employee')->user();

            // Check if employee is active
            if (!$user->active) { // Assuming 'active' column exists on employees table? Use 'status' or check migration.
                // Checking Employee model for active status. 
                // Migration 'create_employees_table' usually has status or active? 
                // Let's assume 'active' or remove check if unsure. 
                // Wait, User table has 'active'. Employee table might not.
            }
            // Actually, let's look at Employee model again.
            // It has `use SoftDeletes`? No.
            // $guarded = ['id'].
            // Previous code checked $user->active. 
            // If I look at `EmployeeController` update method, it has 'terminated_date'.
            // It doesn't seem to have an 'active' boolean.
            // I will remove the active check for now or check termination.

            if ($user->terminated_date && \Carbon\Carbon::parse($user->terminated_date)->isPast()) {
                Auth::guard('employee')->logout();
                return back()->withErrors([
                    'login_id' => 'Your account is terminated.',
                ])->onlyInput('login_id');
            }

            $request->session()->regenerate();

            // Set company context for the employee
            if ($user->company_id) {
                session(['selected_company_id' => $user->company_id]);
            }

            return redirect()->intended(route('employee.dashboard'));
        }

        return back()->withErrors([
            'login_id' => 'The provided credentials do not match our records.',
        ])->onlyInput('login_id');
    }

    public function logout(Request $request)
    {
        Auth::guard('employee')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('employee.login');
    }
}
