<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Site;
use App\Models\Company;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;

class SiteController extends Controller
{
    public function exportPDF(Request $request)
    {
        // Increase memory limit for PDF generation
        ini_set('memory_limit', '256M');

        $query = Site::with(['company', 'country', 'state']);

        // Search
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('site_id', 'like', '%' . $request->search . '%');
            });
        }

        // Company Filter
        if ($request->company_id) {
            $query->where('company_id', $request->company_id);
        }

        $sites = $query->get();

        $headers = [
            'ID',
            'Company',
            'Account Type',
            'Name',
            'Rate',
            'Site ID',
            'Language',
            'Contact Name',
            'Phone Main',
            'Acc. Rep',
            'Address 1',
            'City',
            'Active'
        ];
        // Note: For admin view showing all columns might be too much if we also add Company. 
        // But user asked for "include all columns inn pdf that are even null".
        // I will use detailed columns similar to user side but add Company.

        $headers = [
            'ID',
            'Company',
            'Account Type',
            'Name',
            'Rate',
            'Site ID',
            'Language',
            'Contact Name',
            'Phone Main',
            'Phone Other',
            'Acc. Rep',
            'Sales Rep',
            'Fax',
            'Email',
            'Address 1',
            'Address 2',
            'Country',
            'State',
            'City',
            'Zip',
            'Lat',
            'Long',
            'Tags',
            'Website',
            'Active',
            'Created At'
        ];

        $data = $sites->map(function ($site) {
            return [
                $site->id,
                $site->company->name ?? 'N/A',
                $site->account_type,
                $site->name,
                $site->rate,
                $site->site_id,
                $site->language,
                trim($site->contact_first_name . ' ' . $site->contact_last_name),
                $site->contact_phone_main,
                $site->contact_phone_other,
                $site->account_representative,
                $site->sales_representative,
                $site->fax,
                $site->contact_email,
                $site->address_line_1,
                $site->address_line_2,
                $site->country->name ?? '',
                $site->state->name ?? '',
                $site->city,
                $site->zip_code,
                $site->latitude,
                $site->longitude,
                $site->tags,
                $site->website,
                $site->active ? 'Yes' : 'No',
                $site->created_at->format('Y-m-d'),
            ];
        });

        $pdf = Pdf::loadView('pdf.sites_all', [
            'title' => 'All Sites List',
            'headers' => $headers,
            'data' => $data
        ])->setPaper('a4', 'landscape');

        return $pdf->download('all-sites-export-' . now()->format('Y-m-d') . '.pdf');
    }

    public function index(Request $request)
    {
        $query = Site::with('company');

        // Search
        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('site_id', 'like', '%' . $request->search . '%');
            });
        }

        // Company Filter
        if ($request->company_id) {
            $query->where('company_id', $request->company_id);
        }

        // Sort
        $sort = $request->get('sort', 'name');
        $direction = $request->get('direction', 'asc');
        $query->orderBy($sort, $direction);

        $sites = $query->paginate(100);
        $companies = Company::orderBy('name')->get();

        return view('admin.sites.index', compact('sites', 'companies'));
    }

    public function toggleStatus(Site $site)
    {
        $site->active = !$site->active;
        $site->save();

        return response()->json([
            'success' => true,
            'active' => $site->active,
            'message' => 'Site status updated successfully!'
        ]);
    }

    public function destroy(Site $site)
    {
        $site->delete();
        return redirect()->route('admin.sites.index')->with('success', 'Site deleted successfully!');
    }
}
