<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Schedule;
use App\Models\Site;
use App\Models\Employee;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Mail;

class OperationalReportController extends Controller
{
    public function index(Request $request)
    {
        $company_id = session('selected_company_id');
        $reportType = trim($request->input('type', 'checkin'));

        $query = $this->buildQuery($company_id, $request);
        $reports = $query->paginate(20);

        // Filters Data
        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();
        // Get employees who have schedules or all employees
        $employees = Employee::where('company_id', $company_id)->orderBy('first_name')->get();

        return view('admin.reports.operational.index', compact('reports', 'reportType', 'sites', 'employees'));
    }

    public function exportPdf(Request $request)
    {
        $company_id = session('selected_company_id');
        $company = \App\Models\Company::find($company_id);
        $reportType = $request->query('type', 'checkin');

        $query = $this->buildQuery($company_id, $request);
        $reports = $query->get();

        $pdf = Pdf::loadView('admin.reports.operational.pdf', compact('reports', 'reportType', 'company'));
        return $pdf->download("Admin_Operational_Report_{$reportType}_" . now()->format('Ymd_Hi') . '.pdf');
    }

    public function exportExcel(Request $request)
    {
        $company_id = session('selected_company_id');
        $reportType = $request->query('type', 'checkin');

        $query = $this->buildQuery($company_id, $request);
        $reports = $query->get();

        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Headers
        $headers = ['Date', 'Duty Number', 'Employee', 'Site Name', 'Scheduled Start', 'Scheduled End', 'Actual Start', 'Actual End', 'Status'];
        $sheet->fromArray($headers, NULL, 'A1');

        // Style Headers
        $headerStyle = [
            'font' => [
                'bold' => true,
                'color' => ['argb' => 'FFFFFFFF'],
            ],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => ['argb' => 'FF1E293B'], // Slate-800
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                ],
            ],
            'alignment' => [
                'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER,
            ],
        ];
        $sheet->getStyle('A1:I1')->applyFromArray($headerStyle);

        // Data
        $data = [];
        foreach ($reports as $schedule) {
            $data[] = [
                Carbon::parse($schedule->schedule_date)->format('Y-m-d'),
                $schedule->duty_number,
                $schedule->employee_name ?? 'N/A',
                $schedule->site_name ?? 'Unknown',
                Carbon::parse($schedule->from_datetime)->format('Y-m-d H:i'),
                Carbon::parse($schedule->to_datetime)->format('Y-m-d H:i'),
                $schedule->actual_start_at ? Carbon::parse($schedule->actual_start_at)->format('Y-m-d H:i') : 'N/A',
                $schedule->actual_end_at ? Carbon::parse($schedule->actual_end_at)->format('Y-m-d H:i') : 'N/A',
                ucfirst($schedule->status)
            ];
        }

        if (count($data) > 0) {
            $sheet->fromArray($data, NULL, 'A2');

            // Style Data Rows
            $lastRow = count($data) + 1;
            $dataStyle = [
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                    ],
                ],
            ];
            $sheet->getStyle("A2:I{$lastRow}")->applyFromArray($dataStyle);
        }

        // Auto-size columns
        foreach (range('A', 'I') as $columnID) {
            $sheet->getColumnDimension($columnID)->setAutoSize(true);
        }

        $filename = "Admin_Operational_Report_{$reportType}_" . now()->format('Ymd_Hi') . ".xlsx";

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    private function buildQuery($company_id, Request $request)
    {
        $reportType = trim($request->input('type', 'checkin'));

        if ($reportType === 'scans_history') {
            $query = \Illuminate\Support\Facades\DB::table('checkpoint_scans')
                ->join('schedules', 'checkpoint_scans.schedule_id', '=', 'schedules.id')
                ->join('employees', 'checkpoint_scans.employee_id', '=', 'employees.id')
                ->join('checkpoints', 'checkpoint_scans.checkpoint_id', '=', 'checkpoints.id')
                ->join('sites', 'schedules.site_id', '=', 'sites.id')
                ->leftJoin('tour_routes', 'checkpoint_scans.tour_route_id', '=', 'tour_routes.id')
                ->where('schedules.company_id', $company_id)
                ->select(
                    'checkpoint_scans.id as scan_id',
                    'checkpoint_scans.scanned_at',
                    'checkpoint_scans.latitude',
                    'checkpoint_scans.longitude',
                    'checkpoint_scans.evidence_image',
                    'checkpoint_scans.employee_message',
                    'schedules.id as schedule_id',
                    'schedules.duty_number',
                    'schedules.from_datetime',
                    'schedules.to_datetime',
                    'schedules.status',
                    'sites.name as site_name',
                    'checkpoints.name as checkpoint_name',
                    'checkpoints.checkpoint_id_code',
                    'tour_routes.name as tour_route_name',
                    \Illuminate\Support\Facades\DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                    'employees.id as employee_id'
                )
                ->orderBy('checkpoint_scans.scanned_at', 'desc');

            if ($request->filled('start_date')) {
                $query->whereDate('checkpoint_scans.scanned_at', '>=', $request->start_date);
            }
            if ($request->filled('end_date')) {
                $query->whereDate('checkpoint_scans.scanned_at', '<=', $request->end_date);
            }
            if ($request->filled('site_id')) {
                $query->where('schedules.site_id', $request->site_id);
            }
            if ($request->filled('employee_id')) {
                $query->where('checkpoint_scans.employee_id', $request->employee_id);
            }
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('schedules.duty_number', 'like', "%{$search}%")
                        ->orWhere('checkpoints.name', 'like', "%{$search}%")
                        ->orWhere('checkpoints.checkpoint_id_code', 'like', "%{$search}%");
                });
            }

            return $query;
        }

        $query = \Illuminate\Support\Facades\DB::table('employee_schedule')
            ->join('schedules', 'employee_schedule.schedule_id', '=', 'schedules.id')
            ->join('employees', 'employee_schedule.employee_id', '=', 'employees.id')
            ->join('sites', 'schedules.site_id', '=', 'sites.id')
            ->where('schedules.company_id', $company_id)
            ->select(
                'schedules.id as schedule_id',
                'schedules.schedule_date',
                'schedules.duty_number',
                'schedules.status',
                'schedules.from_datetime',
                'schedules.to_datetime',
                'sites.name as site_name',
                'employees.first_name',
                'employees.last_name',
                'employees.id as employee_id',
                \Illuminate\Support\Facades\DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name"),
                'employee_schedule.actual_start_at',
                'employee_schedule.actual_end_at'
            )
            ->whereNotNull('employee_schedule.actual_start_at') // Must have started for Op Report
            ->orderBy('schedules.schedule_date', 'desc');

        if ($reportType === 'checkout') {
            $query->whereNotNull('employee_schedule.actual_end_at');
        }

        // Filters
        if ($request->filled('start_date')) {
            $query->whereDate('schedules.schedule_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('schedules.schedule_date', '<=', $request->end_date);
        }
        if ($request->filled('site_id')) {
            $query->where('schedules.site_id', $request->site_id);
        }
        if ($request->filled('employee_id')) {
            $query->where('employee_schedule.employee_id', $request->employee_id);
        }
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('schedules.duty_number', 'like', "%{$search}%");
        }

        return $query;
    }
    public function bannedEmployees(Request $request)
    {
        $company_id = session('selected_company_id');

        $query = \Illuminate\Support\Facades\DB::table('employee_site_bans')
            ->join('employees', 'employee_site_bans.employee_id', '=', 'employees.id')
            ->join('sites', 'employee_site_bans.site_id', '=', 'sites.id')
            ->where('employees.company_id', $company_id)
            ->select(
                'employee_site_bans.*',
                'employees.first_name',
                'employees.last_name',
                'sites.name as site_name',
                \Illuminate\Support\Facades\DB::raw("CONCAT(employees.first_name, ' ', employees.last_name) as employee_name")
            );

        if ($request->filled('site_id')) {
            $query->where('employee_site_bans.site_id', $request->site_id);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('employees.first_name', 'like', "%{$search}%")
                    ->orWhere('employees.last_name', 'like', "%{$search}%");
            });
        }

        // Get raw data then group in collection
        $bans = $query->get();

        // Manual hydration for view since we used DB query builder (returning stdClass)
        // Or better, use Models. Let's stick to DB builder for reporting speed but we need site relation to work?
        // Actually, we already joined sites, so we have site_name. 
        // But the view uses `$ban->site->name`. Let's switch to Eloquent for easier Relation access in view
        // OR adjust view to use flat object.
        // Let's ADJUST VIEW to use flat object properties we selected above (site_name). 
        // Wait, I used `$ban->site->name` in the view I just wrote. 
        // I should probably use Eloquent here to make it cleaner or update view. 
        // Eloquent is safer.

        $eloquentQuery = \App\Models\Employee::where('company_id', $company_id)
            ->whereHas('bannedSites', function ($q) use ($request) {
                if ($request->filled('site_id')) {
                    $q->where('sites.id', $request->site_id);
                }
            })
            ->with([
                'bannedSites' => function ($q) use ($request) {
                    if ($request->filled('site_id')) {
                        $q->where('sites.id', $request->site_id);
                    }
                }
            ]);

        if ($request->filled('search')) {
            $eloquentQuery->where(function ($q) use ($request) {
                $q->where('first_name', 'like', "%{$request->search}%")
                    ->orWhere('last_name', 'like', "%{$request->search}%");
            });
        }

        $employeesWithBans = $eloquentQuery->get();

        // Re-structure for the view: Group by Employee Name
        // View expects: $bannedEmployees as Collection: 'Employee Name' => [Ban Object, Ban Object]
        // Actually, the View Iterates: @forelse($bannedEmployees as $employeeName => $bans)
        // With Eloquent, we have Employees -> bans.

        $bannedEmployees = collect();
        foreach ($employeesWithBans as $emp) {
            $bannedEmployees->put($emp->first_name . ' ' . $emp->last_name, $emp->bannedSites);
        }

        $sites = Site::where('company_id', $company_id)->orderBy('name')->get();

        return view('admin.reports.operational.banned_employees', compact('bannedEmployees', 'sites'));
    }
    public function employeePasswords(Request $request)
    {
        $company_id = session('selected_company_id');

        // Fetch Employees
        $query = Employee::where('company_id', $company_id)->orderBy('first_name');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                    ->orWhere('last_name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        $employees = $query->get();

        if ($request->ajax()) {
            return view('admin.reports.operational.partials.password_table_rows', compact('employees'))->render();
        }

        return view('admin.reports.operational.employee_passwords', compact('employees'));
    }

    public function logPasswordView(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:employees,id'
        ]);

        $employee = Employee::findOrFail($request->employee_id);

        \App\Models\ActivityLog::create([
            'company_id' => session('selected_company_id'),
            'user_id' => auth()->id(),
            'subject_type' => Employee::class,
            'subject_id' => $employee->id,
            'description' => "Viewed password hash for employee: {$employee->first_name} {$employee->last_name}",
            'properties' => ['ip' => $request->ip(), 'user_agent' => $request->userAgent()]
        ]);

        return response()->json(['success' => true]);
    }
    public function emailShiftReport(Schedule $schedule, Employee $employee)
    {
        $user = auth()->user();

        // Load necessary relationships for the comprehensive report view
        $schedule->load([
            'site',
            'company',
            'incidents' => function ($q) use ($employee) {
                $q->where('employee_id', $employee->id);
            }
        ]);

        // Get the pivot data for this specific employee on this schedule
        $empWithPivot = $schedule->employees()->where('employees.id', $employee->id)->first();
        if (!$empWithPivot) {
            return back()->with('error', 'Employee not found on this schedule.');
        }
        $pivot = $empWithPivot->pivot;

        $isPdf = true;
        // Reusing the comprehensive job report view
        $pdf = Pdf::loadView('employee.reports.complete', compact('schedule', 'employee', 'pivot', 'isPdf'));

        $pdfContent = $pdf->output();
        $filename = "Shift_Report_{$schedule->duty_number}.pdf";

        try {
            Mail::html("<p>Hello {$user->name},</p><p>Please find attached the comprehensive shift report for <strong>Duty #{$schedule->duty_number}</strong>.</p><p>This report includes operative profile, tactical timeline, geospatial coordinates, and incident logs.</p><p>Regards,<br>Operations Team</p>", function ($message) use ($user, $pdfContent, $filename, $schedule) {
                $message->to($user->email)
                    ->subject("Shift Report: Duty #{$schedule->duty_number}")
                    ->attachData($pdfContent, $filename, [
                        'mime' => 'application/pdf',
                    ]);
            });

            return back()->with('success', 'Shift report has been emailed to you at ' . $user->email);
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to send email: ' . $e->getMessage());
        }
    }
}
