<?php

namespace App\Filament\Team\Resources\OrderAddedNotificationsResource\Pages;

use App\Filament\Team\Resources\OrderAddedNotificationsResource;
use App\Models\NewOrder;
use App\Models\NewOrderProcess;
use App\Models\NewOrderProcessDetail;
use App\Models\NewSite;
use App\Models\PreviousOrder;
use App\Models\User;
use App\Project\NewOrderStatus;
use App\Project\NewSiteStatus;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Set;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Concerns\InteractsWithInfolists;
use Filament\Infolists\Contracts\HasInfolists;
use Filament\Infolists\Infolist;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\Page;
use Filament\Support\Enums\FontWeight;
use Filament\Tables\Actions\Action;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Enums\ActionsPosition;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Table;
use Illuminate\Contracts\View\View;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;

class PushToManager extends Page implements HasForms, HasTable, HasInfolists
{
    use InteractsWithForms;
    use InteractsWithTable;
    use InteractsWithInfolists;

    public ?array $data = [];

    public $id;

    public $new_order;

    public $selected_sites = [];
    public $selected_orders= [];




    protected static string $resource = OrderAddedNotificationsResource::class;

    protected static string $view = 'filament.team.resources.order-added-notifications-resource.pages.push-to-manager';

    public function mount(int | string $record): void
    {
        $this->id = $record;
        $this->new_order = NewOrder::where('team_id',Auth::id())
            ->where('id',$record)
            ->whereIn('new_order_status', [NewOrderStatus::$PENDING_ORDER,NewOrderStatus::$MANAGER_DISAPPROVED_POST_TEAM,NewOrderStatus::$MANAGER_CREATED_ORDER_AND_PUSHED_TO_TEAM])
            ->firstOrFail();

    }

    public function table(Table $table): Table
    {
        return $table
            ->query(
                NewSite::where('site_status',NewSiteStatus::APPROVED)

                    ->where('bulk_upload_status',null)
                    ->where('bulk_upload_id',null)
                    ->where('delete_site',null)

            )
            ->paginated(config("app.pagination"))
            ->modifyQueryUsing(function ($query) {
                if(count($this->selected_sites) > 0 ){
                    $query->whereNotIn('id',collect($this->selected_sites)->pluck("id")->toArray());
                }
            })
            ->columns([
                TextColumn::make('root_domain')->label("Root Domain"),
                TextColumn::make('website_status')->label("Website Status")->badge()->color(function (NewSite $d){
                    if($d->website_status == 'Acceptable'){
                        return 'success';
                    }
                    if($d->website_status == 'Non Acceptable'){
                        return 'danger';
                    }
                })->sortable(),
                TextColumn::make("category")->badge()->sortable(),
                TextColumn::make('id')
                    ->label(function (NewSite $order){
                        if($this->new_order->order_type == 'gp' && $this->new_order->fc ){
                            return "GP Price(FC)";
                        }

                        if($this->new_order->order_type == 'gp' && $this->new_order->fc == 0 ){
                            return "GP Price";
                        }

                        if($this->new_order->order_type == 'niche' && $this->new_order->fc){
                            return "Niche Price(FC)";
                        }

                        if($this->new_order->order_type == 'niche' && $this->new_order->fc == 0){
                            return "Niche Price";
                        }
                    })->formatStateUsing(function (NewSite $order) {
                    if($this->new_order->order_type == 'gp' && $this->new_order->fc ){
                        return $order->fc_gp;
                    }

                    if($this->new_order->order_type == 'gp' && $this->new_order->fc == 0 ){
                        return $order->gp_price;
                    }

                    if($this->new_order->order_type == 'niche' && $this->new_order->fc){
                        return $order->fc_ne;
                    }

                    if($this->new_order->order_type == 'niche' && $this->new_order->fc == 0){
                        return $order->niche_edit_price;
                    }

                }),

                TextColumn::make("dr")->label("DR")->sortable(),
                TextColumn::make("rd")->label("RD")->sortable(),
                TextColumn::make("da")->label("DA")->sortable(),
                TextColumn::make("spam_score")->label("Spam Score")->sortable(),

                TextColumn::make("traffic_source")->label("Traffic Source")->sortable(),

                TextColumn::make("country_source")->label("Country Source")->toggleable(isToggledHiddenByDefault: true),
            ])
            ->recordUrl(null)
            ->filters([
                Filter::make("Search")->form([
                    TextInput::make("root_domain")->label("Search by Root Domain"),
                    TextInput::make("category")->label("Search by Category"),
                    TextInput::make("country_source")->label("Search by Country Source"),
                    Select::make("website_status")->label("Search by website status")->options([
                        "Acceptable" => "Acceptable",
                        "Sudden Decline" => "Sudden Decline",
                        "Acceptable in Worse Case" => "Acceptable in Worse Case",
                        "Inflated Traffic"  => "Inflated Traffic",
                        "Need to Further Review" => "Need to Further Review",
                        "Non Acceptable" => "Non Acceptable",
                        "Acceptable but below 100%" => "Acceptable but below 100%",
                    ])
                ])->query(function (Builder $query,$data){
                    return $query
                        ->when(!empty($data["root_domain"]),function ($query) use ($data){
                            $query->where("root_domain","like","%".$data['root_domain']."%");
                        })
                        ->when(!empty($data["category"]),function ($query) use ($data){
                            $query->where("category","like","%".$data['category']."%");
                        })
                        ->when(!empty($data["country_source"]),function ($query) use ($data){
                            $query->where("country_source","like","%".$data['country_source']."%");
                        })
                        ->when(!empty($data['website_status']),function ($query) use ($data){
                            $query->where("website_status",$data['website_status']);
                        })
                        ;
                })->columns(4)->columnSpanFull()
            ])
            ->filtersLayout(FiltersLayout::AboveContent)
            ->actions([
                Action::make("select")
                    ->action(function (NewSite $newSite){
                        $this->selected_sites[$newSite->id] = ["id"=>$newSite->id,"post_url"=>""];
                        $this->selected_orders = NewSite::whereIn('id', collect($this->selected_sites)->pluck("id")->sortDesc()->toArray())->get()->toArray();
                        $this->form->fill(["selected_orders" => $this->selected_orders]);

                    })
            ],position: ActionsPosition::BeforeColumns)
            ->bulkActions([
                // ...
            ]);
    }


    public function OrderInfolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->record($this->new_order)
            ->schema([
                \Filament\Infolists\Components\Section::make("Order Details")->schema([
                    TextEntry::make("order_id")->weight(FontWeight::Medium)->color('primary'),
                    TextEntry::make("category")->weight(FontWeight::Medium)->color('primary'),
                    TextEntry::make("client_website")->weight(FontWeight::Medium)->color('primary'),
                    TextEntry::make("no_of_links")->weight(FontWeight::Medium)->color('primary'),
                    TextEntry::make("order_type")->weight(FontWeight::Medium)->color('primary'),
                    TextEntry::make("message")->columnSpanFull()->formatStateUsing(function ($state){
                        return new HtmlString($state);
                    })
                ])->columns(5)->collapsed()
            ]);
    }


    public function render(): View
    {
        $new_order= NewOrder::where('team_id',Auth::id())
            ->where('id',$this->id)
            ->whereIn('new_order_status', [NewOrderStatus::$PENDING_ORDER,NewOrderStatus::$MANAGER_DISAPPROVED_POST_TEAM,NewOrderStatus::$MANAGER_CREATED_ORDER_AND_PUSHED_TO_TEAM])
            ->firstOrFail();
        $pending_orders= PreviousOrder::where('past_order_root_domain',$new_order->client_website)->get();


        return parent::render()->with("new_order",$new_order)->with("pending_orders",$pending_orders);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make(function (Get $get){
                    return 'Selected Sites ('.($get('selected_orders') ? count($get('selected_orders')) : 0).')';
                })->schema([

                    Repeater::make("selected_orders")
                        ->label(" ")
                        ->addable(false)
                        ->orderable(false)
                        ->deletable(false)
                        ->extraItemActions([
                            \Filament\Forms\Components\Actions\Action::make("deleteSite")->tooltip("Delete Site")
                                ->icon('heroicon-m-trash')
                                ->action(function (array $arguments,Repeater $component,Get $get,Set $set) {
                                    $selectedData = $component->getState()[$arguments['item']];

                                    $id = $selectedData['id'];
//                                    $this->selected_sites[$newSite->id] = ["id"=>$newSite->id,"post_url"=>""];

                                    $this->selected_sites = array_filter($this->selected_sites,function ($s) use ($id){
                                        return $s['id'] !== $id;
                                    });

                                    $orders = array_filter($get('selected_orders'),function ($s) use ($id){
                                        return $s['id'] !== $id;
                                    });

//                                    dd($orders);

                                    $set('selected_orders', $orders);

//                                    dd($get('selected_orders'));


//                                    $this->selected_orders = NewSite::whereIn('id', collect($this->selected_sites)->pluck("id")->sortDesc()->toArray())->get()->toArray();
//                                    $this->form->fill(["selected_orders" => $this->selected_orders]);

                                })
                        ])

                        ->schema([
                            Placeholder::make("root_domain")
                            ->inlineLabel()
                            ->content(function (Get $get){
                                return $get('root_domain');
                            }),
                            Placeholder::make("gp_price")
                                ->inlineLabel()
                                ->content(function (Get $get){
                                    if($this->new_order->order_type == 'gp' && $this->new_order->fc ){
                                        return $get('fc_gp');
                                    }

                                    if($this->new_order->order_type == 'gp' && $this->new_order->fc == 0 ){
                                        return $get('gp_price');
                                    }

                                    if($this->new_order->order_type == 'niche' && $this->new_order->fc){
                                        return $get('fc_ne');
                                    }

                                    if($this->new_order->order_type == 'niche' && $this->new_order->fc == 0){
                                        return $get('niche_edit_price');
                                    }

                                }),
                            Textarea::make("post_url")->label("Copy Urls")->live(onBlur: true)->hidden(function (){
                                if($this->new_order->order_type == "niche"){
                                    return false;
                                }
                                return true;
                            }),
                            Textarea::make("note")->live(onBlur: true),


                    ]),

                    Textarea::make("note")


                ])
            ])
            ->statePath('data');
    }

    public function create(): void
    {
        $data= $this->form->getState();
//        dd($data['selected_orders']);
//        dd($this->selected_sites);
//        dd($data);
        if (count($this->selected_sites) < $this->new_order->no_of_links ){
             Notification::make()
                ->title("Select minimum ".$this->new_order->no_of_links . " sites")
                ->success()
                ->send();
        }else{
            try{
                DB::beginTransaction();
                $this->new_order->new_order_status = NewOrderStatus::$TEAM_PUSHED_TO_MANAGER;
                $this->new_order->save();

                $process= new NewOrderProcess();
                $process->new_order_id= $this->new_order->id;
                $process->team_id= Auth::id();
                $process->manager_id = $this->new_order->manager_id;
                $process->note= $data['note'];
                $process->status= NewOrderStatus::$TEAM_PUSHED_TO_MANAGER;
                $process->save();


                foreach ($data['selected_orders'] as $selected_site) {
                    $new_site= NewSite::where('id',$selected_site['id'])->firstOrFail();

                    $detail= new NewOrderProcessDetail();
                    $detail->new_order_process_id= $process->id;
                    $detail->new_site_id= $selected_site['id'];
                    if($this->new_order->order_type == 'niche') {
                        $detail->url = $selected_site['post_url'];
                    }
                    $detail->new_note= $selected_site['note'];
                    if($this->new_order->order_type == 'gp' && $this->new_order->fc == 1){
                        $detail->price= $new_site->fc_gp;
                    }
                    if($this->new_order->order_type == 'gp' && $this->new_order->fc != 1){
                        $detail->price= $new_site->gp_price;
                    }
                    if($this->new_order->order_type == 'niche' && $this->new_order->fc == 1){
                        $detail->price= $new_site->fc_ne;
                    }
                    if($this->new_order->order_type == 'niche' && $this->new_order->fc != 1){
                        $detail->price= $new_site->niche_edit_price;
                    }

                    $detail->save();
                }



                Notification::make("success")->title("Successfully pushed to manager")->send();

                $admin = admin_record();
                $manager = User::where('role','manager')->where('id',$this->new_order->manager_id)->first();

                $order_id = NewOrder::where('id',$this->new_order->id)->first();
                Notification::make()->title("Order No. ".$order_id->order_id." Pushed to Manager")
                    ->body("Pushed by ".Auth::user()->name.'('.Auth::user()->email.')')
                    ->success()
                    ->sendToDatabase([$admin,$manager]);

                DB::commit();

                $this->redirect(OrderAddedNotificationsResource::getUrl("index"));
//              return redirect()->route("team.dashboard")->with("success","Successfully pushed to manager");
            }catch (\Exception $exception){
                DB::rollBack();
                dd($exception->getMessage());
                session()->flash("error",$exception->getMessage());
            }
        }

    }
}
