<?php

namespace App\Filament\Team\Resources;

use App\Filament\Team\Resources\CompletedOrdersResource\Pages;
use App\Filament\Team\Resources\CompletedOrdersResource\RelationManagers;
use App\Models\CompletedOrders;
use App\Models\NewOrder;
use App\Models\User;
use App\Notifications\DummyNotification;
use App\Project\NewOrderStatus;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Auth;
use Malzariey\FilamentDaterangepickerFilter\Filters\DateRangeFilter;

class CompletedOrdersResource extends Resource
{
    protected static ?string $model = NewOrder::class;
    protected static ?string $label = "Completed Orders";


    protected static ?string $navigationIcon = 'heroicon-o-credit-card';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function getNavigationBadge(): ?string
    {
        return team_completed_orders();
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(
                 NewOrder::where('team_id',Auth::id())
                    ->whereNotIn('new_order_status',[
                        NewOrderStatus::$PENDING_ORDER,
                        NewOrderStatus::$TEAM_PUSHED_TO_MANAGER,
                        NewOrderStatus::$MANAGER_DISAPPROVED_POST_TEAM,
                        NewOrderStatus::$MANAGER_CREATED_ORDER_AND_PUSHED_TO_TEAM
                    ])

            )
//            ->deferLoading()
            ->defaultSort("updated_at","desc")
            ->striped()
            ->filtersLayout(Tables\Enums\FiltersLayout::AboveContentCollapsible)
            ->paginated(config("app.pagination"))
            ->recordUrl(null)
            ->columns([

                Tables\Columns\TextColumn::make('order_id')->label("Order ID")->wrap(),

                Tables\Columns\TextColumn::make('order_type')->label("Order Type")->sortable(),
                Tables\Columns\TextColumn::make('category')->toggleable()->badge()->wrap(),
                Tables\Columns\TextColumn::make('manager.name')->toggleable()->sortable(),
                Tables\Columns\TextColumn::make('client_name')->toggleable()->label("Client Name")->wrap(),
                Tables\Columns\TextColumn::make('client_website')->toggleable()->label("Client Website")->wrap(),
                Tables\Columns\TextColumn::make('no_of_links')->toggleable()->label("No of Links")->sortable(),

                Tables\Columns\TextColumn::make('order_package')->toggleable()->label("Order Package")->wrap(),
                Tables\Columns\TextColumn::make('created_at')->label("Pushed Date")->toggleable(isToggledHiddenByDefault: true),

            ])
            ->filters([
                DateRangeFilter::make("created_at")->label("Pushed Date"),
                Tables\Filters\Filter::make("Search")->form([
                    Forms\Components\TextInput::make("order_id"),
                    Forms\Components\Select::make("order_type")->options([
                        "gp"    =>  "GP",
                        "niche" =>  "Niche",
                    ]),
                    Forms\Components\Select::make("manager")->options(User::where('role','manager')->pluck("email","id"))


                ])

                    ->query(function (Builder $query, array $data): Builder {
                        return $query

                            ->when($data['order_id'],function (Builder $query) use ($data){
                                return $query->where('order_id','LIKE','%'.$data['order_id'].'%');
                            })
                            ->when($data['order_type'],function (Builder $query) use ($data){
                                return $query->where('order_type',$data['order_type']);
                            })
                            ->when($data['manager'],function (Builder $query) use ($data){
                                return $query->where('manager_id',$data['manager']);
                            })
                            ;
                    })
                    ->columns(6)->columnSpanFull()
            ])
            ->actions([
                Tables\Actions\Action::make("detail")
                    ->icon("heroicon-o-eye")
                    ->label(" ")
                    ->tooltip("Detail")
                    ->url(function (NewOrder $record) {
                        return CompletedOrdersResource::getUrl("detail", ["record" => $record->id]);
                    })
            ])
            ->bulkActions([

            ]);
    }



    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCompletedOrders::route('/'),
            'create' => Pages\CreateCompletedOrders::route('/create'),
            'edit' => Pages\EditCompletedOrders::route('/{record}/edit'),
            'detail' => Pages\OrderDetailPage::route("/{record}/detail"),
        ];
    }
}
